<?php namespace Visiosoft\TrendyolModule\IntegratedProduct\Listener;

use Visiosoft\AdvsModule\Adv\Event\EditedAd;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\TrendyolModule\IntegratedProduct\IntegratedProductRepository;
use Visiosoft\TrendyolModule\IntegratedProduct\Form\CreateProductOnTrendyolFormHandler;
use \Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\FilesModule\File\FileModel;
use Anomaly\Streams\Platform\Message\MessageBag;

class UpdateProduct extends CreateProductOnTrendyolFormHandler
{
    public function handleUpdate(EditedAd $updatedAdvModel)
    {

        $updatedProduct = $updatedAdvModel->getAdDetail();

        $integratedProducts = app(IntegratedProductRepository::class);

        // Detect whether the updated product is an integrated product.

        if ($integratedProductToBeUpdated = $integratedProducts->findBy('product_id', $updatedProduct->id)) {
            
            // Yes! Let's update the product on Trendyol as well.

            $this->getCredentialsForTrendyolApi();
            $this->detectEndpoint();

            // First, let's compare the product information previously uploaded to Trendyol with the updated product information and detect the changed information.

            $productOnTrendyol = json_decode($this->returnProductInformation('createdProductOnTrendyol', $integratedProductToBeUpdated['barcode'], $this->messages))->content[0];

            $imagesOfUpdatedProduct = array();

            $advModel = new AdvModel();
            $fileModel = new FileModel();

            $productImagesArr = $advModel->getAdvimage($updatedProduct->id);

            foreach ($productImagesArr as $image) {
                array_push($imagesOfUpdatedProduct, (object)array(
                    "url" => $image->make()->url()
                ));
            }

            $updatedProductOcInfos = array(
                'title' => $updatedProduct->name,
                'description' => $updatedProduct->advs_desc,
                'images' => json_encode($imagesOfUpdatedProduct),
            );

            $trendyolProductInfos = array(
                'title' => $productOnTrendyol->title,
                'description' => $productOnTrendyol->description,
                'images' => json_encode($productOnTrendyol->images),
            );

            $changedProductInfos = array_diff_assoc($updatedProductOcInfos, $trendyolProductInfos);

            if (count($changedProductInfos) > 0) {
                // Yes, One or more of the (title, description, image information) has changed.
                // Let's use 'updateProducts' service.

                $productInformation = $this->prepareProductForInfoUpdateOnTrendyol($updatedProduct, $integratedProductToBeUpdated, $imagesOfUpdatedProduct);
    
                $batchRequestId = $this->sendRequestToTrendyol($productInformation, $this->messages, "PUT");
    
                if ($this->messages->has('error')) {
                    return;
                }
    
                $requestResult = $this->returnProductInformation('resultOfSendRequest', $batchRequestId, $this->messages);
    
                if ($this->messages->has('error')) {
                    return;
                }
    
                $requestResultArr = (array) json_decode($requestResult);
    
                if ($requestResultArr['failedItemCount'] === 0) {
                    $changedInfos = array();
                    foreach($changedProductInfos as $info => $val) {
                        array_push($changedInfos, $info);
                    }
                    $this->messages->success('Your product\'s ' . implode(", ", $changedInfos) . ' information has been updated also on Trendyol.');
                } else {
                    return ['error' => true];
                }
            }

            // Updating price and stock information requires the use of a different service than updating other product information on Trendyol.

            $this->updateStockAndPrice($integratedProductToBeUpdated, $updatedProduct, $productOnTrendyol);
            
        } else {
            // No. Trendyol module has nothing to do.
        }
    }

    public function prepareProductForInfoUpdateOnTrendyol($updatedProduct, $integratedProductToBeUpdated, $imagesArrToBeSentToTrendyol)
    {
        $trendyolSpesificInformation = json_decode($integratedProductToBeUpdated['trendyol_spesific_information']);

        preg_match_all("/\d+/", $trendyolSpesificInformation->brandId, $brandInputMatches);
        preg_match_all("/\d+/", $trendyolSpesificInformation->categoryId, $categoryInputMatches);
        preg_match_all("/\d+/", $trendyolSpesificInformation->cargoCompanyId, $cargoCompanyInputMatches);

        $productInformation = (object)array(
                "items" => [
                    (object)[
                        'barcode' => $integratedProductToBeUpdated['barcode'],
                        'title' => $updatedProduct->name,
                        'productMainId' => $trendyolSpesificInformation->mainProductId,
                        'brandId' => intval($brandInputMatches[0][0]),
                        'categoryId' => intval($categoryInputMatches[0][0]),
                        'stockCode' => $trendyolSpesificInformation->stockCode,
                        'dimensionalWeight' => $trendyolSpesificInformation->dimensionalWeight,
                        'description' => $updatedProduct->advs_desc,
                        'cargoCompanyId' => intval($cargoCompanyInputMatches[0][0]),
                        'images' =>  $imagesArrToBeSentToTrendyol,
                        'vatRate' => $trendyolSpesificInformation->taxRate,
                        'attributes' => $trendyolSpesificInformation->attributes,
                    ],
                ],
        );
        
        return $productInformation;
    }

    public function updateStockAndPrice($integratedProductToBeUpdated, $updatedProduct, $productOnTrendyol)
    {
        // Detect if one or more of the (stock quantity, price, list price information) has changed.
        // But first, change operation if it is necessary.

        if ($updatedProduct->currency !== 'TRY') {
            $enabledCurrencies = setting_value('visiosoft.module.advs::enabled_currencies');

            if (!array_search("TRY", $enabledCurrencies)) {
                $this->messages->error('The operation failed. The foreign currency of a product to be created in Trendyol must be "TRY". 
                The price type of the product you created on this site is not "TRY". 
                However, if you select "TRY" as "enabled currencies" from the "Adv" module settings, we will do the conversion for you!');
                return;
            }

            $settings = app(\Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface::class);
            $updatedProduct->foreignCurrency($updatedProduct->currency, $updatedProduct->price, 
            $updatedProduct->id, $settings, false);
            $currencyList = (array)(json_decode($updatedProduct->foreign_currencies));
            $updatedProduct->price = $currencyList["TRY"];

            $advModel = new AdvModel();

            $updatedProduct->foreignCurrency($updatedProduct->currency, $updatedProduct->standard_price, 
            $updatedProduct->id, $settings, false);
            $currencyList = (array)(json_decode($advModel->getAdv($updatedProduct->id)->foreign_currencies));
            // geçici çözüm
            $updatedProduct->standard_price = $currencyList["TRY"] + 20;
        }

        $updatedProductOcInfos = [
            'stock' => $updatedProduct->stock,
            'listPrice' => $updatedProduct->standard_price,
            'salePrice' => $updatedProduct->price,
        ];

        $trendyolProductInfos = [
            'stock' => $productOnTrendyol->quantity,
            'listPrice' => $productOnTrendyol->listPrice,
            'salePrice' => $productOnTrendyol->salePrice,
        ];

        $changedProductInfos = array_diff($updatedProductOcInfos, $trendyolProductInfos);

        if (count($changedProductInfos) > 0) {
            // Yes, One or more of the (stock quantity, price, list price information) has changed.
            // Let's use 'updatePriceAndInventory' service.

            $productInformation = $this->prepareProductForStockAndPriceUpdate($updatedProduct, $integratedProductToBeUpdated);

            if ($this->messages->has('error')) {
                return;
            }

            $batchRequestId = $this->sendRequestToTrendyol($productInformation, $this->messages, "POST_UPDATE");

            if ($this->messages->has('error')) {
                return;
            }

            $requestResult = $this->returnProductInformation('resultOfSendRequest', $batchRequestId, $this->messages);

            if ($this->messages->has('error')) {
                return;
            }

            $requestResultArr = (array) json_decode($requestResult);

            if ($requestResultArr['failedItemCount'] === 0) {
                $changedInfos = array();
                foreach($changedProductInfos as $info => $val) {
                    array_push($changedInfos, $info);
                }
                $this->messages->success('Your product\'s ' . implode(", ", $changedInfos) . ' information has been updated also on Trendyol.                     
                If you have not changed the price information, there has been a change due to the exchange rate difference.');               
            } else {
                return ['error' => true];
            }            
        }
    }

    public function prepareProductForStockAndPriceUpdate($updatedProduct, $integratedProductToBeUpdated)
    {
        if ($updatedProduct->price >= $updatedProduct->standard_price) {
            $this->messages->error('The operation failed. In order to update a product on Trendyol, the list price (standard price) of your product must be higher than the current price.');
            return;
        }

        $productInformation = (object)array(
            "items" => [
                (object)[
                    'barcode' => $integratedProductToBeUpdated['barcode'],
                    "quantity" => $updatedProduct->stock,
                    "salePrice" => $updatedProduct->price,
                    "listPrice" => $updatedProduct->standard_price
                ],
            ],
        );

        return $productInformation;
    }
}