<?php namespace Visiosoft\TicketsModule\Http\Controller\Api;

use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\TicketsModule\Content\Contract\ContentRepositoryInterface;
use Visiosoft\TicketsModule\Department\Contract\DepartmentRepositoryInterface;
use Visiosoft\TicketsModule\Http\Request\AnswerRequest;
use Visiosoft\TicketsModule\Http\Request\TicketCreateRequest;
use Visiosoft\TicketsModule\Priority\Contract\PriorityRepositoryInterface;
use Visiosoft\TicketsModule\Ticket\Contract\TicketRepositoryInterface;
use Visiosoft\TicketsModule\Ticket\Event\NewTicketEmailEvent;
use Visiosoft\TicketsModule\Ticket\Event\TicketAnsweredEmailEvent;
use Visiosoft\TicketsModule\Ticket\Event\TicketClosedEmailEvent;
use Visiosoft\TicketsModule\Traits\ApiReturnResponseTrait;

class TicketController extends ResourceController
{
    use ApiReturnResponseTrait;

    protected $user, $ticketRepository, $contentRepository, $departmentRepository, $priorityRepository;

    public function __construct(TicketRepositoryInterface     $ticketRepository,
                                ContentRepositoryInterface    $contentRepository,
                                DepartmentRepositoryInterface $departmentRepository,
                                PriorityRepositoryInterface   $priorityRepository)
    {
        $this->ticketRepository = $ticketRepository;
        $this->contentRepository = $contentRepository;
        $this->departmentRepository = $departmentRepository;
        $this->priorityRepository = $priorityRepository;
        parent::__construct();
    }

    public function siteAuth(Request $request)
    {
        try {
            $inputs = $request->all();
            $modules = app('module.collection')->installed();
            if (!$modules->get('visiosoft.module.site')) {
                throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Site module']));
            }
            $siteRepository = app(SiteRepositoryInterface::class);
            if (!$site = $siteRepository->newQuery()->where('api_token', $inputs['token'])->first()) {
                throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Site']));
            }
            $accessToken = app(\Visiosoft\ConnectModule\User\UserModel::class)->find($site->assign_id)->createToken($site->assign_id)->accessToken;
            return $this->sendResponse(['access_token' => $accessToken], '');
        } catch (\Exception $e) {
            return $this->sendExceptionError($e);
        }
    }

    public function index()
    {
        try {
            $results = $this->ticketRepository->newQuery()
                ->where('user_id', Auth::id())
                ->orderBy('updated_at', 'desc')->paginate(25);
            return $this->sendResponse($results, '');
        } catch (\Exception $e) {
            return $this->sendExceptionError($e);
        }
    }

    public function thread($id)
    {
        try {
            $ticket = $this->ticketRepository->getModel()->where('id', $id)->where('user_id', Auth::id())->first();
            if (empty($ticket)) {
                return $this->sendError(trans('visiososft.module.tickets::message.ticket_not_found'), '404');
            }
            return $this->sendResponse($this->ticketRepository->getTicketThreadById($id), '');
        } catch (\Exception $e) {
            return $this->sendExceptionError($e);
        }
    }

    public function getReferences()
    {
        try {
            $priorityOptions = [];
            $departmentOptions = [];
            $departments = $this->departmentRepository->all();
            $priorities = $this->priorityRepository->all();
            if (!empty($priorities)) {
                foreach ($priorities as $priority) {
                    $priorityOptions[$priority->id] = $priority->name;
                }
            }
            if (!empty($departments)) {
                foreach ($departments as $department) {
                    $departmentOptions[$department->id] = $department->name;
                }
            }
            return $this->sendResponse(['departments' => $departmentOptions, 'priorities' => $priorityOptions], '');
        } catch (\Exception $e) {
            return $this->sendExceptionError($e);
        }
    }

    public function create(TicketCreateRequest $request)
    {
        try {
            $inputs = $request->all();
            $ticketNumber = $this->ticketRepository->getTicketNumber();
            $user = Auth::user();
            $email = $user->email;
            $name = $user->username;
            if (!empty($user->first_name) || !empty($user->last_name)) {
                $name = $user->first_name . ' ' . $user->last_name;
            }
            $ticket = $this->ticketRepository->create([
                'subject' => $inputs['subject'],
                'status_id' => 1,
                'user_id' => Auth::id(),
                'department_id' => $inputs['department_id'],
                'email' => $email,
                'name' => $name,
                'priority_id' => $inputs['priority_id'],
                'supervisor_id' => null,
                'ticket_number' => $ticketNumber
            ]);
            $file = (!empty($inputs['file'])) ? $inputs['file'] : NULL;
            $this->contentRepository->create([
                'ticket_id' => $ticket->id,
                'user_id' => Auth::id(),
                'message' => $inputs['message'],
                'file_id' => $file,
            ]);

            event(new NewTicketEmailEvent($ticket));
            return $this->sendResponse([], trans('visiosoft.module.tickets::message.ticket_created'));
        } catch (\Exception $e) {
            return $this->sendExceptionError($e);
        }
    }

    public
    function answer(AnswerRequest $request, $id)
    {
        try {
            $inputs = $request->all();
            $ticket = $this->ticketRepository->newQuery()->where('id', $id)->where('user_id', Auth::id())->first();
            if (empty($ticket)) {
                return $this->sendError(trans('visiosoft.module.tickets::message.ticket_not_found'), '404');
            }
            $file = null;
            if (!empty($inputs['file'])) {
                $file = $inputs['file'];
            }
            $this->contentRepository->create([
                'ticket_id' => $ticket->id,
                'message' => $inputs['content'],
                'user_id' => Auth::id(),
                'file_id' => $file
            ]);
            $ticket->update(['status_id' => 1]);
            $person = $ticket->user;
            event(new TicketAnsweredEmailEvent($person, $ticket));
            return $this->sendResponse([], trans('visiosoft.module.tickets::message.message_sent'));
        } catch (\Exception $e) {
            return $this->sendExceptionError($e);
        }
    }

    public
    function close($id)
    {
        try {
            $ticket = $this->ticketRepository->newQuery()->where('id', $id)->where('user_id', Auth::id())->first();
            if (empty($ticket)) {
                return $this->sendError(trans('visiososft.module.tickets::message.ticket_not_found'), 404);
            }
            $ticket->update([
                'status_id' => 4,
                'closed_by_id' => Auth::id(),
                'closed_at' => date('Y-m-d H:i:s')
            ]);
            $person = $ticket->user;
            event(new TicketClosedEmailEvent($person, $ticket));
            return $this->sendResponse([], trans('visiosoft.module.tickets::message.ticket_closed'));
        } catch (\Exception $e) {
            return $this->sendExceptionError($e);
        }
    }
}