<?php

namespace Visiosoft\SwishModule\Http\Controller\Api;

use Illuminate\Support\Facades\Validator;
use Visiosoft\ConnectModule\Http\Controller\ApiBaseController;
use Visiosoft\SwishModule\Payment\Contract\PaymentRepositoryInterface;
use Visiosoft\SwishModule\Payment\Models\PaymentStatus;
use Visiosoft\SwishModule\SwishModule;
use Visiosoft\SwishModule\Helpers\PriceFormatterHelper;

class PaymentApiController extends ApiBaseController
{

    private array $request;
    private PaymentRepositoryInterface $paymentRepository;

    public function __construct()
    {
        $this->request = request()->all();
        $this->paymentRepository = app(PaymentRepositoryInterface::class);
    }

    public function index($instruction_uuid)
    {
        $intent = $this->paymentRepository->findBy('intent_uuid', $instruction_uuid);
        if (!$intent) {
            return $this->sendError('', 'Instruction Not Found');
        }


        return $this->sendResponse([
            'payer_alias' => $intent->payer,
            'amount' => (string)$intent->amount,
            'message' => $intent->message,
            'instruction_uuid' => $intent->intent_uuid,
            'payment_request_token' => $intent->intent_token,
            'status' => PaymentStatus::getKeyByValue($intent->status),
            'message' => $intent->message,
            'payee_payment_reference' => $intent->external_uid,
            'request_date' => $intent->intent_date,
            'success_date' => $intent->success_date
        ]);
    }

    public function store(): \Illuminate\Http\JsonResponse
    {
        if ($this->validateIntentParams()->fails()) {
            return $this->sendError('', $this->getValidatorIntentErrors(), 422);
        }

        $history = $this->paymentRepository->createHistory($this->request['payer_alias'], $this->request['amount']);

        try {
            $response = app(SwishModule::class)->createRequest("POST", 'v1/paymentrequests', [], json_encode($this->getIntentParams()));
            $intentData = $this->getIntentData($response);

            if ($intentData['status']) {
                $history->setExternalUID($this->request['payee_payment_reference']);
                $history->setIntentToken($intentData['payment_request_token'] ?? '');
                $history->setIntentUUID($intentData['instruction_uuid']);
                $history->setMessage($this->request['message'] ?? '');

                return $this->sendResponse([
                    'instruction_uuid' => $history->intent_uuid,
                    'payment_request_token' => $history->intent_token
                ]);

            } else {
                $history->setStatus(PaymentStatus::REQUEST_REJECTED());
                $history->setMessage($intentData['message']);

                return $this->sendError('', $intentData['message'], 422);
            }

        } catch (\Exception $e) {
            $history->setStatus(PaymentStatus::REQUEST_CANT_CREATED());
            return $this->sendError('', 'Server Error', 500);
        }
    }


    private function validateIntentParams(): \Illuminate\Contracts\Validation\Validator
    {
        Validator::extend('custom_amount_rule', function ($attribute, $value, $parameters, $validator) {

            $decimalAmount = (new PriceFormatterHelper())->toDecimal($value);

            if ($decimalAmount < 0.01) {
                return $validator->errors()->add('custom_amount_rule', 'Amount must be greater than 0.01.');
            }

            if ($decimalAmount > 999999999999.99) {
                return $validator->errors()->add('custom_amount_rule', 'Amount must be less than 999999999999.99');
            }

            return true;
        });

        return Validator::make($this->request, [
            'request_type' => 'required|string|in:E_COMMERCE,M_COMMERCE',
            'payer_alias' => 'required|string|min:10',
            'amount' => 'required|string|custom_amount_rule',
            'message' => 'string|max:50|regex:/^[a-zA-Z0-9!?()öÖ\s]+$/',
            'payee_payment_reference' => 'required|string|max:255'
        ]);
    }

    private function getValidatorIntentErrors(): string
    {
        return implode(" ", $this->validateIntentParams()->errors()->all());
    }

    private function getIntentParams(): array
    {
        $apiMode = setting_value('visiosoft.module.swish::api_mode');
        if ($apiMode === "live") {
            $callbackUrl = setting_value('visiosoft.module.swish::api_callback_url');
            $payeeAlias = setting_value('visiosoft.module.swish::swish_number');
        } else {
            $callbackUrl = setting_value('visiosoft.module.swish::api_sandbox_callback_url');
            $payeeAlias = setting_value('visiosoft.module.swish::sandbox_swish_number');
        }

        $intentParams = [
            'payeePaymentReference' => $this->request['payee_payment_reference'],
            'callbackUrl' => $callbackUrl,
            'payeeAlias' => $payeeAlias,
            "amount" => $this->request['amount'],
            "currency" => 'SEK',
            "message" => $this->request['message'] ?? ''
        ];

        if ($this->request['request_type'] === "E_COMMERCE") {
            $intentParams['payerAlias'] = $this->request['payer_alias'];
        }

        return $intentParams;
    }

    private function getIntentData($response): array
    {
        if ($response->success) {
            $headers = $response->data->headers;
            $data['status'] = true;
            $data['instruction_uuid'] = str_replace($response->data->url . "/", "", $headers->location[0]);
            $data['payment_request_token'] = isset($headers->paymentrequesttoken) ? $headers->paymentrequesttoken[0] : '';
            $data['message'] = "Intent Created....";
        } else {
            $data['status'] = false;
            $data['error_code'] = $response->data->response[0]->errorCode;
            $data['message'] = $response->data->response[0]->errorMessage;
        }
        return $data;
    }
}