<?php

namespace Visiosoft\SwishModule\Http\Controller\Api;

use Illuminate\Support\Facades\Validator;
use Visiosoft\ConnectModule\Http\Controller\ApiBaseController;
use Visiosoft\SwishModule\Payment\Contract\PaymentRepositoryInterface;
use Visiosoft\SwishModule\Payment\Models\PaymentStatus;
use Visiosoft\SwishModule\SwishModule;

class PaymentApiController extends ApiBaseController
{

    private array $request;
    private PaymentRepositoryInterface $paymentRepository;

    public function __construct()
    {
        $this->request = request()->all();
        $this->paymentRepository = app(PaymentRepositoryInterface::class);
    }

    public function index($intentUUID)
    {
        $intent = $this->paymentRepository->findBy('intent_uuid', $intentUUID);
        if (!$intent) {
            return $this->sendError('', 'Intent Not Found');
        }


        return $this->sendResponse([
            'payer' => $intent->payer,
            'amount' => $intent->amount,
            'message' => $intent->message,
            'intent_uuid' => $intent->intent_uuid,
            'intent_token' => $intent->intent_token,
            'intent_date' => $intent->date,
            'status' => PaymentStatus::getKeyByValue($intent->status),
            'message' => $intent->message,
            'external_uid' => $intent->external_uid,
            'intent_date' => $intent->intent_date,
            'success_date' => $intent->success_date
        ]);
    }

    public function store(): \Illuminate\Http\JsonResponse
    {
        if ($this->validateIntentParams()->fails()) {
            return $this->sendError('', $this->getValidatorIntentErrors(), 422);
        }

        $history = $this->paymentRepository->createHistory($this->request['payer'], $this->request['amount']);

        try {
            $response = app(SwishModule::class)->createRequest("POST", 'v1/paymentrequests', [], json_encode($this->getIntentParams()));
            $intentData = $this->getIntentData($response);

            if ($intentData['status']) {
                $history->setIntentToken($intentData['intent_token'] ?? '');
                $history->setIntentUUID($intentData['intent_uuid']);
                $history->setMessage($this->request['message'] ?? '');

                return $this->sendResponse([
                    'intent_uuid' => $history->intent_uuid,
                    'intent_token' => $history->intent_token
                ]);

            } else {
                $history->setStatus(PaymentStatus::INTENT_REJECTED());
                $history->setMessage($intentData['message']);

                return $this->sendError('', $intentData['message'], 422);
            }

        } catch (\Exception $e) {
            $history->setStatus(PaymentStatus::INTENT_CANT_CREATED());
            return $this->sendError('', 'Server Error', 500);
        }
    }


    private function validateIntentParams(): \Illuminate\Contracts\Validation\Validator
    {
        return Validator::make($this->request, [
            'payer' => 'required|string|min:10',
            'amount' => 'required|numeric|regex:/^\d+\.\d{2}$/',
            'message' => 'string|max:50|regex:/^[a-zA-Z0-9!?()öÖ]+$/',
            'external_uid' => 'required|string|max:255',
        ]);
    }

    private function getValidatorIntentErrors(): string
    {
        return implode(" ", $this->validateIntentParams()->errors()->all());
    }

    private function getIntentParams(): array
    {
        $apiMode = setting_value('visiosoft.module.swish::api_mode');
        if ($apiMode === "live") {
            $callbackUrl = setting_value('visiosoft.module.swish::api_callback_url');
            $payeeAlias = setting_value('visiosoft.module.swish::swish_number');
        } else {
            $callbackUrl = setting_value('visiosoft.module.swish::api_sandbox_callback_url');
            $payeeAlias = setting_value('visiosoft.module.swish::sandbox_swish_number');
        }

        return [
            'payeePaymentReference' => $this->request['external_uid'],
            'callbackUrl' => $callbackUrl,
            'payeeAlias' => $payeeAlias,
            'payerAlias' => $this->request['payer'],
            "amount" => $this->request['amount'],
            "currency" => 'SEK',
            "message" => $this->request['message'] ?? ''
        ];
    }

    private function getIntentData($response): array
    {
        if ($response->success) {
            $headers = $response->data->headers;
            $data['status'] = true;
            $data['intent_uuid'] = str_replace($response->data->url . "/", "", $headers->location[0]);
            $data['intent_token'] = isset($headers->paymentrequesttoken) ? $headers->paymentrequesttoken[0] : '';
            $data['message'] = "Intent Created....";
        } else {
            $data['status'] = false;
            $data['error_code'] = $response->data->response[0]->errorCode;
            $data['message'] = $response->data->response[0]->errorMessage;
        }
        return $data;
    }
}