<?php namespace Visiosoft\SubscriptionsModule\Subscription;

use DateTime;
use Visiosoft\SubscriptionsModule\Subscription\Contract\SubscriptionRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Visiosoft\SubscriptionsModule\Subscription\Event\SubscriptionEnabled;
use Visiosoft\SubscriptionsModule\Subscription\Event\SubscriptionSuspended;

class SubscriptionRepository extends EntryRepository implements SubscriptionRepositoryInterface
{

    /**
     * The entry model.
     *
     * @var SubscriptionModel
     */
    protected $model;

    /**
     * Create a new SubscriptionRepository instance.
     *
     * @param SubscriptionModel $model
     */
    public function __construct(SubscriptionModel $model)
    {
        $this->model = $model;
    }

    /**
     * @param $subscription_id
     * @return mixed|string|null
     */
    public function getRemaining($subscription_id, $type = "table")
    {
        $subscription = $this->find($subscription_id);
        if (!is_null($subscription)) {
            $expires = $subscription->expires_at;
            $start = now()
                ->format('Y-m-d H:i:s');

            $dStart = new DateTime($start);
            $dEnd = new DateTime($expires);
            $dDiff = $dDiff = $dStart
                ->diff($dEnd)
                ->format('%r%a');

            if ($type == "table") {
                return ($dDiff > 0) ? $dDiff . " "
                    . trans('visiosoft.module.subscriptions::field.day.name')
                    : "<font color='red'>" . trans('visiosoft.module.subscriptions::field.expired.name') . "</font>";
            }
            return $dDiff;
        }
        return null;
    }

    public function createdSubscription($subscription, $trial_enabled = true)
    {
        $trial = null;
        $expires = null;
        $start = date('Y-m-d H:i:s');

        if ($trial_enabled and !is_null($subscription->plan->trial_interval) and $subscription->plan->trial_interval != 0) {
            $trial = date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s') . ' + ' .
                $subscription->plan->trial_interval . ' ' . $subscription->plan->trial_period));
            $expires = $trial;
            $start = $trial;
        }

        if (!is_null($subscription->plan->interval) and $subscription->plan->interval != 0) {
            $expires = date('Y-m-d H:i:s', strtotime($start . ' + ' .
                $subscription->plan->interval . ' ' . $subscription->plan->interval_period));
        }

        $subscription->update(['expires_at' => $expires, 'trial_expires_at' => $trial]);
    }

    public function findByAssign($user_id)
    {
        return $this->newQuery()
            ->where('assign_id', $user_id)
            ->orderByDesc('id')
            ->first();
    }

    public function suspend($id)
    {
        if (!is_null($subscription = $this->find($id))) {
            $subscription->update([
                'enabled' => false,
                'suspend_at' => date('Y-m-d H:i:s')
            ]);

            event(new SubscriptionSuspended($subscription));
        }
    }

    public function enabled($id, $reNew = true)
    {
        if (!is_null($subscription = $this->find($id))) {
            $subscription->update([
                'enabled' => true,
                'suspend_at' => null
            ]);
            if ($reNew) {
                event(new SubscriptionEnabled($subscription));
            }
        }
    }
}
