<?php namespace Visiosoft\SubscriptionsModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Visiosoft\CartsModule\Cart\Command\GetCart;
use Visiosoft\OrdersModule\Order\Contract\OrderRepositoryInterface;
use Visiosoft\PaymentModule\Payment\Commands\Card;
use Visiosoft\PaymentModule\Payment\Commands\Pay;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;
use Visiosoft\PaymentModule\PaymentMethod\Contract\PaymentMethodRepositoryInterface;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SubscriptionsModule\Log\Contract\LogRepositoryInterface;
use Visiosoft\SubscriptionsModule\Plan\Contract\PlanRepositoryInterface;
use Visiosoft\SubscriptionsModule\Subscription\Contract\SubscriptionRepositoryInterface;
use Visiosoft\SubscriptionsModule\Subscription\Event\ExpiredSubscription;
use Visiosoft\SubscriptionsModule\Subscription\Event\RemainingSubscription;
use Visiosoft\SubscriptionsModule\Subscription\Event\SubscriptionChanged;
use Visiosoft\SubscriptionsModule\Subscription\Event\UnSuspendedSubscription;
use Visiosoft\SubscriptionsModule\Subscription\SubscriptionPaymentModel;

class SubscriptionsController extends ResourceController
{
    private $subscription;
    private $plan;
    private $site;
    private $orderRepository;
    private $log;
    private $paymentMethodRepository;
    private $paymentRepository;
    public function __construct(
        SubscriptionRepositoryInterface $subscription,
        PlanRepositoryInterface         $plan,
        SiteRepositoryInterface         $site,
        OrderRepositoryInterface        $orderRepository,
        LogRepositoryInterface          $log,
        PaymentMethodRepositoryInterface $paymentMethodRepository,
        PaymentRepositoryInterface $paymentRepository
    ) {
        $this->subscription = $subscription;
        $this->log = $log;
        $this->site = $site;
        $this->orderRepository = $orderRepository;
        $this->plan = $plan;
        $this->paymentMethodRepository = $paymentMethodRepository;
        $this->paymentRepository = $paymentRepository;
        parent::__construct();
    }

    public function subscriptionControl()
    {
        $subscriptions = $this->subscription->getActiveSubscriptions();

        foreach ($subscriptions as $subscription) {

            $remaining = $subscription->getRemaining();

            if (in_array((int)$remaining, [1, 3, 5])) {

                $findRemainingMail = $this->log->getFirstRemainingLog($subscription->getId(), $remaining);

                if (!$findRemainingMail) {

                    event(new RemainingSubscription($remaining, $subscription));

                    $this->log->createRemainingLog($subscription, $remaining);

                }

            } else if ($remaining < 0) {

                $findSuspendMail = $this->log->getFirstSuspendLog($subscription->getId());

                if (!$findSuspendMail) {

                    event(new ExpiredSubscription($subscription));

                    $subscription->suspend();

                    $this->log->createSuspendLog($subscription);
                }
            }
        }
    }

    public function detail($id)
    {
        if ($subscription = $this->subscription->find($id)) {
            $this->template->set('meta_title', $subscription->plan->name);

            return $this->view->make('visiosoft.module.subscriptions::subscriptions.detail', compact('subscription'));
        }
        $this->messages->error(trans('visiosoft.module.subscriptions::message.found_subscriptions'));

        return $this->redirect->route('visiosoft.module.subscriptions::my-subscriptions');
    }

    public function changePlanForPaddle($subscription_id, $paddle_plan_id)
    {
        $response = null;

        $params = array_merge([
            'subscription_id' => $subscription_id,
            'plan_id' => $paddle_plan_id,
            'vendor_id' => setting_value('visiosoft.module.paddle::vendor_id'),
            'vendor_auth_code' => setting_value('visiosoft.module.paddle::vendor_auth_code'),
        ]);

        $url = "https://vendors.paddle.com/api/2.0/subscription/users/update";
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
        curl_setopt($ch, CURLOPT_POST, 1);

        try {
            $response = curl_exec($ch);
        } catch (\Exception $exception) {
            $this->messages->error([$exception->getMessage()]);
        }
        curl_close($ch);

        return $response;
    }


    public function mySubscriptions()
    {
        $this->template->set('meta_title', trans('visiosoft.module.subscriptions::field.my_subscriptions'));

        return $this->view->make('visiosoft.module.subscriptions::subscriptions.list');
    }

    public function renew($id)
    {
        if ($subscription = $this->subscription->find($id)) {
            $plans = $this->plan->newQuery()->get();
            $this->template->set('meta_title', $subscription->plan->name);
            return $this->view->make('visiosoft.module.subscriptions::plan.upgrade', compact('plans', 'subscription'));
        }
        abort(404);
    }

    public function buyOrTry($id, $type)
    {
        if ($plan = $this->plan->find($id)) {
            if ($type == "try") {
                $all_subscriptions = $this->subscription->getByAssign(Auth::id())->pluck('plan_id')->all();
                if (!in_array($plan->id, $all_subscriptions)) {
                    //"payment false" means no payment has been made.
                    try {
                        $subscription = $this->subscription->createNew(Auth::id(), $plan, false);
                    } catch (\Exception $exception) {
                        $this->messages->error([$exception->getMessage()]);
                        return $this->redirect->route('buy-plan');
                    }
                    return $this->redirect->route('visiosoft.module.subscriptions::detail_subscription', ['id' => $subscription->getId()]);

                } else {

                    $this->messages->info(trans('visiosoft.module.subscriptions::message.trial_feature_user'));
                    return $this->redirect->route('buy-plan');

                }
            } else {
                try {
                    $cart = $this->dispatch(new GetCart());
                    $cart->add($plan, 1);
                } catch (\Exception $exception) {
                    $this->messages->error([$exception->getMessage()]);
                    return $this->redirect->route('buy-plan');
                }
                return redirect('cart');

            }
        }
    }

    public function upgradeOrRenew($source_subscription_id, $type, $target_plan_id)
    {
        if ($target_plan = $this->plan->find($target_plan_id) and $source_subscription = $this->subscription->find($source_subscription_id)) {
            if ($type == "renew") {
                if ($plan = $source_subscription->plan) {
                    try {
                        $cart = $this->dispatch(new GetCart());
                        $plan->related = $source_subscription;
                        $cart->add($plan, 1);
                    } catch (\Exception $exception) {
                        $this->messages->error([$exception->getMessage()]);
                        return $this->redirect->route('visiosoft.module.subscriptions::upgrade_subscription', ['id' => $source_subscription->getId()]);
                    }
                } else {
                    $this->messages->error(trans('visiosoft.module.subscriptions::message.not_found_plan'));
                    return $this->redirect->route('visiosoft.module.subscriptions::upgrade_subscription', ['id' => $source_subscription->getId()]);
                }
            } else {
                try {
                    $cart = $this->dispatch(new GetCart());
                    $target_plan->related = $source_subscription;
                    $cart->add($target_plan, 1);
                } catch (\Exception $exception) {
                    $this->messages->error([$exception->getMessage()]);
                    return $this->redirect->route('visiosoft.module.subscriptions::upgrade_subscription', ['id' => $source_subscription->getId()]);
                }
            }
            return redirect('cart');
        }
    }

    /**
     * @param $id
     * @return void
     * If the plan is paid, it redirects to the payment page. If it is free, it starts the subscription.
     */
    public function choosePlan($id)
    {
        $subscriptions = $this->subscription->getByAssign(Auth::id());
        if (count($subscriptions)) {
            $this->messages->error(trans('visiosoft.module.subscriptions::message.available_active_subscriptions'));
            return $this->redirect->to($this->request->get('redirect','/my-subscriptions'));
        }

        if ($plan = $this->plan->find($id)) {
            $price = $plan->price;

            // Free Plan
            if (!$price || $price == "0.00") {
                $this->subscription->createNew(Auth::id(), $plan, true);

                return $this->redirect->to($this->request->get('redirect','/my-subscriptions'));
            }

            // For paid subscription plans, an order record is first created and
            // directed to the payment page of this order.
            return $this->view->make('visiosoft.module.subscriptions::subscriptions/payment', compact('plan'));
        }
        abort(404);
    }

    /**
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function cancel($id)
    {
        // find subscription of user
        if ($subscription = $this->subscription->getByAssign(Auth::id())->find($id)) {
            // soft delete subscription
            $subscription->update(['enabled' => false]);
            $subscription->delete();
            $this->messages->success(trans('visiosoft.module.subscriptions::message.cancel_subscription_confirm'));
            return $this->redirect->to('/my-subscriptions');
        }
        abort(404);
    }

    /**
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Psr\Container\ContainerExceptionInterface
     * @throws \Psr\Container\NotFoundExceptionInterface
     *
     * From /choose-plan/{plan_id} page, this function is called. If the plan is paid, this function will run.
     * this will save card and run payment function.
     */
    public function saveAndPay()
    {
        $user = Auth::user();
        $card = new Card();
        $plan_id = $this->request->get('plan_id');

        try {
            $cardUserKey = $this->paymentMethodRepository->getCustomerIdByService($user->id, $card->getProviderSlug());

            $providerResponse = $card->createCard(
                $this->request->get('name'),
                preg_replace('/\s+/', '', $this->request->get('ccno')),
                $this->request->get('expire_year'),
                $this->request->get('expire_month'),
                $this->request->get('cvv'),
                $user->email,
                $cardUserKey
            );


            if ($providerResponse['status']) {
                $createCard = [
                    'owner_id' => Auth::id(),
                    'service' => $providerResponse['data']['service'],
                    'service_payment_method_id' => $providerResponse['data']['service_payment_method_id'],
                    'customer' => $providerResponse['data']['customer'],
                    'card' => $providerResponse['data']['card'],
                    'holder_name' => $user->display_name,
                ];

                $this->paymentMethodRepository->addPaymentMethod($createCard);


            } else {
                $this->messages->error(trans('visiosoft.module.payment::message.cant_add_card'));
                return $this->redirect->route('subscriptions::choose-plan', [$plan_id]);
            }

            $plan = $this->plan->find($plan_id);
            $subscription = $this->subscription->createNew($user->id, $plan, false);
            if ($this->paySubscription($subscription)) {
                $this->messages->success(trans('visiosoft.module.payment::message.subscription_payment_success'));
            } else {
                $this->messages->error(trans('visiosoft.module.payment::message.subscription_payment_failed'));
            }

            return $this->redirect->route('visiosoft.module.subscriptions::my-subscriptions');
        } catch (\Exception $exception) {
            $logger = new Logger('subscription_payment');
            $logger->pushHandler(new StreamHandler(storage_path('logs/subscription_payment.log')), Logger::ERROR);
            $logger->error($exception->getMessage());
            $this->messages->error(trans('visiosoft.module.payment::message.cant_add_card'));
            return $this->redirect->route('subscriptions::choose-plan', [$plan_id]);
        }

    }

    public function periodicPay() {
        try {
            $this->subscription->getActiveSubscriptions()->each(function ($subscription) {
                if ($subscription->expires_at->isToday()) {
                    if ($this->paySubscription($subscription)) {

                    }
                }
            });
        } catch (\Exception $exception) {
            $logger = new Logger('subscription_payment');
            $logger->pushHandler(new StreamHandler(storage_path('logs/subscription_payment.log')), Logger::ERROR);
            $logger->error($exception->getMessage());
        }
    }

    public function paySubscription($subscription) {
        $user = $subscription->assign;
        $plan = $subscription->plan;
        $payment = new Pay();

        $paymentMethod = $this->paymentMethodRepository->getCardToPay($user->id);
        if (!$paymentMethod) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.card')]));
        };

        $providerUserID = $paymentMethod->customer;
        $defaultCardID = $paymentMethod->service_payment_method_id;
        $price = $plan->price;
        $currency = setting_value('streams::currency');
        $items = [
            [
            "name" => $plan->name,
            "price" => $plan->price,
            "quantity" => 1,
            "description" => trans('visiosoft.module.subscriptions::message.subscription_payment_success')
        ]
        ];
        $invoice_id = $subscription->id;

        try {
            $paymentRequest = $payment->payBySavedCard($providerUserID, $defaultCardID, $price, $currency, $items, [], $invoice_id);
            // payment status check for saved card payment
            if ($paymentRequest['data']['status'] == "success") {
                // change subscription to enabled
                $subscription->update(['enabled' => 1]);

                // send status as paid
                $payment_status = 2;

                // send user a notification
                event(new UnSuspendedSubscription($subscription));
            } else if ($paymentRequest['data']['status'] == "fail") {
                // delete the subscription
                $subscription->delete();

                // send status as failed
                $payment_status = 4;

                // send user a notification
                event(new ExpiredSubscription($subscription));
            } else {
                // send status as pending
                $payment_status = 1;
            }


            // create payment
            return $this->createPayment($user->id, $price, $paymentRequest['data']['payment_intent_id'], $subscription->id, $payment_status, 0, $currency);

        } catch (\Exception $e) {
            throw new \Exception(trans('visiosoft.module.payment::message.error_payment'));
        }
    }

    /**
     * IMPORTANT: $transactionId will hold the id of the subscription.
     */
    public function createPayment($userId, $amount, $paymentIntentId, $transactionId, $status, $categoryId = 0, $currency = 'TRY') {
        // TODO: transaction_id holds the subscription id, a new column should be added to the payments table.
        $paymentData = [
            'user_id' => $userId,
            'amount' => $amount,
            'usage' => $amount,
            'transaction_id' => $transactionId,
            'status_id' => $status,
            'category_id' => $categoryId,
            'currency' => $currency,
            'payment_intent_id' => $paymentIntentId,
            "created_at" =>  \Carbon\Carbon::now(),
            "updated_at" => \Carbon\Carbon::now(),
            "created_by_id" => $userId,
            "updated_by_id" => $userId,
        ];


        $paymentData['tax_percent'] = setting_value('visiosoft.module.subscriptions::tax_percent', 20);
        $paymentData['payment_amount'] = $amount;
        $paymentData['tax_amount'] = $amount * $paymentData['tax_percent'] / 100;
        $paymentData['additional_fees'] = 0;

        // inserting payment to database with DB facade because of the model has
        // unnecessary relations (charging, vehicle etc.)
        // TODO: This is not best practice but it is the fastest way to do it.
        $create = DB::table('payment_payment')->insert($paymentData);

        if (!$create) {
            return false;
        }

        return $create;
    }


    public function listPayments() {

        $user = Auth::user();
        $subscriptions = $this->subscription
            ->getByAssign($user->id);
        if (!$subscriptions) {
            return $this->redirect->back();
        } else {
            $subscriptions = $subscriptions->pluck('id')->toArray();
        }

        $subscriptionPayments = $this->paymentRepository
            ->newQuery()
            ->whereIn('transaction_id', $subscriptions)
            ->get();

        $paymentCard = $this->paymentMethodRepository->getCardToPay($user->id)->card;

        return $this->view->make('visiosoft.module.subscriptions::subscriptions/payment-list', compact('subscriptionPayments', 'paymentCard'));

    }



}
