<?php namespace Visiosoft\SubscriptionsModule\Subscription;

use DateTime;
use Visiosoft\SubscriptionsModule\Subscription\Contract\SubscriptionRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Visiosoft\SubscriptionsModule\Subscription\Event\SubscriptionEnabled;
use Visiosoft\SubscriptionsModule\Subscription\Event\SubscriptionSuspended;

class SubscriptionRepository extends EntryRepository implements SubscriptionRepositoryInterface
{

    /**
     * The entry model.
     *
     * @var SubscriptionModel
     */
    protected $model;

    /**
     * Create a new SubscriptionRepository instance.
     *
     * @param SubscriptionModel $model
     */
    public function __construct(SubscriptionModel $model)
    {
        $this->model = $model;
    }

    /**
     * @param $subscription_id
     * @return mixed|string|null
     */
    public function getRemaining($subscription_id, $type = "table")
    {
        $subscription = $this->find($subscription_id);
        if (!is_null($subscription)) {
            $expires = $subscription->expires_at;
            $start = now()
                ->format('Y-m-d H:i:s');

            $dStart = new DateTime($start);
            $dEnd = new DateTime($expires);
            $dDiff = $dDiff = $dStart
                ->diff($dEnd)
                ->format('%r%a');

            if ($type == "table") {
                return ($dDiff > 0) ? $dDiff . " "
                    . trans('visiosoft.module.subscriptions::field.day.name')
                    : "<font color='red'>" . trans('visiosoft.module.subscriptions::field.expired.name') . "</font>";
            }
            return $dDiff;
        }
        return null;
    }

    public function createdSubscription($subscription, $trial_enabled = true)
    {
        $trial = null;
        $expires = null;
        $start = date('Y-m-d H:i:s');
        $plan = $subscription->plan;

        if ($trial_enabled and $plan->trial_interval) {
            //Deneme var ve kullanılabilir

            $trial = date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s') . ' + ' . $plan->trial_interval . ' ' . $plan->trial_period));
            $expires = $trial;
            $start = $trial;
        }

        if ($plan->interval) {
            $expires = date('Y-m-d H:i:s', strtotime($start . ' + ' . $plan->interval . ' ' . $plan->interval_period));
        }

        $subscription->update(['expires_at' => $expires, 'trial_expires_at' => $trial]);
    }

    public function getByAssign($user_id)
    {
        return $this->newQuery()
            ->where('assign_id', $user_id)
            ->orderByDesc('id')
            ->get();
    }


    /**
     * @param $id
     * @return mixed|void
     */
    public function enabled($id)
    {
        $this->find($id)->update([
            'enabled' => true,
            'suspend_at' => null
        ]);
    }

    public function checkSubscription($id)
    {
        $remaining = $this->getRemaining($id, null);

        if ((int)$remaining > -1 and $remaining !== "-0") {
            return $this->find($id);
        }
        return null;
    }

    /**
     * @param $id
     * @return mixed
     */
    public function findByPaddle($id)
    {
        $subscription = $this->newQuery()->where('paddle_subscription_id', $id)->get();
        return ($this->count($subscription)) ? $subscription->first() : null;
    }

    public function getActiveSubscriptions()
    {
        return $this->newQuery()
            ->where('suspend_at', null)
            ->get();
    }

    public function getWithExpired($date, $operator = '=')
    {
        return $this->newQuery()
            ->where('expires_at', $operator, $date)
            ->get();
    }
}
