<?php namespace Visiosoft\StoreModule\Store;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Visiosoft\CatsModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\StoreModule\Store\Contract\StoreRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Visiosoft\StoreModule\User\Contract\UserRepositoryInterface;

class StoreRepository extends EntryRepository implements StoreRepositoryInterface
{
    protected $model;
    protected $categoryRepository;
    protected $storeUserRepository;

    public function __construct(
        StoreModel $model,
        CategoryRepositoryInterface $categoryRepository,
        UserRepositoryInterface $storeUserRepository
    )
    {
        $this->model = $model;
        $this->categoryRepository = $categoryRepository;
        $this->storeUserRepository = $storeUserRepository;
    }

    public function searchStores($param = null, $limit = null, $get_query = false)
    {
        $query = $this->model;
        $query = $query->where('store_store.slug', '!=', "")->where('store_store.status', 'approved');
        $query = $query->leftJoin('store_store_translations', function ($join) {
            $join->on('store_store.id', '=', 'store_store_translations.entry_id');
            $join->where('store_store_translations.locale', '=', Request()->session()->get('_locale', setting_value('streams::default_locale')));
        });

        if (!empty($param['keyword'])) {
            $query = $query->where('store_store.summary', 'LIKE', '%' . $param['keyword'] . '%')
                ->orWhere('store_store_translations.name', 'like', '%' . $param['keyword'] . '%');
        }
        if (!empty($param['country'])) {
            $query = $query->where('country_id', $param['country']);
        }
        if (isset($param['city']) and (!empty($param['city']))) {
            $citiesIds = explode(',', $param['city'][0]);
            $query = $query->whereIn('city', $citiesIds);
        }
        if (!empty($param['cat'])) {
            if (is_array($param['cat'])){
                $query = $query->whereIn('category', $param['cat']);
            } else {
                $query = $query->where('category', $param['cat']);
            }
        }
        if (!empty($param['gold'])) {
            $query = $query->where('gold', 1);
        }
        if (!empty($param['date'])) {
            if ($param['date'] === 1) {
                $query = $query->where('store_store.created_at', '>=', Carbon::now()->subYears(1));
            } elseif ($param['date'] === 2) {
                $query = $query->where('store_store.created_at', '>=', Carbon::now()->subYears(2));
            } elseif ($param['date'] === 3) {
                $query = $query->where('store_store.created_at', '>=', Carbon::now()->subYears(3));
            }
        }

        if ($get_query) {
            return $query;
        }

        $query = $query->orderBy('store_store.created_at', 'desc');

        return $query->paginate(setting_value('streams::per_page'));
    }

    public function findBySlug($slug)
    {
        return $this->findBy('slug', $slug);
    }

    public function findByUser($userId)
    {
        return $this->findBy('user_id', $userId);
    }

    public function getApprovedStores($orderBy = 'created_at', $orderByDir = 'desc', $limit = true)
    {
        $stores = $this->newQuery()
            ->where('status', 'approved')
            ->orderBy($orderBy, $orderByDir);

        if ($limit) {
            $stores = $stores->limit(setting_value('visiosoft.module.store::homeShowcaseLimit', 5));
        }

        return $stores->get();
    }

    public function getGoldStores($orderBy = 'created_at', $orderByDir = 'desc', $limit = true)
    {
        $stores = $this->newQuery()
            ->where('gold_supplier', 1)
            ->where('status', 'approved')
            ->orderBy($orderBy, $orderByDir);

        if ($limit) {
            $stores = $stores->limit(setting_value('visiosoft.module.store::homeShowcaseLimit', 5));
        }

        return $stores->get();
    }

    public function getNonGoldStores($orderBy = 'created_at', $orderByDir = 'desc', $limit = true)
    {
        $stores = $this->newQuery()
            ->where('gold_supplier', 0)
            ->where('status', 'approved')
            ->orderBy($orderBy, $orderByDir);

        if ($limit) {
            $stores = $stores->limit(setting_value('visiosoft.module.store::homeShowcaseLimit', 5));
        }

        return $stores->get();
    }

    public function countByCat($catID)
    {
        return DB::table('store_store')
            ->where('status', 'approved')
            ->where('category', $catID)
            ->count();
    }

    public function getStoresCount()
    {
        return DB::table('store_store')->count();
    }

    public function getApprovedStoresByIds(array $ids)
    {
        return $this->newQuery()
            ->where('status', 'approved')
            ->whereIn('id', $ids)
            ->get();
    }

    public function getStoresByCoordinates($lat, $lng, $distance = 50)
    {
        return $this
            ->getModel()
            ->where('status', 'approved')
            ->whereNotNull('lat')
            ->whereNotNull('lng')
            ->select(
                DB::raw("*, ( 3959 * acos( cos( radians('$lat') ) * cos( radians( lat ) ) * cos( radians( lng ) - radians('$lng') ) + sin( radians('$lat') ) * sin( radians( lat ) ) ) ) AS distance")
            )
            ->havingRaw("distance < $distance")
            ->orderBy('distance')
            ->get();
    }
}
