<?php namespace Visiosoft\StoreModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Str;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;
use Visiosoft\CartsModule\Cart\Command\GetCart;
use Visiosoft\CatsModule\Category\CategoryModel;
use Visiosoft\CatsModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\StoreModule\Category\Contract\CategoryRepositoryInterface as StoreCategory;
use Visiosoft\StoreModule\Price\Contract\PriceRepositoryInterface;
use Visiosoft\StoreModule\Store\Command\ConfigureSubdomain;
use Visiosoft\StoreModule\Store\Contract\StoreRepositoryInterface;
use Visiosoft\StoreModule\Store\Form\StoreFormBuilder;
use Visiosoft\StoreModule\Store\StoreModel;

class StoreController extends PublicController
{
    private $advModel;
    private $storeUserRepository;
    private $advRepository;
    private $storeRepository;
    private $categoryRepository;
    private $categoryModel;
    private $userRepository;
    private $priceRepository;
    private $storeCategory;
    private $storeModel;

    public function __construct(
        AdvModel $advModel,
        \Visiosoft\StoreModule\User\Contract\UserRepositoryInterface $storeUserRepository,
        AdvRepositoryInterface $advRepository,
        StoreRepositoryInterface $storeRepository,
        CategoryRepositoryInterface $categoryRepository,
        CategoryModel $categoryModel,
        UserRepositoryInterface $userRepository,
        PriceRepositoryInterface $priceRepository,
        StoreCategory $storeCategory,
        StoreModel $storeModel
    )
    {
        parent::__construct();
        $this->advModel = $advModel;
        $this->storeUserRepository = $storeUserRepository;
        $this->advRepository = $advRepository;
        $this->storeRepository = $storeRepository;
        $this->categoryRepository = $categoryRepository;
        $this->categoryModel = $categoryModel;
        $this->userRepository = $userRepository;
        $this->priceRepository = $priceRepository;
        $this->storeCategory = $storeCategory;
        $this->storeModel = $storeModel;
    }

    public function index($slug)
    {
        // Redirect if subdomain is allowed
        $domain = setting_value('streams::domain');
        if (\request()->getHttpHost() == $domain && setting_value('visiosoft.module.store::use_subdomain_routing')) {
            return redirect(\request()->getScheme() . '://' . $slug . '.' . $domain);
        }

        $slug = str_slug($slug, '_');
        $store = $this->storeRepository->findBy('slug', $slug);
        if ($store && ($store->isApproved() || $store->getOwner()->id === \auth()->id())) {
            $storeUsers = $store->getStoreUsers();
            $users = $storeUsers;

            $user = null;
            if ($userId = \request()->userId) {
                $isStoreUser = $store->user_id === $userId;
                if (!$isStoreUser) {
                    $isStoreUser = $this->storeUserRepository->findUser($userId, $store->id, true);
                }
                if ($isStoreUser) {
                    $user = $this->userRepository->newQuery()->where('id', $userId)->get();
                    if (count($user)) {
                        $users = $user;
                    }
                }
            }
            $usersIds = $users->pluck('id');

            $rawAds = $this->advRepository->getByUsersIDs($usersIds);
            if ($catReq = \request()->cat) {
                $rawAds = $this->checkCatParam($catReq, $rawAds);
            }
            $ads = clone $rawAds;
            $ads = $ads->paginate(setting_value('streams::per_page'));

            if (setting_value('visiosoft.module.advs::hide_out_of_stock_products_without_listing')) {
                $ads = $ads->filter(
                    function ($entry) {
                        return (($entry->is_get_adv == true && $entry->stock > 0) || ($entry->is_get_adv == false));
                    }
                );
            }

            $advs = $this->advRepository->addAttributes($ads);
            foreach ($advs as $index => $ad) {
                $advs[$index]->detail_url = $this->advModel->getAdvDetailLinkByModel($ad, 'list');
                $advs[$index] = $this->advModel->AddAdsDefaultCoverImage($ad);
            }

            $previousCat = null;
            $subCats = array();
            for ($i = 1; $i <= 10; $i++) {
                $cats = $rawAds
                    ->groupBy('cat' . $i)
                    ->pluck('cat' . $i)
                    ->toArray();
                if (count($cats) === 1 && !is_null($cats[0])) {
                    $previousCat = $cats;
                    continue;
                }
                if (!is_null($previousCat)) {
                    if ($mainCats = $this->categoryModel->getMains($previousCat)) {
                        $currentCat = $this->categoryModel->getCat($previousCat);
                        $mainCats[] = [
                            'id' => $currentCat->id,
                            'val' => $currentCat->name,
                        ];
                        $subCats = array();
                        foreach ($cats as $cat) {
                            $subCats[] = $this->categoryRepository->find($cat);
                        }
                    }
                } else {
                    $rawCats = array();
                    foreach ($cats as $cat) {
                        $rawCats[] = $this->categoryRepository->find($cat);
                    }
                    $mainCats = array();
                    $subCats = array();
                    foreach ($rawCats as $rawCat) {
                        if ($rawCat) {
                            $mainCats[] = [
                                'id' => $rawCat->id,
                                'val' => $rawCat->name,
                                'parentCat' => true
                            ];
                        }
                    }
                }
                break;
            }

            $default_detail_page = setting_value('visiosoft.module.store::default_detail_page');
            $default_detail_page = $this->renderStoreHtml($store, $default_detail_page);

            $compact = compact('advs', 'mainCats', 'subCats', 'store', 'storeUsers', 'user', 'default_detail_page');
            return $this->view->make('visiosoft.module.store::ads-list/list', $compact);
        }

        return $this->view->make('visiosoft.module.advs::error-msg')
            ->with('message', trans('visiosoft.module.store::message.this_store_is_not_available'));
    }

    public function renderStoreHtml($store, $default_detail_page)
    {
        $fields = array(
            "land_phone", "gsm_phone", "slug", "summary", "detailed_description", "address",
            "email", "web_site", "facebook", "instagram", "twitter", "tax_number", "lat", "lng"
        );

        foreach ($fields as $field) {
            $default_detail_page = str_replace("{{" . $field . "}}", $store->{$field}, $default_detail_page);
        }

        return $default_detail_page;
    }

    public function checkCatParam($catParam, $userAds)
    {
        $cat_d = $this->categoryRepository->find($catParam);

        if ($cat_d->parent_category_id == null) {
            $catLevel = 1;
        } else {
            $catLevel = $this->categoryRepository->getModel()->getCatLevel($catParam);
        }

        $catLevel = "cat" . $catLevel;

        return $userAds->where($catLevel, $catParam);
    }

    public function edit(StoreFormBuilder $form, $slug, $id)
    {
        $store = $this->storeRepository->find($id);
        if ($store and $store->user_id == Auth::id()) {
            if (\request()->action == "save") {
                $form->skipField('user');
                $form->skipField('status');
                if ($form->hasFormErrors()) {
                    return back();
                }
                $form->render($id);

                if ($form->hasFormErrors()) {

                    foreach ($form->getFormErrors()->getMessages() as $formError) {
                        $this->messages->error($formError);
                    }
                    return Redirect::back()->with('error', $this->messages->get('error'))->withInput();
                }

                $this->messages->success('visiosoft.module.store::message.store_updated_successfully');

                return redirect(route('store::edit', [$store->slug, $store->id]));
            }

            $otherFields = $store->getAssignments()->notLocked()->fieldSlugs();

            return $this->view->make('visiosoft.module.store::edit', compact('store', 'otherFields'));
        }
        abort(403);
    }

    public function storeDetail($slug)
    {
        $store = $this->storeRepository->findBy('slug', $slug);
        if ($store && ($store->isApproved() || $store->getOwner()->id === \auth()->id())) {
            $storeUsers = $store->getStoreUsers();

            //Set Meta
            $this->template->set('meta_title', ($store->meta_title) ? $store->meta_title : $store->name);
            $this->template->set('meta_description', ($store->meta_description) ? $store->meta_description : strip_tags($store->detailed_description, ''));
            $this->template->set('meta_keywords', implode(', ', $store->meta_keywords));

            return $this->view->make('visiosoft.module.store::detail', compact('store', 'storeUsers'));
        }
        abort(404);
    }

    public function create(StoreFormBuilder $form)
    {
        if (\request()->action == "save") {
            $error = $form->build()->validate()->getFormErrors()->getMessages();
            if (!empty($error)) {
                return back();
            }

            // Check if slug is taken
            $locale = config('app.locale');
            $slug = \request('slug') ?: Str::slug(\request("name_$locale"), '_');
            if ($this->storeRepository->findBySlug($slug)) {
                $this->messages->error(trans('visiosoft.module.store::message.the_store_name_has_been_taken'));
                return back();
            }

            $form->setFormValue('slug', $slug);

            $form->render();

            if (setting_value('visiosoft.module.store::enable_store_prices')
                && is_module_installed('visiosoft.module.carts')) {
                $store = $this->storeRepository->findBy('slug', $slug);
                $storePrice = $this->priceRepository->findBy('category', $store->category);
                $store->price = $storePrice ? $storePrice->price
                    : setting_value('visiosoft.module.store::default_store_price');

                if (is_module_installed('visiosoft.module.carts')) {
                    $cart = $this->dispatch(new GetCart());
                    $cart->add($store, 1);
                    return redirect()->route('visiosoft.module.carts::cart');
                }
            }

            return redirect(route('store::view', [$slug]));
        }

        $mainCats = $this->categoryRepository->getMainCategories();

        return $this->view->make('visiosoft.module.store::create', compact('mainCats'));
    }

    public function addUser(Request $request, UserRepositoryInterface $userRepository, $slug)
    {
        $this->middleware('auth');

        $store = $this->storeRepository->findBy('slug', $slug);
        $users = $this->storeUserRepository->getUsers($store->id);
        if ($store and $store->user_id == Auth::id()) {
            if ($request->action == "save") {
                $user = $request->user_email ?
                    $userRepository->findByEmail($request->user_email) :
                    $userRepository->findBy('gsm_phone', $request->user_phone);
                if ($user != null) {
                    $findUser = $this->storeUserRepository->findUser($user->id, $store->id);
                    if ($store->user_id != $user->id and count($findUser) == 0) {
                        $this->storeUserRepository->addUser($user->id, $store->id);
                        return redirect('s/' . $store->slug . '/add-user')->withSuccess([trans('visiosoft.module.store::message.user_added')]);
                    } else {
                        return $this->redirect->back()->with('error', [trans('visiosoft.module.store::message.already_defined_user')]);
                    }
                } else {
                    return $this->redirect->back()->with('error', [trans('visiosoft.module.store::message.user_not_found')]);
                }
            }
            return $this->view->make('visiosoft.module.store::add-user', compact('store', 'users'));
        }
        abort(403);
    }

    public function removeUser($slug, $userId)
    {
        $this->middleware('auth');

        $store = $this->storeRepository->findBy('slug', $slug);
        if ($store and $store->user_id == Auth::id()) {
            $find_user = $this->storeUserRepository->findUser($userId, $store->id);
            if ($store->user_id != $userId and count($find_user) != 0) {
                $this->storeUserRepository->removeUser($userId, $store->id);
                return redirect('s/' . $store->slug . '/add-user')->withSuccess([trans('visiosoft.module.store::message.user_deleted')]);
            } else {
                return $this->redirect->back()->with('error', [trans('visiosoft.module.store::message.user_not_found')]);
            }
        }
        abort(403);
    }

    public function listStores(Request $request)
    {
        $params = $request->toArray();

        $stores = $this->storeRepository->searchStores($params);

        $cats = $this->categoryRepository->getMainCategories();

        $fields = $this->storeModel->getAssignments()->notLocked();

        $storeCats = $this->storeCategory->all();
        $storeCategories = $this->storeCategory->getStoreCategories();
        $hideLocationFilter = false;
        if (
            ($locationFilterCats = setting_value('visiosoft.module.store::hide_location_filter_categories'))
            && is_array($locationFilterCats)
            && count($locationFilterCats)
            && \request()->cat
        ) {
            $hideLocationFilter = in_array(\request()->cat, $locationFilterCats);
        }

        $is_store = true;
        //For AddBlock Override View
        $filter_params = [];

        $compact = compact('stores', 'request', 'cats', 'storeCats', 'fields', 'hideLocationFilter','filter_params', 'is_store', 'storeCategories');

        Cookie::queue(Cookie::make('last_search', $request->getRequestUri(), 84000));

        $seoKeyword = setting_value('visiosoft.module.store::list_seo_keywords');
        if ($seoKeyword && count($seoKeyword)) {
            $this->template->set('meta_keywords', implode(', ', $seoKeyword));
        }

        return $this->view->make('visiosoft.module.store::list/list', $compact);
    }

    public function listCompanies(Request $request)
    {
        $params = $request->toArray();

        $stores = $this->storeRepository->searchCompanies($params);

        $cats = $this->categoryRepository->getMainCategories();

        $fields = $this->storeModel->getAssignments()->notLocked();

        $storeCats = $this->storeCategory->all();
        $companyCategories = $this->storeCategory->getCompanyCategories();
        $hideLocationFilter = false;
        if (
            ($locationFilterCats = setting_value('visiosoft.module.store::hide_location_filter_categories'))
            && is_array($locationFilterCats)
            && count($locationFilterCats)
            && \request()->cat
        ) {
            $hideLocationFilter = in_array(\request()->cat, $locationFilterCats);
        }

        //For AddBlock Override View
        $filter_params = [];

        $is_company = true;

        $compact = compact('stores', 'request', 'cats', 'storeCats', 'fields', 'hideLocationFilter','filter_params', 'is_company', 'companyCategories');

        Cookie::queue(Cookie::make('last_search', $request->getRequestUri(), 84000));

        $seoKeyword = setting_value('visiosoft.module.store::list_seo_keywords');
        if ($seoKeyword && count($seoKeyword)) {
            $this->template->set('meta_keywords', implode(', ', $seoKeyword));
        }

        return $this->view->make('visiosoft.module.store::company-list/list', $compact);
    }

    public function panel($slug)
    {
        $store = $this->storeRepository->findBySlug($slug);

        if ($store->user_id !== \auth()->id()) {
            abort(403, trans('visiosoft.module.store::message.unauthorized_access'));
        }

        $storeUsers = $store->getStoreUsers();
        $usersIds = $storeUsers->pluck('id');

        $ads = $this->advRepository
            ->getByUsersIDs($usersIds, false, true)
            ->get();

        $ads = $this->advRepository->addAttributes($ads);
        foreach ($ads as $index => $ad) {
            $ads[$index]->detail_url = $this->advModel->getAdvDetailLinkByModel($ad, 'list');
            $ads[$index] = $this->advModel->AddAdsDefaultCoverImage($ad);
        }

        return view('visiosoft.module.store::store.panel', compact('store', 'ads'));
    }

    public function addDomain($id)
    {
        if (($store = $this->storeRepository->find($id)) && !$store->has_domain) {
            $this->dispatch(new ConfigureSubdomain($store));

            $store->update([
                'has_domain' => true
            ]);
        }

        return redirect()->route('visiosoft.module.store::admin_store');
    }

    public function showConsultant($storeSlug, $consultantID)
    {
        if (!$store = $this->storeRepository->findBySlug($storeSlug)) {
            abort(404);
        }

        if (!in_array($consultantID, $store->getStoreUsers()->pluck('id')->all())) {
            abort(404);
        }

        $consultant = $this->userRepository->find($consultantID);

        return view('visiosoft.module.store::store.consultant')->with('consultant', $consultant);
    }
}
