<?php namespace Visiosoft\SocialModule\Provider;

use Visiosoft\SocialModule\Authentication\Contract\AuthenticationRepositoryInterface;
use Anomaly\UsersModule\Role\Command\GetRole;
use Anomaly\UsersModule\User\Contract\UserInterface;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Anomaly\UsersModule\User\Event\UserHasRegistered;
use Illuminate\Contracts\Auth\Guard;
use Illuminate\Contracts\Config\Repository;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Laravel\Socialite\Contracts\User;
use Visiosoft\SocialModule\Events\SocialRegisterWasCreated;

/**
 * Class ProviderUsers
 *
 * @link   http://visiosoft.com.tr/
 * @author Visiosoft, Inc. <support@visiosoft.com.tr>
 * @author Vedat Akdoğan <vedat@visiosoft.com.tr>
 */
class ProviderUsers
{

    use DispatchesJobs;

    /**
     * The auth guard.
     *
     * @var Guard
     */
    protected $auth;

    /**
     * The users repository.
     *
     * @var UserRepositoryInterface
     */
    protected $users;

    /**
     * The config repository.
     *
     * @var Repository
     */
    protected $config;

    /**
     * The event dispatcher.
     *
     * @var Dispatcher
     */
    protected $events;

    /**
     * The authentication repository.
     *
     * @var AuthenticationRepositoryInterface
     */
    protected $authentications;

    /**
     * Create a new ProviderUsers instance.
     *
     * @param Guard $auth
     * @param Repository $config
     * @param Dispatcher $events
     * @param UserRepositoryInterface $users
     * @param AuthenticationRepositoryInterface $authentications
     */
    public function __construct(
        Guard $auth,
        Repository $config,
        Dispatcher $events,
        UserRepositoryInterface $users,
        AuthenticationRepositoryInterface $authentications
    )
    {
        $this->auth = $auth;
        $this->users = $users;
        $this->config = $config;
        $this->events = $events;
        $this->authentications = $authentications;
    }

    /**
     * Resolve a user instance from
     * a social provider user account.
     *
     * @param User $account
     * @param null $register
     * @return UserInterface|\Illuminate\Contracts\Auth\Authenticatable|null
     */
    public function user(User $account, $register = null)
    {

        /**
         * If a user is logged in then
         * return that user instance.
         */
        if ($user = $this->auth->user()) {
            return $user;
        }

        /**
         * Check and see if this account has already
         * been saved and use the authentication user.
         */
        $authentication = $this->authentications->findByUid($account->getId());

        if ($authentication && $user = $authentication->getUser()) {
            return $user;
        }

        /**
         * If we can find a user by their email
         * then return the matched user instance.
         */
        if ($user = $this->users->findByEmail($account->getEmail())) {
            return $user;
        }

        /**
         * Make sure it's OK to register.
         */
        if ($register !== true) {
            return null;
        }

        /**
         * Create some default information just in case.
         */
        $email = str_slug($account->getName(), '-') . '@' . $account->getId() . '.com';

        /**
         * If we get this far we need to create
         * a user and return the new user instance.
         *
         * @var UserInterface $user
         */
        $user = $this->users->unguard()->create(
            [
                'activated' => true,
                'password' => str_random(),
                'display_name' => $account->getName(),
                'first_name' => explode(' ', $account->getName())[0],
                'last_name' => isset(explode(' ', $account->getName())[1]) ? explode(' ', $account->getName())[1] : '',
                'email' => $account->getEmail() ?: $email,
                'username' => $account->getNickname() . $account->getId(),
            ]
        );

        $this->users->guard();

        foreach ($this->config->get('visiosoft.module.social::config.registration_groups', []) as $role) {
            $user->attachRole($this->dispatch(new GetRole($role)));
        }

        event(new SocialRegisterWasCreated($user));

        $this->events->dispatch(new UserHasRegistered($user));

        return $user;
    }
}
