<?php namespace Visiosoft\SocialModule\Http\Controller;

use GuzzleHttp\Client;
use GuzzleHttp\RequestOptions;
use Illuminate\Support\Arr;
use Visiosoft\SocialModule\Authentication\Contract\AuthenticationRepositoryInterface;
use Visiosoft\SocialModule\Provider\Command\GetProviderExtension;
use Visiosoft\SocialModule\Provider\ProviderAuthentication;
use Visiosoft\SocialModule\Provider\ProviderExtension;
use Visiosoft\SocialModule\Provider\ProviderUsers;
use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Anomaly\Streams\Platform\Model\EloquentModel;
use Anomaly\Streams\Platform\Support\Authorizer;
use Anomaly\UsersModule\User\Contract\UserInterface;
use Anomaly\UsersModule\User\UserAuthenticator;
use Illuminate\Contracts\Auth\Guard;
use Illuminate\Session\Store;
use Laravel\Socialite\Contracts\Provider;
use Laravel\Socialite\Contracts\User;
use Laravel\Socialite\Two\AbstractProvider;

/**
 * Class SocialController
 *
 * @link   http://visiosoft.com.tr/
 * @author Visiosoft, Inc. <support@visiosoft.com.tr>
 * @author Vedat Akdoğan <vedat@visiosoft.com.tr>
 */
class SocialController extends PublicController
{

    /**
     * Return the auth redirect.
     *
     * @param Store $session
     * @param Authorizer $authorizer
     * @param            $provider
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function auth(Store $session, Authorizer $authorizer, $provider)
    {

        /* @var ProviderExtension $extension */
        if (!$extension = $this->dispatch(new GetProviderExtension($provider))) {
            abort(404);
        }

        /**
         * Set the authentication parameter
         * from the request for the callback.
         */
        $session->put(
            'social::allow_registration',
            filter_var($this->request->get('register'), FILTER_VALIDATE_BOOLEAN)
        );

        /**
         * If we are authenticating the application
         * then make sure the user is authorized.
         */
        if (
            $session->get('social::application_authentication', false)
            && !$authorizer->authorize('visiosoft.module.social::providers.authenticate')
        ) {
            abort(403);
        }

        /* @var Provider $provider */
        $provider = $extension->make();

        /**
         * We will need this for later.
         */
        $session->put(
            'url.intended',
            $this->request->input(
                'redirect',
                $session->get('url.intended', '/')
            )
        );

        /**
         * If this is an oAuth2 provider then
         * send the scopes along with the request.
         */
        if ($provider instanceof AbstractProvider) {
            $provider = $provider->scopes(
                $extension->scopes(
                    $session->get('social::application_authentication', false)
                )
            );
        }

        return $provider->redirect();
    }

    public function OneTapAccountGoogle()
    {
        $token = $this->request->token;
        $client = new Client();
        $response = $client->get('https://www.googleapis.com/oauth2/v3/tokeninfo?id_token='.$token, [
            RequestOptions::HEADERS => [
                'Accept' => 'application/json',
            ],
        ]);

        $response_user = json_decode($response->getBody(), true);

        // Deprecated: Fields added to keep backwards compatibility in 4.0. These will be removed in 5.0
        $user['id'] = Arr::get($response_user, 'sub');
        $user['verified_email'] = Arr::get($response_user, 'email_verified');
        $user['link'] = Arr::get($response_user, 'profile');

        $user = (new \Laravel\Socialite\Two\User)->setRaw($user)->map([
            'id' => Arr::get($response_user, 'sub'),
            'nickname' => Arr::get($response_user, 'nickname'),
            'name' => Arr::get($response_user, 'name'),
            'email' => Arr::get($response_user, 'email'),
            'avatar' => $avatarUrl = Arr::get($response_user, 'picture'),
            'avatar_original' => $avatarUrl,
        ]);

        return $user->setToken($token)
            ->setRefreshToken(Arr::get($response_user, 'refresh_token'))
            ->setExpiresIn(Arr::get($response_user, 'expires_in'));


    }

    /**
     * Handle the oauth callback.
     *
     * @param ProviderAuthentication $authentication
     * @param UserAuthenticator $authenticator
     * @param ProviderUsers $users
     * @param Store $session
     * @param                        $provider
     * @return \Illuminate\Http\RedirectResponse
     */
    public function callback(
        ProviderAuthentication $authentication,
        UserAuthenticator $authenticator,
        ProviderUsers $users,
        Store $session,
        $provider
    )
    {
        if ($provider == 'google-one-tap') {
            $account = $this->OneTapAccountGoogle();
            $extension = $this->dispatch(new GetProviderExtension('google'));
        } else {

            /* @var ProviderExtension $extension */
            if (!$extension = $this->dispatch(new GetProviderExtension($provider))) {
                abort(404);
            }

            /* @var Provider $provider */
            $provider = $extension->make();

            /**
             * This is the user from the provider network.
             *
             * @var User $account
             */
            $account = $provider->user();
        }

        /**
         * Pull out a couple parameters
         * from session to determine
         * the next steps.
         */
        $register = $session->pull('social::allow_registration', false);
        $application = $session->pull('social::application_authentication', false);

        // Block registration if desired.
        if (config('visiosoft.module.social::config.allow_registration', false) === false) {
            $register = false;
        } else {
            $register = true;
        }


        /**
         * If we are authenticating the
         * application then we don't need a
         * system user for the authentication.
         *
         * @var UserInterface $user
         */
        if ($account->name == null) {
            $account->name = "User" . rand(9, 999);
        }

        if ($application === true) {

            $authentication->make($account, $extension);

            return $this->redirect->intended('/');
        }

        /**
         * Resolve a local user to authenticate.
         */
        if (!$user = $users->user($account, $register)) {

            $this->messages->error('visiosoft.module.social::message.user_not_found');

            return $this->redirect->back();
        }

        $authentication->make($account, $extension, $user);

        $authenticator->login($user);

        return $this->redirect->intended('/profile');
    }

    /**
     * Disconnect a social account from a user.
     *
     * @param AuthenticationRepositoryInterface $authentications
     * @param Guard $auth
     * @param                                   $provider
     * @return \Illuminate\Http\RedirectResponse
     */
    public function disconnect(
        AuthenticationRepositoryInterface $authentications,
        Guard $auth,
        $provider
    )
    {

        /* @var UserInterface $user */
        if (!$user = $auth->user()) {
            abort(404);
        }

        /* @var ProviderExtension $provider */
        if (!$provider = $this->dispatch(new GetProviderExtension($provider))) {
            abort(404);
        }

        /* @var EloquentModel $authentication */
        if ($authentication = $user->call('authentication', compact('provider'))) {
            $authentications->delete($authentication);
        }

        if ($redirect = $this->request->get('redirect')) {
            return $this->redirect->to($redirect);
        }

        return $this->redirect->back();
    }
}
