import { Vector as SourceVector } from "ol/source";
import { GeoJSON } from "ol/format";
import { Style, Icon, Stroke, Fill, Text, Circle } from "ol/style";
import { Vector as LayerVector } from "ol/layer";

import map from "./list-map"

let expandedVector = []
let vectorSource;
import pin from './images/pin.svg'

const drawOnMap = (
    url,
    fill = [213, 0, 0, .5],
    stroke = "#9E0303",
    strokeWidth = 1,
    hover = '#8E1522',
    removeOldLayer = true
) => {
    if (expandedVector && expandedVector.length && removeOldLayer) {
        for (let i = 0; i < expandedVector.length; i++) {
            map.map.removeLayer(expandedVector[i]);
        }
    }

    vectorSource = new SourceVector({
        // url: "http://10.10.5.49:3040/geoserver/mapilio/ows?service=WFS&version=1.0.0&request=GetFeature&typeName=mapilio%3ABasaksehirMahalle&maxFeatures=50&outputFormat=application%2Fjson&CQL_FILTER=objectid=1",
        // url: "http://10.10.5.49:3040/geoserver/mapilio/ows?service=WFS&version=1.0.0&request=GetFeature&typeName=mapilio%3ABasaksehirKapi&maxFeatures=500&outputFormat=application%2Fjson&CQL_FILTER=yol_id=1055",
        url : url,
        format: new GeoJSON(),
    });

    function styleMe(feature) {
        let style;
        if (feature.getGeometry().getType() == "Point"){
            style = new Style({
                image: new Icon({
                    src: pin
                })
            })
        } else {
            let objStyle = {
                fill: new Stroke({
                    color: fill
                }),
                stroke: new Stroke({
                    color: stroke,
                    width: strokeWidth
                }),
            }

            if (feature.getGeometry().getType() === 'MultiPolygon') {
                objStyle.text = new Text({
                    text: feature.get('adi_numara'),
                    font: '400 12px Poppins, Helvetica, sans-serif',
                    fill: new Fill({
                        color: '#fff',
                    }),
                    overflow: true
                })
            }

            style = new Style(objStyle);
        }
        return style;
    }

    expandedVector.unshift(new LayerVector({
        source: vectorSource,
        style: styleMe,
        title: "expandedSearch"
    }))

    map.map.addLayer(expandedVector[0]);

    if (hover) {
        var highlightStyle = new Style({
            fill: new Stroke({
                color: hover
            }),
            stroke: new Stroke({
                color: hover,
                width: strokeWidth
            }),
            zIndex: 100,
            text: new Text({
                font: '400 12px Poppins, Helvetica, sans-serif',
                overflow: true,
                fill: new Fill({
                    color: '#fff',
                }),
            }),
        })
        var selected = null;
        map.map.on('pointermove', function (e) {
            if (selected !== null) {
                selected.setStyle(undefined);
                selected = null;
            }

            map.map.forEachFeatureAtPixel(e.pixel, function (f) {
                selected = f;
                if (f.getGeometry().getType() !== "Point") {
                    f.setStyle(highlightStyle);
                }

                if (f.getGeometry().getType() === "MultiPolygon") {
                    highlightStyle.getText().setText(f.get('adi_numara'))
                }
                return true;
            });
        });
    }

    /* Redirect on marker click */
    map.map.on('click', function (evt) {
        var feature = map.map.forEachFeatureAtPixel(evt.pixel, function (feature) {
            return feature;
        });
        if (feature && feature.getGeometry().getType() === "Point") {
            window.location.href = '/admin/taxpayer/detail/1'
        }
    });
    /* End Redirect on marker click */

    /* Show feature data on hover */
    var info = $('#info');
    info.tooltip({
        animation: false,
        trigger: 'manual',
    });

    var displayFeatureInfo = function (pixel) {
        info.css({
            left: pixel[0] + 'px',
            top: pixel[1] - 5 + 'px',
        });
        var feature = map.map.forEachFeatureAtPixel(pixel, function (feature) {
            return feature;
        });
        if (feature) {
            if (feature.id_.startsWith('BasaksehirKapi')) {
                info.attr('data-original-title', feature.get('kapi_no')).tooltip('show');
            } else if (feature.id_.startsWith('BasaksehirYol')) {
                info.attr('data-original-title', feature.get('yol_adi')).tooltip('show');
            } else {
                info.tooltip('hide');
            }
        } else {
            info.tooltip('hide');
        }
    };

    map.map.on('pointermove', function (evt) {
        if (evt.dragging) {
            info.tooltip('hide');
            return;
        }
        displayFeatureInfo(map.map.getEventPixel(evt.originalEvent));
    });
    /* End Show feature data on hover */

    global.expandedVector = expandedVector[0];

    var listenerKey = vectorSource.on('addfeature', function(e) {
        if (vectorSource.getState() == 'ready') {
            map.map.getView().fit(vectorSource.getExtent(), {
                size: map.map.getSize(),
                maxZoom: 19,
                padding: [30, 30, 30, 30],
            });
            // ol_observable.unByKey(listenerKey);
        }
    });
}

/* Handle map click */
map.map.on("singleclick", function (evt) {
    this.forEachFeatureAtPixel(evt.pixel, function (feature, layer) {
        const featureID = feature.id_
        let type
        if (featureID.startsWith('BasaksehirMahalle')) {
            type = 'street'
        } else if (featureID.startsWith('BasaksehirYol')) {
            type = map.conn.firstRoad ? 'road' : false
            map.conn.firstRoad = false
        }

        if (type) {
            if (type === 'road') {
                const types = ['road', 'allDoors']

                for (let i = 0; i < 2; i++) {
                    const params = map.conn[types[i]]

                    let urls = params.rel ? params.url + params.mapRel + "=" + feature.values_.objectid : params.url
                    drawOnMap(urls, params.fill, params.stroke, params.width, params.hover, i === 0)
                }
            } else {
                const params = map.conn[type]

                let urls = params.rel ? params.url + params.mapRel + "=" + feature.values_.objectid : params.url
                drawOnMap(urls, params.fill, params.stroke, params.width, params.hover)
            }

            const newURL = addMapParams(type, feature.values_.objectid);
            handleBreadcrumb(type, newURL)
        }
    });
});

function addMapParams(type, value) {
    const urlParams = new URLSearchParams(window.location.search);

    if (type === 'street') {
        urlParams.delete('road');
    }

    urlParams.set(type, value);

    const refresh = `${window.location.protocol}//${window.location.host + window.location.pathname}?${urlParams}`;
    window.history.pushState({ path: refresh }, '', refresh);

    return refresh;
}
/* End Handle map click */

/* Handle breadcrumb */
function handleBreadcrumb(type, url) {
    const breadcrumb = $(`#${type}BC`)

    if (breadcrumb.length) {
        breadcrumb.nextAll().remove()
        breadcrumb.remove()
    }

    if (map.conn[type].breadcrumb) {
        $('#breadcrumbContainer').append(`
            <h6 class="mt-2 mb-2 mr-5 time-navigation" id="${type}BC">
                <a href="${url}">
                    <i class="fa fa-chevron-right fa-sm"></i>
                    ${map.conn[type].breadcrumb}
                </a>
            </h6>
        `)
    }
}

$(document).ready(function() {
    const urlParams = new URLSearchParams(window.location.search);
    let selectedSet = false;
    for (var value of urlParams.keys()) {
        if (['street', 'road'].includes(value)) {
            const url = addMapParams(value, urlParams.get(value));
            handleBreadcrumb(value, url)
            if (!selectedSet || value === 'road') {
                selectSelected(value, urlParams.get(value));
                selectedSet = true
            }
        }
    }
})
/* End Handle breadcrumb */

/* Handle onload */
function selectSelected (type, value) {
    if (type === 'road') {
        const types = ['road', 'allDoors']

        for (let i = 0; i < 2; i++) {
            const params = map.conn[types[i]]

            let urls = params.rel ? params.url + params.mapRel + "=" + value : params.url
            drawOnMap(urls, params.fill, params.stroke, params.width, params.hover, i === 0)
        }

        map.conn.firstRoad = false
    } else {
        const params = map.conn[type]

        let urls = params.rel ? params.url + params.mapRel + "=" + value : params.url
        drawOnMap(urls, params.fill, params.stroke, params.width, params.hover)
    }
}
/* End Handle onload */

export { drawOnMap, handleBreadcrumb, addMapParams }
