<?php namespace Visiosoft\SmartaxModule\Http\Controller\Admin;

use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Visiosoft\SmartaxModule\Bbox\Contract\BboxRepositoryInterface;
use Visiosoft\SmartaxModule\Commands\getGeoDistricts;
use Visiosoft\SmartaxModule\Commands\getGeoDoors;
use Visiosoft\SmartaxModule\Commands\getGeoRoads;
use Visiosoft\SmartaxModule\Feature\Contract\FeatureRepositoryInterface;
use Visiosoft\SmartaxModule\Location\Contract\LocationRepositoryInterface;
use Visiosoft\SmartaxModule\Measurement\Contract\MeasurementRepositoryInterface;
use Visiosoft\SmartaxModule\Point\Contract\PointRepositoryInterface;
use Visiosoft\SmartaxModule\Relation\Contract\RelationRepositoryInterface;
use Visiosoft\SmartaxModule\Segmentation\Contract\SegmentationRepositoryInterface;
use Yajra\DataTables\DataTables;

class ApiController extends AdminController
{
    public function geoFeatures($type)
    {
        switch ($type) {
            case "mahalle":
                $geo = $this->dispatch(new getGeoDistricts($this->request->all()));
                break;
            case "yol":
                $geo = $this->dispatch(new getGeoRoads($this->request->all()));
                break;
            case "kapi":
                $geo = $this->dispatch(new getGeoDoors($this->request->all()));
                break;
        }

        $geo = $geo ?: "{}";

        return $this->response->json(json_decode($geo, true));
    }

    public function features(FeatureRepositoryInterface $featureRepository)
    {
        $area_price = setting_value('visiosoft.module.smartax::tax-price',70);

        $select_columns = [
            'smartax_feature.*',
            DB::raw("round( CAST((default_smartax_feature.area*$area_price) as numeric), 2) as price"),
            'smartax_feature.detected_district_id as district',
            'smartax_feature.detected_road_id as road',
            'smartax_feature.detected_door_id as door',
            'smartax_point.uploaded_hash',
            'smartax_point.resolution',
            'smartax_point.filename',
        ];

        if (Schema::hasTable('basaksehir_yol')) {
            $select_columns[] = 'basaksehir_yol.yol_adi AS road_name';
        }

        if (Schema::hasTable('basaksehir_mahalle')) {
            $select_columns[] = 'basaksehir_mahalle.adi_numara AS district_name';
        }

        if (Schema::hasTable('basaksehir_kapi')) {
            $select_columns[] = 'basaksehir_kapi.kapi_no AS door_name';
        }

        $query = $featureRepository->newQuery()
            ->select($select_columns)
            ->where('class_code', 'inst-sign-store')
            ->where('area', '<', number_format(setting_value('visiosoft.module.smartax::max-area', '20.0'), '1', '.', ''))
            ->where('area', '>', number_format(setting_value('visiosoft.module.smartax::min-area', '3.0'), '1', '.', ''))
            ->whereNotNull('thumbnail_photo_uuid')
            ->whereNull('smartax_feature.deleted_at')
            ->leftJoin('smartax_point', 'smartax_point.photo_uuid', 'smartax_feature.thumbnail_photo_uuid');

        if (Schema::hasTable('basaksehir_yol')) {
            $query = $query->leftJoin('basaksehir_yol', function ($join) {
                $join->on('basaksehir_yol.objectid', 'smartax_feature.detected_road_id');
                $join->on('basaksehir_yol.mahalle_id', 'smartax_feature.detected_district_id');
            });
        }

        if (Schema::hasTable('basaksehir_mahalle')) {
            $query = $query->leftJoin('basaksehir_mahalle', 'basaksehir_mahalle.objectid', 'smartax_feature.detected_district_id');
        }

        if (Schema::hasTable('basaksehir_kapi')) {
            $query = $query->leftJoin('basaksehir_kapi', 'basaksehir_kapi.objectid', 'smartax_feature.detected_door_id');
        }

        $query = $query->where('smartax_point.filename','!=',NULL)->get();


        return DataTables::of($query)->toJson();
    }

    public function pointDetail($id)
    {
        $featureRepository = app(FeatureRepositoryInterface::class);
        $bbox_repository = app(BboxRepositoryInterface::class);
        $relation_repository = app(RelationRepositoryInterface::class);
        $point_repository = app(PointRepositoryInterface::class);

        $feature = $featureRepository->newQuery()->find($id);

        $relations = $relation_repository->newQuery()
            ->where('feature_id', $feature->id)
            ->get();

        $bbox_id_entries = array();

        foreach ($relations as $relation) {
            $bbox_id_entries[] = $relation->bbox_id_1;
            $bbox_id_entries[] = $relation->bbox_id_2;
        }

        $bbox_id_entries = array_unique($bbox_id_entries);

        $bbox_entries = $bbox_repository->newQuery()->whereIn('id', $bbox_id_entries)->get();

        $bbox_entries = $bbox_entries->sortBy('score');

        foreach ($bbox_entries as $bbox_entry) {
            $bbox_entry->detail = $point_repository->newQuery()->where('photo_uuid', $bbox_entry->photo_uuid)->first();
        }

        /**
         * 30.11.2022
         * Hides oversized BBOX data
         */

        $bbox_entries = $bbox_entries->filter(function($entry) {
            return ($entry->y_max - $entry->y_min) < 500;
        });

        $feature->bbox_entries = $bbox_entries->values();

        return response()->json($feature);
    }

    public function editPoint(FeatureRepositoryInterface $featureRepository,
                              BboxRepositoryInterface $bboxRepository,
                              RelationRepositoryInterface $relationRepository,
                              LocationRepositoryInterface $locationRepository,
                              MeasurementRepositoryInterface $measurementRepository,
                              $id)
    {
        $status = false;
        $required_fields = ['lat', 'lon', 'area', 'calculate_at', 'height', 'width'];

        try {
            if (request()->has($required_fields) and $feature = $featureRepository->find($id)) {

                $unique_key = Str::random(17);

                //Save Calculation
                $feature->update([
                    'lat' => $this->request->lat,
                    'lon' => $this->request->lon,
                    'area' => $this->request->area,
                    'confidence' => 1.0,
                ]);

                // Save Geom for Calculation
                DB::statement("UPDATE default_smartax_feature
	                                        SET  geom=ST_GeomFromGeoJSON('" . json_encode($this->request->geometry) . "')
	                                        WHERE id=" . $feature->id . ";");

                $measurementRepository->newQuery()
                    ->where('feature_id', $id)
                    ->update([
                        'avg_width' => $this->request->width,
                        'avg_height' => $this->request->height,
                        'avg_area' => $this->request->area,
                    ]);

                // DELETE OLD ENTRIES
                $this->deleteOldEntries($id);

                foreach ($this->request->matchedPoints as $item) {
                    $sequence_uuid = null;

                    $bbox_1_unique_key = $unique_key . $item['obj_1'];
                    $bbox_2_unique_key = $unique_key . $item['obj_2'];

                    // Save BBOX 1
                    if (!$bbox_1 = $bboxRepository->newQuery()->where('bbox_key', $bbox_1_unique_key)->first()) {

                        $old_bbox_1 = $bboxRepository->findWithTrashed($item['obj_1']);

                        $sequence_uuid = $old_bbox_1->sequence_uuid;

                        $bbox_1 = $bboxRepository->newQuery()
                            ->create([
                                'score' => 1.0,
                                'photo_uuid' => $old_bbox_1->photo_uuid,
                                'bbox_key' => $bbox_1_unique_key,
                                'sequence_uuid' => $old_bbox_1->sequence_uuid,
                                'x_min' => $item['bbox_1'][0],
                                'y_min' => $item['bbox_1'][1],
                                'x_max' => $item['bbox_1'][2],
                                'y_max' => $item['bbox_1'][3],
                            ]);
                    }

                    // Save BBOX 2
                    if (!$bbox_2 = $bboxRepository->newQuery()->where('bbox_key', $bbox_2_unique_key)->first()) {

                        $old_bbox_2 = $bboxRepository->findWithTrashed($item['obj_2']);

                        $bbox_2 = $bboxRepository->newQuery()
                            ->create([
                                'score' => 1.0,
                                'photo_uuid' => $old_bbox_2->photo_uuid,
                                'bbox_key' => $bbox_2_unique_key,
                                'sequence_uuid' => $old_bbox_2->sequence_uuid,
                                'x_min' => $item['bbox_2'][0],
                                'y_min' => $item['bbox_2'][1],
                                'x_max' => $item['bbox_2'][2],
                                'y_max' => $item['bbox_2'][3],
                            ]);
                    }

                    //Save RELATION

                    if (!$relation = $relationRepository->newQuery()
                        ->where('bbox_id_1', $bbox_1->id)
                        ->where('bbox_id_2', $bbox_2->id)
                        ->where('feature_id', $id)
                        ->first()) {
                        $relation = $relationRepository->newQuery()
                            ->create([
                                'bbox_id_1' => $bbox_1->id,
                                'bbox_id_2' => $bbox_2->id,
                                'feature_id' => $id,
                                'sequence_uuid' => $sequence_uuid
                            ]);
                    }

                    //Save LOCATION
                    if (!$location = $locationRepository->newQuery()
                        ->where('relation_id', $relation->id)
                        ->first()) {

                        $location = $locationRepository->newQuery()
                            ->create([
                                'sequence_uuid' => $sequence_uuid,
                                'relation_id' => $relation->id,
                                'score' => 1.0,
                                'lat' => $this->request->lat,
                                'lon' => $this->request->lon,
                            ]);

                        // Save Geom for Location
                        DB::statement("UPDATE default_smartax_location
	                                        SET  geom=ST_GeomFromGeoJSON('" . json_encode($this->request->geometry) . "')
	                                        WHERE id=" . $location->id . ";");
                    }
                }

                $status = true;
            }
        } catch (\Exception $exception) {
        }

        return response()->json(['status' => $status]);
    }

    public function deleteOldEntries($feature_id)
    {
        $bboxRepository = app(BboxRepositoryInterface::class);
        $relationRepository = app(RelationRepositoryInterface::class);
        $locationRepository = app(LocationRepositoryInterface::class);

        //Get OLD Relations
        $old_relations = $relationRepository->newQuery()
            ->where('feature_id', $feature_id)
            ->get();

        //Collect OLD BBOX ID's
        $old_bbox_ids = array();

        foreach ($old_relations as $old_relation) {
            $old_bbox_ids[] = $old_relation->bbox_id_1;
            $old_bbox_ids[] = $old_relation->bbox_id_2;
        }

        /*  Remove duplicate id's  */
        $old_bbox_ids = array_unique($old_bbox_ids);


        //Delete OLD BBOX Entries
        $bboxRepository->newQuery()
            ->whereIn('id', $old_bbox_ids)
            ->delete();

        //Delete OLD LOCATION Entries
        $locationRepository->newQuery()
            ->whereIn('relation_id', $old_relations->pluck('id')->all())
            ->delete();

        //Delete OLD RELATION Entries
        $relationRepository->newQuery()
            ->where('feature_id', $feature_id)
            ->delete();
    }

    public function getDistrictReport()
    {
        $query = "SELECT CASE WHEN 
                    DISTRICT.ADI_NUMARA IS NULL 
                    THEN 'Diğer' ELSE DISTRICT.ADI_NUMARA || ' ' || 'Mahallesi'
                    END AS NAME,
                    round( CAST(COUNT(*) * 100.0 / SUM(COUNT(*)) OVER() as numeric), 2) as percentage
                    FROM DEFAULT_SMARTAX_FEATURE AS FEATURE
                    LEFT JOIN DEFAULT_BASAKSEHIR_MAHALLE AS DISTRICT ON FEATURE.DETECTED_DISTRICT_ID = DISTRICT.OBJECTID
                    where 
                    class_code = 'inst-sign-store' and 
                    area < ".number_format(setting_value('visiosoft.module.smartax::max-area', '20.0'), '1', '.', '')." and
                    area > ".number_format(setting_value('visiosoft.module.smartax::min-area', '3.0'), '1', '.', '')." and 
                    thumbnail_photo_uuid is not null and
                    FEATURE.DELETED_AT is null 
                    GROUP BY DETECTED_DISTRICT_ID,DISTRICT.ADI_NUMARA ORDER BY DISTRICT.ADI_NUMARA;";

        $entries = DB::select($query);

        $chart_data = array();
        foreach (collect($entries) as $entry) {
            $chart_data[] = [
                $entry->name,
                floatval($entry->percentage)
            ];
        }

        return $this->response->json($chart_data);
    }

    public function getDistricts()
    {
        try {
            $districts = DB::table('basaksehir_mahalle')
                ->select(DB::raw('default_basaksehir_mahalle.*'),
                    DB::raw('default_basaksehir_mahalle.adi_numara as name'),
                    DB::raw('default_basaksehir_mahalle.objectid as id')
                );

            if ($this->request->has('id')) {
                $districts = $districts->where('objectid', $this->request->id)
                    ->first();
            } else {
                $districts = $districts->get();
            }

            return $this->response->json(['status' => true, 'data' => $districts]);
        } catch (\Exception $exception) {
            return $this->response->json(['status' => false, 'message' => $exception->getMessage()]);
        }
    }

    public function getStreets()
    {
        try {
            $streets = DB::table('basaksehir_yol')
                ->select(DB::raw('default_basaksehir_yol.*'),
                    DB::raw('default_basaksehir_yol.yol_adi as name'),
                    DB::raw('default_basaksehir_yol.objectid as id'));

            if ($this->request->has('parent_id')) {
                $streets = $streets->where('mahalle_id', $this->request->parent_id)
                    ->get();
            } elseif ($this->request->has('id')) {
                $streets = $streets->where('objectid', $this->request->id)
                    ->first();
            } else {
                $streets = $streets->get();
            }

            return $this->response->json(['status' => true, 'data' => $streets]);
        } catch (\Exception $exception) {
            return $this->response->json(['status' => false, 'message' => $exception->getMessage()]);
        }
    }

    public function getDoors()
    {
        try {
            $doors = DB::table('basaksehir_kapi')
                ->select(DB::raw('default_basaksehir_kapi.*'),
                    DB::raw('default_basaksehir_kapi.kapi_no as name'),
                    DB::raw('default_basaksehir_kapi.objectid as id'));

            if ($this->request->has('parent_id')) {
                $doors = $doors->where('yol_id', $this->request->parent_id)
                    ->get();
            } elseif ($this->request->has('id')) {
                $doors = $doors->where('objectid', $this->request->id)
                    ->first();
            } else {
                $doors = $doors->get();
            }

            return $this->response->json(['status' => true, 'data' => $doors]);
        } catch (\Exception $exception) {
            return $this->response->json(['status' => false, 'message' => $exception->getMessage()]);
        }
    }

    public function chartYear()
    {
        try {
            $area_price = setting_value('visiosoft.module.smartax::tax-price');

            $entries = DB::select("SELECT TO_CHAR(DATE_TRUNC('year',CREATED_AT),'YYYY') AS YEAR,
                                    ROUND(CAST((SUM(AREA) * $area_price) AS numeric),2) AS TOTAL_PRICE,
                                    ROUND(CAST(SUM(AREA) AS numeric),2) AS TOTAL_AREA
                                FROM DEFAULT_SMARTAX_FEATURE
                                where 
                                class_code = 'inst-sign-store' and 
                                area < ".number_format(setting_value('visiosoft.module.smartax::max-area', '20.0'), '1', '.', '')." and
                                area > ".number_format(setting_value('visiosoft.module.smartax::min-area', '3.0'), '1', '.', '')." and 
                                thumbnail_photo_uuid is not null and
                                DEFAULT_SMARTAX_FEATURE.DELETED_AT is null 
                                GROUP BY YEAR");

            return $this->response->json(['status' => true, 'data' => $entries]);
        } catch (\Exception $exception) {
            return $this->response->json(['status' => false, 'message' => $exception->getMessage()]);
        }
    }

    public function editProperties(FeatureRepositoryInterface $repository, $id)
    {
        try {
            $feature = $repository->find($id);

            if (!$feature) {
                throw new \Exception('Feature not found!');
            }

            $validator = Validator::make(request()->all(), [
                'district' => 'required',
                'street' => 'required',
                'door' => 'required',
                'area' => 'required',
                'coordinate' => 'required',
            ]);

            $properties = $feature->properties ? json_decode($feature->properties, true) : [];
            foreach (request()->all() as $array => $value) {
                if (strpos($array, 'properties_') === 0) {
                    $properties[substr($array, 11)] = $value;
                }
            }

            if ($validator->fails()) {
                throw new \Exception($validator->errors()->first());
            }

            $coordinate = explode(',', $this->request->coordinate);

            $feature->update([
                'selected_district_id' => $this->request->district,
                'selected_road_id' => $this->request->street,
                'selected_door_id' => $this->request->door,
                'area' => $this->request->area,
                'lat' => $coordinate[1],
                'lon' => $coordinate[0],
                'properties' => json_encode($properties)
            ]);

            return $this->response->json(['status' => true]);
        } catch (\Exception $exception) {
            return $this->response->json(['status' => false, 'message' => $exception->getMessage()], 400);
        }
    }

    public function getLabelFeatures(
        PointRepositoryInterface $repository,
        FeatureRepositoryInterface $featureRepository,
        BboxRepositoryInterface $bboxRepository,
        RelationRepositoryInterface $relationRepository)
    {
        $features = $featureRepository->newQuery()->orderBy('id', 'ASC')->get();

        $features->filter(function ($feature) use ($relationRepository, $bboxRepository, $repository) {
            $relations = $relationRepository->findAllBy('feature_id', $feature->id);

            $images = [];

            foreach ($relations as $relation) {
                $bbox_entries = $bboxRepository->newQuery()
                    ->where('id', $relation->bbox_id_1)
                    ->orWhere('id', $relation->bbox_id_2)
                    ->get();


                foreach ($bbox_entries as $bbox_entry) {
                    $point = $repository->findByKey($bbox_entry->photo_uuid);

                    $images[$point->id] = [
                        'id' => $feature->id,
                        'sequence_uuid' => $point->sequence_uuid,
                        'width' => $bbox_entry->x_max - $bbox_entry->x_min,
                        'height' => $bbox_entry->y_max - $bbox_entry->y_min,
                        'points' => [
                            intval($bbox_entry->x_min),
                            intval($bbox_entry->y_min),
                            $bbox_entry->x_max - $bbox_entry->x_min,
                            $bbox_entry->y_max - $bbox_entry->y_min,
                        ],
                        'bbox' => [
                            intval($bbox_entry->x_min),
                            intval($bbox_entry->y_min),
                            intval($bbox_entry->x_max),
                            intval($bbox_entry->y_max),
                        ],
                    ];

                }
            }
            $feature->images = $images;
        });

        return $this->response->json(['status' => true, 'data' => $features]);
    }

    public function getLabelImages(PointRepositoryInterface $repository)
    {
        $images = $repository->newQuery()
            ->select(['smartax_point.*', DB::raw('default_smartax_point.resolution as org_resolution')])
            ->orderBy('capture_time', 'ASC')
            ->get();

        return $this->response->json(['status' => true, 'data' => $images]);
    }

    public function saveLabelFeatures(
        MeasurementRepositoryInterface $measurementRepository,
        RelationRepositoryInterface $relationRepository,
        SegmentationRepositoryInterface $segmentationRepository,
        BboxRepositoryInterface $bboxRepository,
        FeatureRepositoryInterface $featureRepository)
    {
        try {
            $features = $this->request->features;

            foreach ($features as $feature) {

                $mixing_images = [];

                foreach (array_keys($feature['images']) as $image) {
                    $mixing_images = $this->mixingArray($image, array_keys($feature['images']), $mixing_images);
                }

                $sequence_uuid = array_first($feature['images'])['sequence_uuid'];
                $thumbnail = array_first($feature['images']);
                $thumbnail_bbox = [
                    intval($thumbnail['bbox'][0]),
                    intval($thumbnail['bbox'][1]),
                    intval($thumbnail['bbox'][2]),
                    intval($thumbnail['bbox'][3]),
                ];

                if (!isset($feature['id'])) {
                    // Entry Save (Feature)
                    $feature_entry = $featureRepository->newQuery()->create([
                        'created_by_id' => Auth::id(),
                        'lat' => $feature['calculation']['geometry']['coordinates'][1],
                        'lon' => $feature['calculation']['geometry']['coordinates'][0],
                        'class_code' => $feature['class_code'],
                        'confidence' => 1.0,
                        'sequence_uuid' => $sequence_uuid,
                        'area' => $feature['calculation']['properties']['area'],
                        'verification' => true,
                        'selected_district_id' => $feature["district"],
                        'selected_road_id' => $feature["street"],
                        'selected_door_id' => $feature["door"],
                        'properties' => json_encode(['name' => $feature['name']]),
                        'thumbnail_photo_uuid' => $thumbnail['photo_uuid'],
                        'thumbnail_photo_bbox' => json_encode($thumbnail_bbox),
                    ]);

                    // Geom Save (Feature)
                    DB::statement("UPDATE default_smartax_feature
	                                        SET  geom=ST_GeomFromGeoJSON('" . json_encode($feature['calculation']['geometry']) . "')
	                                        WHERE id=" . $feature_entry->id . ";");

                    // Entry Save (Measurement)
                    $measurementRepository->newQuery()
                        ->create([
                            'created_by_id' => Auth::id(),
                            'sequence_uuid' => $sequence_uuid,
                            'feature_id' => $feature_entry->id,
                            'avg_width' => $feature['calculation']['properties']['width'],
                            'avg_height' => $feature['calculation']['properties']['height'],
                            'avg_area' => $feature['calculation']['properties']['area'],
                        ]);

                    $mixing_images = [];
                    foreach (array_keys($feature['images']) as $image) {
                        $mixing_images = $this->mixingArray($image, array_keys($feature['images']), $mixing_images);
                    }

                    foreach ($mixing_images as $image_id => $relation_images) {
                        foreach ($relation_images as $relation_image) {

                            $check_relation = $relationRepository->newQuery()
                                ->whereIn('bbox_id_1', [$image_id, $relation_image])
                                ->whereIn('bbox_id_2', [$image_id, $relation_image])
                                ->first();

                            if (!$check_relation) {
                                $bbox_key_1 = $feature_entry->id . $image_id;
                                $bbox_key_2 = $feature_entry->id . $relation_image;

                                // Entry Save (BBOX)
                                if (!$bbox_1 = $bboxRepository->findByKey($bbox_key_1)) {
                                    $bbox_1_detail = $feature['images'][$image_id];
                                    $bbox_1 = $bboxRepository->newQuery()
                                        ->create([
                                            'created_by_id' => Auth::id(),
                                            'x_min' => $bbox_1_detail['bbox'][0],
                                            'y_min' => $bbox_1_detail['bbox'][1],
                                            'sequence_uuid' => $sequence_uuid,
                                            'x_max' => $bbox_1_detail['bbox'][2],
                                            'y_max' => $bbox_1_detail['bbox'][3],
                                            'score' => 1.0,
                                            'bbox_key' => $bbox_key_1,
                                            'photo_uuid' => $bbox_1_detail['photo_uuid'],
                                        ]);
                                }

                                if (!$bbox_2 = $bboxRepository->findByKey($bbox_key_2)) {
                                    $bbox_2_detail = $feature['images'][$relation_image];
                                    $bbox_2 = $bboxRepository->newQuery()
                                        ->create([
                                            'created_by_id' => Auth::id(),
                                            'x_min' => $bbox_2_detail['bbox'][0],
                                            'y_min' => $bbox_2_detail['bbox'][1],
                                            'sequence_uuid' => $sequence_uuid,
                                            'x_max' => $bbox_2_detail['bbox'][2],
                                            'y_max' => $bbox_2_detail['bbox'][3],
                                            'score' => 1.0,
                                            'bbox_key' => $bbox_key_2,
                                            'photo_uuid' => $bbox_2_detail['photo_uuid'],
                                        ]);
                                }

                                // Entry Save (Relation)
                                $relation = $relationRepository->newQuery()
                                    ->create([
                                        'created_by_id' => Auth::id(),
                                        'bbox_id_1' => $bbox_1->id,
                                        'bbox_id_2' => $bbox_2->id,
                                        'sequence_uuid' => $sequence_uuid,
                                        'feature_id' => $feature_entry->id,
                                    ]);

//                            // Entry Save (Location)
//                            $location = $locationRepository->newQuery()
//                                ->create([
//                                    'created_by_id' => $created_by_id,
//                                    'lat' => $bbox['geometry']['coordinates'][1],
//                                    'lon' => $bbox['geometry']['coordinates'][0],
//                                    'sequence_uuid' => $sequence_uuid,
//                                    'score' => ($bbox['properties']['score_1'] + $bbox['properties']['score_2']) / 2,
//                                    'relation_id' => $relation->id,
//                                ]);
//
//                            // Geom Save (Location)
//                            DB::statement("UPDATE default_mapilio_location
//	                                        SET  geom=ST_GeomFromGeoJSON('" . json_encode($bbox['geometry']) . "')
//	                                        WHERE id=" . $location->id . ";");

                                // Entry Save (Segmentation)
                                $segmentationRepository->newQuery()
                                    ->create([
                                        'created_by_id' => Auth::id(),
                                        'bbox_id' => $bbox_1->id,
                                        'sequence_uuid' => $sequence_uuid,
                                        'segmentation' => null,
                                    ]);

                                $segmentationRepository->newQuery()
                                    ->create([
                                        'created_by_id' => Auth::id(),
                                        'bbox_id' => $bbox_2->id,
                                        'sequence_uuid' => $sequence_uuid,
                                        'segmentation' => null,
                                    ]);
                            }
                        }
                    }
                }
            }

            return $this->response->json(['success' => true]);
        } catch (\Exception $exception) {
            return $this->response->json(['success' => false, 'message' => $exception->getMessage()]);
        }

    }

    function mixingArray($value, $values, $response)
    {
        $response[$value] = [];
        foreach ($values as $item) {
            if ($item != $value) {
                $response[$value][] = $item;
            }
        }
        return $response;
    }
}
