class SmartaxApiClient {
    constructor() {
        this.baseUrl = '/api/smartax/features';
        this.page = 1;
        this.minFilter = 0;
        this.maxFilter = 25;
        this.limit = 10;
    }

    setPage(page) {
        this.page = page;
    }

    setMinFilter(number) {
        this.minFilter = number;
    }

    setMaxFilter(number) {
        this.maxFilter = number;
    }

    setLimit(limit) {
        this.limit = limit;
    }

    async getFeatures(suffix = '') {
        try {
            const response = await fetch(`${this.baseUrl}?page=${this.page}&min-filter=${this.minFilter}&max-filter=${this.maxFilter}&limit=${this.limit}` + suffix);
            const data = await response.json();
            return data;
        } catch (error) {
            console.error('Error:', error);
            return null;
        }
    }
}

class Table {
    constructor(table, data, entries, pagination) {
        this.enties = entries;
        this.data = data;
        this.table = table;
        this.pagination = pagination;
        this.rowLinkPrefix = '';
        this.schema = {
            'id': ':value',
            'thumbnail': '<img src=":value" height="75" width="150" style="object-fit: cover">',
            'area': '<b>:value m²</b>',
            'neighborhood': ':value',
            'road': ':value',
            'number': ':value',
            'price': '<b>:value</b>',
            'confidence': ':value',
            'created_at': ':value',
        }
    }

    setRowLinkPrefix(value) {
        this.rowLinkPrefix = value;
    }

    showLoader() {
        return Swal.fire({
            imageUrl: loading_url,
            allowOutsideClick: false,
            showConfirmButton: false,
            imageWidth: 200,
            imageHeight: 200,
        });
    }

    setPaginationClickFunction(fn) {
        this.paginationClickFunction = fn;
    }

    setTable(element) {
        this.table = element;
    }

    setPagination(element) {
        this.pagination = element;
    }

    setData(data) {
        this.data = data;
        this.setEntries(data.entries)
    }

    setEntries(entries) {
        this.enties = entries;
    }

    reset() {
        this.table.getElementsByTagName('tbody')[0].innerHTML = "";
        this.pagination.innerHTML = "";
    }

    renderInformation() {
        let text = '';
        let total = this.data.total;
        let result = this.data.result;
        let limit = this.data.limit;
        let page = this.data.page;

        text = `<b>${total}</b> kayıttan <b>${(parseInt(page) - 1) * parseInt(result)}</b> ile <b>${(parseInt(page)) * parseInt(limit)}</b> arası listeleniyor`;

        document.getElementById('table-record-info').innerHTML = text;
    }

    renderPagination() {
        let currentPage = parseInt(this.data.page);
        let pageCount = Math.round(parseInt(this.data.total) / parseInt(this.data.limit));
        let startPagination = currentPage > 3 ? currentPage - 2 : 1;
        let endPagination = currentPage + 2 < pageCount ? currentPage + 2 : pageCount;

        for (let i = startPagination; i <= endPagination; i++) {
            let row = document.createElement('li');
            row.className = 'paginate_button page-item';

            if (currentPage === i) {
                row.className = row.className + " active";
            }

            let rowLink = document.createElement('a');
            rowLink.className = 'page-link cursor-pointer';
            rowLink.innerHTML = i;

            rowLink.onclick = function (page) {
                return function () {
                    table.paginationClickFunction(i);
                };
            }(i);

            row.appendChild(rowLink);
            this.pagination.appendChild(row);
        }

        if (currentPage > 3) {
            let row = document.createElement('li');
            row.className = 'paginate_button page-item';
            let rowLink = document.createElement('a');
            rowLink.className = 'page-link cursor-pointer';
            rowLink.innerHTML = '<<';
            rowLink.onclick = function (page) {
                return function () {
                    table.paginationClickFunction(1);
                };
            }(1);
            row.appendChild(rowLink);
            this.pagination.prepend(row);
        }

        if (pageCount > endPagination) {
            let row = document.createElement('li');
            row.className = 'paginate_button page-item';
            let rowLink = document.createElement('a');
            rowLink.className = 'page-link cursor-pointer';
            rowLink.innerHTML = '>>';
            rowLink.onclick = function (page) {
                return function () {
                    table.paginationClickFunction(pageCount);
                };
            }(pageCount);
            row.appendChild(rowLink);
            this.pagination.appendChild(row);
        }
    }

    render() {
        this.reset();
        this.renderInformation();
        this.enties.forEach(item => {
            let row = document.createElement('tr');
            let element;

            item.area = parseFloat(item.area).toFixed(2);
            item.price = parseFloat(parseFloat(item.area).toFixed(2) * parseFloat(this.data.tax_price).toFixed(2)).toFixed(2) + '₺';
            item.created_at = moment(item.created_at);
            item.created_at = item.created_at.format('DD/MM/YYYY HH:mm')

            for (const schema_index in this.schema) {
                element = document.createElement('td');
                element.innerHTML = this.schema[schema_index].replace(':value', item[schema_index] ?? 'eşleşmedi');
                row.appendChild(element);
            }
            this.table.getElementsByTagName('tbody')[0].appendChild(row);
        })
    }
}

const smartaxApi = new SmartaxApiClient();
const table = new Table();
const urlParams = new URLSearchParams(window.location.search);
var neighborhood = urlParams.get('neighborhood');
var road = urlParams.get('road');
var minFilter = urlParams.get('min-filter');
var maxFilter = urlParams.get('max-filter');
var url;
let suffix;
let loader;
const initPage = urlParams.get('page');

function renderSuffix() {
    suffix = neighborhood ? '&neighborhood=' + neighborhood : '';
    suffix += road ? '&road=' + road : '';
}

const paginationClickFunction = function (page) {
    loader = table.showLoader();
    renderSuffix();
    smartaxApi.setPage(page);
    smartaxApi.getFeatures(suffix)
        .then(data => {
            table.setData(data);
            table.render();
            table.renderPagination();
            loader.close();
        })
        .catch(error => {
            console.error('Error:', error);
        });

    // Set URL parameter
    urlParams.set('page', page);
    url = `${window.location.protocol}//${window.location.host + window.location.pathname}?${urlParams}`;
    window.history.pushState({path: url}, '', url);

};

function getRoads(neighborhood_id) {
    return $.ajax({
        url: '/api/smartax/get-roads',
        type: 'GET',
        data: {'parent_id': neighborhood_id},
        error: function () {
            errorS();
        }
    });
}

function getNeighborhood() {
    return $.ajax({
        url: '/api/smartax/get-neighborhoods',
        type: 'GET',
        error: function () {
            errorS();
        }
    });
}

new Promise(function (callback) {
    table.setTable(document.getElementById('featureList'));
    table.setPagination(document.getElementById('table-pagination'));
    table.setRowLinkPrefix(REQUEST_ROOT_PATH + '/admin/smartax/detail/')
    table.setPaginationClickFunction(paginationClickFunction);
    renderSuffix();

    smartaxApi.setPage(initPage ?? 1)
    smartaxApi.setMinFilter(minFilter ?? default_min_filter)
    smartaxApi.setMaxFilter(maxFilter ?? default_max_filter)
    loader = table.showLoader();

    smartaxApi.getFeatures(suffix)
        .then(data => {
            table.setData(data);
            table.render();
            table.renderPagination()
            renderDownloadURL();

            getNeighborhood(neighborhood).then((r) => {
                $('.filter-dd-wrapper[data-name="neighborhood"]').find('.dropdown-menu').html(`<a class="dropdown-item py-2 cursor-pointer active" data-val="">
                                    <span class="text-truncate">Tümü</span>
                                </a>`);
                $.each(r.data, function (index, value) {
                    $('.filter-dd-wrapper[data-name="neighborhood"]').find('.dropdown-menu').append(`<a class="dropdown-item py-2 cursor-pointer" data-val="${value.id}">
                                    <span class="text-truncate">${value.name}</span>
                                </a>`);
                });

                callback(true);
                loader.close();
            });
        })
        .catch(error => {
            console.error('Error:', error);
        });
}).then((r) => {
    if (neighborhood) {
        new Promise(function (callback) {
            let element = $('.filter-dd-wrapper[data-name="neighborhood"]').find('a[data-val="' + neighborhood + '"]');
            element
                .addClass('active')
                .siblings()
                .removeClass('active')
                .closest('.dropdown')
                .find('.dropdown-toggle')
                .addClass('active')
                .find('span')
                .text(element.text());

            callback(true);
        }).then(() => {
            new Promise(function (callback) {
                getRoads(neighborhood).then((r) => {
                    $.each(r.data, function (index, value) {
                        $('.filter-dd-wrapper[data-name="road"]').find('.dropdown-menu').append(`<a class="dropdown-item py-2 cursor-pointer" data-val="${value.id}">
                                    <span class="text-truncate">${value.name}</span>
                                </a>`);

                    })

                    callback(true);
                })
            }).then((r) => {
                let element = $('.filter-dd-wrapper[data-name="road"]').find('a[data-val="' + road + '"]');
                element
                    .addClass('active')
                    .siblings()
                    .removeClass('active')
                    .closest('.dropdown')
                    .find('.dropdown-toggle')
                    .addClass('active')
                    .find('span')
                    .text(element.text());
            })
        });
    }
});

$('.filter-dd-wrapper[data-name="neighborhood"]').on('click', '.dropdown-menu .dropdown-item', function () {
    loader = table.showLoader();
    $(this).addClass('active')
        .siblings()
        .removeClass('active')
        .closest('.dropdown')
        .find('.dropdown-toggle')
        .addClass('active')
        .find('span')
        .text($(this).text())

    neighborhood = $(this).data('val') == "" ? null : $(this).data('val');
    road = null;
    renderSuffix();

    urlParams.delete('page');
    urlParams.delete('road');
    if (neighborhood == null) {
        urlParams.delete('neighborhood');
    } else {
        urlParams.set('neighborhood', neighborhood);
    }
    url = `${window.location.protocol}//${window.location.host + window.location.pathname}?${urlParams}`;
    window.history.pushState({path: url}, '', url);

    reRenderTable();

    if (neighborhood != null) {
        getRoads(neighborhood).then((r) => {
            $('.filter-dd-wrapper[data-name="road"]').find('.dropdown-menu').html('');
            $.each(r.data, function (index, value) {
                $('.filter-dd-wrapper[data-name="road"]').find('.dropdown-menu').append(`<a class="dropdown-item py-2 cursor-pointer" data-val="${value.id}">
                                    <span class="text-truncate">${value.name}</span>
                                </a>`);
            });
        });
    } else {
        $('.filter-dd-wrapper[data-name="road"]')
            .find('.dropdown-menu')
            .html('')
            .closest('.dropdown')
            .find('.dropdown-toggle')
            .removeClass('active')
            .find('span')
            .text("Cadde/Sokak/Yol ara")
    }
});

$('.filter-dd-wrapper[data-name="road"]').on('click', '.dropdown-menu .dropdown-item', function () {
    loader = table.showLoader();
    $(this).addClass('active')
        .siblings()
        .removeClass('active')
        .closest('.dropdown')
        .find('.dropdown-toggle')
        .addClass('active')
        .find('span')
        .text($(this).text())

    road = $(this).data('val');
    renderSuffix();

    urlParams.delete('page');
    urlParams.set('road', road);
    url = `${window.location.protocol}//${window.location.host + window.location.pathname}?${urlParams}`;
    window.history.pushState({path: url}, '', url);
    reRenderTable();
});

$('input[name="min-filter"]').on('change', function () {
    minFilter = $(this).val();
    urlParams.delete('page');
    urlParams.set('min-filter', minFilter);
    url = `${window.location.protocol}//${window.location.host + window.location.pathname}?${urlParams}`;
    window.history.pushState({path: url}, '', url);
    reRenderTable();
});

$('input[name="max-filter"]').on('change', function () {
    maxFilter = $(this).val();
    urlParams.delete('page');
    urlParams.set('max-filter', maxFilter);
    url = `${window.location.protocol}//${window.location.host + window.location.pathname}?${urlParams}`;
    window.history.pushState({path: url}, '', url);
    reRenderTable();
});

function reRenderTable() {
    loader = table.showLoader();
    smartaxApi.setMinFilter(minFilter ?? default_min_filter)
    smartaxApi.setMaxFilter(maxFilter ?? default_max_filter)

    smartaxApi.setPage(initPage ?? 1);
    smartaxApi.getFeatures(suffix)
        .then(data => {
            table.setData(data);
            table.render();
            table.renderPagination()
            loader.close();
            renderDownloadURL();
        })
        .catch(error => {
            console.error('Error:', error);
        });
}

function renderDownloadURL() {
    let url = `/admin/smartax/download-report?min-filter=${minFilter ?? default_min_filter}&max-filter=${maxFilter ?? default_max_filter}` + suffix;

    $('a[data-action="export-excel"]').attr('href', url);
}