<?php namespace Visiosoft\SmartaxModule\Commands;

use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Request;
use function GuzzleHttp\Psr7\str;
use Illuminate\Support\Facades\DB;
use Visiosoft\SmartaxModule\Bbox\Contract\BboxRepositoryInterface;
use Visiosoft\SmartaxModule\Feature\Contract\FeatureRepositoryInterface;
use Visiosoft\SmartaxModule\Location\Contract\LocationRepositoryInterface;
use Visiosoft\SmartaxModule\Measurement\Contract\MeasurementRepositoryInterface;
use Visiosoft\SmartaxModule\Point\Contract\PointRepositoryInterface;
use Visiosoft\SmartaxModule\Relation\Contract\RelationRepositoryInterface;

class SyncMapilioFeature
{
    protected $token;
    protected $project_key;
    protected $organization_key;

    public function __construct($token, $project_key, $organization_key)
    {
        $this->token = $token;
        $this->project_key = $project_key;
        $this->organization_key = $organization_key;
    }

    public function handle(
        MeasurementRepositoryInterface $measurementRepository,
        BboxRepositoryInterface $bboxRepository,
        RelationRepositoryInterface $relationRepository,
        LocationRepositoryInterface $locationRepository,
        FeatureRepositoryInterface $featureRepository
    )
    {
        $counter = 0;
        $skip = $featureRepository->newQuery()->count();

        // Get Features With CURL Service
        $features = $this->getFeatures($skip);

        // Save Features
        foreach ($features as $item) {

            $properties = $item['properties'];
            $is_feature = $featureRepository->newQuery()
                ->where('unique_id', $properties['id'])
                ->first();

            if (!$is_feature) {

                $near_door_features = $this->getNearDoorFeatures($properties['feature']['lat'], $properties['feature']['lon']);
                $near_road_features = $this->getNearRoadFeatures($properties['feature']['lat'], $properties['feature']['lon']);

                $detected_district_id = null;
                $detected_road_id = null;
                $detected_door_id = null;

                if ($near_door_features) {
                    $near_door_feature = array_first($near_door_features['features']);

                    if ($near_door_feature) {
                        $detected_district_id = $near_door_feature['properties']['mahalle_id'];
                        $detected_road_id = $near_door_feature['properties']['yol_id'];
                        $detected_door_id = $near_door_feature['properties']['objectid'];
                    }

                    if ($near_road_features) {
                        $near_road_feature = array_first($near_road_features['features']);


                        if (!$near_door_feature && $near_road_feature) {
                            $detected_district_id = $near_road_feature['properties']['mahalle_id'];
                            $detected_road_id = $near_road_feature['properties']['objectid'];
                        }
                    }

                }

                /**
                 * Save Feature
                 */
                $feature = $featureRepository->newQuery()
                    ->create([
                        'lat' => $properties['feature']['lat'],
                        'lon' => $properties['feature']['lon'],
                        'created_at' => $properties['feature']['created_at'],
                        'class_code' => $properties['feature']['class_code'],
                        'confidence' => $properties['feature']['confidence'],
                        'area' => $properties['feature']['area'],
                        'verification' => $properties['feature']['verification'],
                        'unique_id' => $properties['id'],
                        'sequence_uuid' => $properties['sequence_uuid'],
                        'detected_district_id' => $detected_district_id,
                        'detected_road_id' => $detected_road_id,
                        'detected_door_id' => $detected_door_id
                    ]);

                // Geom Save (Feature)
                DB::statement("UPDATE default_smartax_feature
	                                        SET  geom=ST_GeomFromGeoJSON('" . $properties['feature']['geom'] . "')
	                                        WHERE id=" . $feature->id . ";");

                /**
                 * Save Measurement
                 */
                $measurementRepository->newQuery()
                    ->create([
                        'feature_id' => $feature->id,
                        'avg_width' => $properties['measurement']['avg_width'],
                        'avg_height' => $properties['measurement']['avg_height'],
                        'avg_area' => $properties['measurement']['avg_area'],
                        'sequence_uuid' => $properties['sequence_uuid'],
                    ]);


                /**
                 * Save BBOX
                 */

                $thumb_image_uuid = null;
                $thumb_image_score = 0.0;
                $thumb_image_bbox = [];

                foreach ($properties['matchedPoints'] as $matchedPoint) {

                    // Save BBOX 1
                    if (!$bbox_1 = $bboxRepository->newQuery()->where('bbox_key', $matchedPoint['properties']['objId_1'])->first()) {
                        $bbox_1 = $bboxRepository->newQuery()
                            ->create([
                                'x_min' => $matchedPoint['properties']['bbox_1'][0],
                                'y_min' => $matchedPoint['properties']['bbox_1'][1],
                                'x_max' => $matchedPoint['properties']['bbox_1'][2],
                                'y_max' => $matchedPoint['properties']['bbox_1'][3],
                                'score' => $matchedPoint['properties']['score_1'],
                                'photo_uuid' => $matchedPoint['properties']['photo_uuid_1'],
                                'bbox_key' => $matchedPoint['properties']['objId_1'],
                                'sequence_uuid' => $properties['sequence_uuid'],
                            ]);

                        //Set Thumbnail for Best Score
                        if ($thumb_image_score < $matchedPoint['properties']['score_1']) {
                            $thumb_image_score = $matchedPoint['properties']['score_1'];
                            $thumb_image_uuid = $matchedPoint['properties']['photo_uuid_1'];
                            $thumb_image_bbox = $matchedPoint['properties']['bbox_1'];
                        }

                    }

                    // Save BBOX 2
                    if (!$bbox_2 = $bboxRepository->newQuery()->where('bbox_key', $matchedPoint['properties']['objId_2'])->first()) {
                        $bbox_2 = $bboxRepository->newQuery()
                            ->create([
                                'x_min' => $matchedPoint['properties']['bbox_2'][0],
                                'y_min' => $matchedPoint['properties']['bbox_2'][1],
                                'x_max' => $matchedPoint['properties']['bbox_2'][2],
                                'y_max' => $matchedPoint['properties']['bbox_2'][3],
                                'score' => $matchedPoint['properties']['score_2'],
                                'photo_uuid' => $matchedPoint['properties']['photo_uuid_2'],
                                'bbox_key' => $matchedPoint['properties']['objId_2'],
                                'sequence_uuid' => $properties['sequence_uuid'],
                            ]);

                        //Set Thumbnail for Best Score
                        if ($thumb_image_score < $matchedPoint['properties']['score_2']) {
                            $thumb_image_score = $matchedPoint['properties']['score_2'];
                            $thumb_image_uuid = $matchedPoint['properties']['photo_uuid_2'];
                            $thumb_image_bbox = $matchedPoint['properties']['bbox_2'];
                        }
                    }

                    // Save Relation
                    $relation = $relationRepository->newQuery()
                        ->create([
                            'bbox_id_1' => $bbox_1->id,
                            'bbox_id_2' => $bbox_2->id,
                            'feature_id' => $feature->id,
                            'sequence_uuid' => $properties['sequence_uuid'],
                        ]);

                    // Save Location
                    $location = $locationRepository->newQuery()
                        ->create([
                            'lat' => $matchedPoint['geometry']['coordinates'][1],
                            'lon' => $matchedPoint['geometry']['coordinates'][0],
                            'score' => ($matchedPoint['properties']['score_1'] + $matchedPoint['properties']['score_2']) / 2,
                            'relation_id' => $relation->id,
                            'sequence_uuid' => $properties['sequence_uuid'],
                        ]);

                    // Geom Save (Location)
                    DB::statement("UPDATE default_smartax_location
	                                        SET  geom=ST_GeomFromGeoJSON('" . json_encode($matchedPoint['geometry']) . "')
	                                        WHERE id=" . $location->id . ";");
                }

                //Update Thumbnail Attribute
                $feature->setAttribute('thumbnail_photo_uuid', $thumb_image_uuid);
                $feature->setAttribute('thumbnail_photo_bbox', json_encode($thumb_image_bbox));
                $feature->save();

                $counter++;
            }
        }

        return [
            'counter' => $counter
        ];
    }

    public function getFeatures($skip)
    {
        $features = $this->newRequest($this->token, $this->organization_key, $this->project_key, $skip);

        if (isset($features['data']['features']) && count($features['data']['features'])) {
            return $features['data']['features'];
        }

        return [];
    }

    public function newRequest($token, $organization_key, $project_key, $skip = 0)
    {
        try {
            $url = setting_value('visiosoft.module.smartax::mapilio_sync_url') . "/api/features/list" . '?' . http_build_query([
                    'token' => $token,
                    'organization_key' => $organization_key,
                    'project_key' => $project_key,
                    'skip' => $skip
                ]);

            $client = new Client([
                'cookies' => true,
                'http_errors' => false,
                'verify' => false
            ]);

            $response = $client->request('GET', $url);

            $response = $response->getBody()->getContents();

            $response = json_decode($response, true);

            if (!$response['status']) {
                throw new \Exception([$response['message']]);
            }

            return $response['response'];

        } catch (\Exception $exception) {
            throw new \Exception(["The incoming data is not correct."]);
            die;
        }
    }

    public function getNearRoadFeatures($lat, $lon, $meter = 25)
    {
        $client = new Client([
            'cookies' => true,
            'http_errors' => false,
            'verify' => false
        ]);
        $km = $meter * 0.001;
        $url = "https://geo.mapilio.com/geoserver/visiosoft/ows?service=WFS&version=1.0.0&request=GetFeature&typeName=visiosoft%3Abasaksehir_yol&maxFeatures=2&outputFormat=application%2Fjson&CQL_FILTER=DWITHIN(wkb_geometry,Point($lon $lat),$km,kilometers)";
        $request = new Request('GET', $url);
        $res = $client->sendAsync($request)->wait();
        $response = $res->getBody()->getContents();

        return json_decode($response, true);
    }

    public function getNearDoorFeatures($lat, $lon, $meter = 10)
    {
        $client = new Client([
            'cookies' => true,
            'http_errors' => false,
            'verify' => false
        ]);
        $km = $meter * 0.001;
        $url = "https://geo.mapilio.com/geoserver/visiosoft/ows?service=WFS&version=1.0.0&request=GetFeature&typeName=visiosoft%3Abasaksehir_kapi&maxFeatures=2&outputFormat=application%2Fjson&CQL_FILTER=DWITHIN(wkb_geometry,Point($lon $lat),$km,kilometers)";
        $request = new Request('GET', $url);
        $res = $client->sendAsync($request)->wait();
        $response = $res->getBody()->getContents();

        return json_decode($response, true);
    }
}