<?php namespace Visiosoft\SmartaxModule\Http\Controller\Admin;

use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Visiosoft\SmartaxModule\Bbox\Contract\BboxRepositoryInterface;
use Visiosoft\SmartaxModule\Commands\getGeoDistricts;
use Visiosoft\SmartaxModule\Commands\getGeoDoors;
use Visiosoft\SmartaxModule\Commands\getGeoRoads;
use Visiosoft\SmartaxModule\Feature\Contract\FeatureRepositoryInterface;
use Visiosoft\SmartaxModule\Location\Contract\LocationRepositoryInterface;
use Visiosoft\SmartaxModule\Measurement\Contract\MeasurementRepositoryInterface;
use Visiosoft\SmartaxModule\Point\Contract\PointRepositoryInterface;
use Visiosoft\SmartaxModule\Relation\Contract\RelationRepositoryInterface;
use Yajra\DataTables\DataTables;

class ApiController extends AdminController
{
    public function geoFeatures($type)
    {
        switch ($type) {
            case "mahalle":
                $geo = $this->dispatch(new getGeoDistricts($this->request->all()));
                break;
            case "yol":
                $geo = $this->dispatch(new getGeoRoads($this->request->all()));
                break;
            case "kapi":
                $geo = $this->dispatch(new getGeoDoors($this->request->all()));
                break;
        }

        return $this->response->json(json_decode($geo, true));
    }

    public function features(FeatureRepositoryInterface $featureRepository)
    {
        $area_price = setting_value('visiosoft.module.smartax::tax-price');

        $features = DB::select(
            "select * from (
                    SELECT default_smartax_feature.*,
                        round( CAST((default_smartax_feature.area*$area_price) as numeric), 2) as price,
                        default_smartax_feature.detected_district_id as district,
                        default_smartax_feature.detected_road_id as road,
                        default_smartax_feature.detected_door_id as door,
                        default_basaksehir_mahalle.adi_numara AS district_name,
                        default_basaksehir_yol.yol_adi AS road_name,
                        default_basaksehir_kapi.kapi_no AS door_name,
                        default_smartax_point.uploaded_hash,
                        default_smartax_point.resolution,
                        default_smartax_point.filename
                    FROM default_smartax_feature
                    LEFT JOIN default_basaksehir_mahalle ON default_smartax_feature.detected_district_id = default_basaksehir_mahalle.objectid
                    LEFT JOIN default_basaksehir_yol ON default_smartax_feature.detected_road_id = default_basaksehir_yol.objectid
                    LEFT JOIN default_basaksehir_kapi ON default_smartax_feature.detected_door_id = default_basaksehir_kapi.objectid
                    LEFT JOIN default_smartax_point ON default_smartax_feature.thumbnail_photo_uuid = default_smartax_point.photo_uuid
                    WHERE class_code = 'inst-sign-store'
                        AND area < " . number_format(setting_value('visiosoft.module.smartax::max-area', '20.0'), '1', '.', '') . "
                        AND area > " . number_format(setting_value('visiosoft.module.smartax::min-area', '3.0'), '1', '.', '') . "
                        AND thumbnail_photo_uuid IS NOT NULL
                        AND default_smartax_feature.deleted_at IS NULL
                ) as default_smartax_feature"
        );

        return DataTables::of($features)->toJson();
    }

    public function pointDetail($id)
    {
        $featureRepository = app(FeatureRepositoryInterface::class);
        $bbox_repository = app(BboxRepositoryInterface::class);
        $relation_repository = app(RelationRepositoryInterface::class);
        $point_repository = app(PointRepositoryInterface::class);

        $feature = $featureRepository->newQuery()->find($id);

        $relations = $relation_repository->newQuery()
            ->where('feature_id', $feature->id)
            ->get();

        $bbox_id_entries = array();

        foreach ($relations as $relation) {
            $bbox_id_entries[] = $relation->bbox_id_1;
            $bbox_id_entries[] = $relation->bbox_id_2;
        }

        $bbox_id_entries = array_unique($bbox_id_entries);

        $bbox_entries = $bbox_repository->newQuery()->whereIn('id', $bbox_id_entries)->get();

        $bbox_entries = $bbox_entries->sortBy('score');

        foreach ($bbox_entries as $bbox_entry) {
            $bbox_entry->detail = $point_repository->newQuery()->where('photo_uuid', $bbox_entry->photo_uuid)->first();
        }

        $feature->bbox_entries = $bbox_entries;

        return response()->json($feature);
    }

    public function editPoint(FeatureRepositoryInterface $featureRepository,
                              BboxRepositoryInterface $bboxRepository,
                              RelationRepositoryInterface $relationRepository,
                              LocationRepositoryInterface $locationRepository,
                              MeasurementRepositoryInterface $measurementRepository,
                              $id)
    {
        $status = false;
        $required_fields = ['lat', 'lon', 'area', 'calculate_at', 'height', 'width'];

        try {
            if (request()->has($required_fields) and $feature = $featureRepository->find($id)) {

                $unique_key = Str::random(17);

                //Save Calculation
                $feature->update([
                    'lat' => $this->request->lat,
                    'lon' => $this->request->lon,
                    'area' => $this->request->area,
                    'confidence' => 1.0,
                ]);

                // Save Geom for Calculation
                DB::statement("UPDATE default_smartax_feature
	                                        SET  geom=ST_GeomFromGeoJSON('" . json_encode($this->request->geometry) . "')
	                                        WHERE id=" . $feature->id . ";");

                $measurementRepository->newQuery()
                    ->where('feature_id', $id)
                    ->update([
                        'avg_width' => $this->request->width,
                        'avg_height' => $this->request->height,
                        'avg_area' => $this->request->area,
                    ]);

                // DELETE OLD ENTRIES
                $this->deleteOldEntries($id);

                foreach ($this->request->matchedPoints as $item) {
                    $sequence_uuid = null;

                    $bbox_1_unique_key = $unique_key . $item['obj_1'];
                    $bbox_2_unique_key = $unique_key . $item['obj_2'];

                    // Save BBOX 1
                    if (!$bbox_1 = $bboxRepository->newQuery()->where('bbox_key', $bbox_1_unique_key)->first()) {

                        $old_bbox_1 = $bboxRepository->findWithTrashed($item['obj_1']);

                        $sequence_uuid = $old_bbox_1->sequence_uuid;

                        $bbox_1 = $bboxRepository->newQuery()
                            ->create([
                                'score' => 1.0,
                                'photo_uuid' => $old_bbox_1->photo_uuid,
                                'bbox_key' => $bbox_1_unique_key,
                                'sequence_uuid' => $old_bbox_1->sequence_uuid,
                                'x_min' => $item['bbox_1'][0],
                                'y_min' => $item['bbox_1'][1],
                                'x_max' => $item['bbox_1'][2],
                                'y_max' => $item['bbox_1'][3],
                            ]);
                    }

                    // Save BBOX 2
                    if (!$bbox_2 = $bboxRepository->newQuery()->where('bbox_key', $bbox_2_unique_key)->first()) {

                        $old_bbox_2 = $bboxRepository->findWithTrashed($item['obj_2']);

                        $bbox_2 = $bboxRepository->newQuery()
                            ->create([
                                'score' => 1.0,
                                'photo_uuid' => $old_bbox_2->photo_uuid,
                                'bbox_key' => $bbox_2_unique_key,
                                'sequence_uuid' => $old_bbox_2->sequence_uuid,
                                'x_min' => $item['bbox_2'][0],
                                'y_min' => $item['bbox_2'][1],
                                'x_max' => $item['bbox_2'][2],
                                'y_max' => $item['bbox_2'][3],
                            ]);
                    }

                    //Save RELATION

                    if (!$relation = $relationRepository->newQuery()
                        ->where('bbox_id_1', $bbox_1->id)
                        ->where('bbox_id_2', $bbox_2->id)
                        ->where('feature_id', $id)
                        ->first()) {
                        $relation = $relationRepository->newQuery()
                            ->create([
                                'bbox_id_1' => $bbox_1->id,
                                'bbox_id_2' => $bbox_2->id,
                                'feature_id' => $id,
                                'sequence_uuid' => $sequence_uuid
                            ]);
                    }

                    //Save LOCATION
                    if (!$location = $locationRepository->newQuery()
                        ->where('relation_id', $relation->id)
                        ->first()) {

                        $location = $locationRepository->newQuery()
                            ->create([
                                'sequence_uuid' => $sequence_uuid,
                                'relation_id' => $relation->id,
                                'score' => 1.0,
                                'lat' => $this->request->lat,
                                'lon' => $this->request->lon,
                            ]);

                        // Save Geom for Location
                        DB::statement("UPDATE default_smartax_location
	                                        SET  geom=ST_GeomFromGeoJSON('" . json_encode($this->request->geometry) . "')
	                                        WHERE id=" . $location->id . ";");
                    }
                }

                $status = true;
            }
        } catch (\Exception $exception) {
        }

        return response()->json(['status' => $status]);
    }

    public function deleteOldEntries($feature_id)
    {
        $bboxRepository = app(BboxRepositoryInterface::class);
        $relationRepository = app(RelationRepositoryInterface::class);
        $locationRepository = app(LocationRepositoryInterface::class);

        //Get OLD Relations
        $old_relations = $relationRepository->newQuery()
            ->where('feature_id', $feature_id)
            ->get();

        //Collect OLD BBOX ID's
        $old_bbox_ids = array();

        foreach ($old_relations as $old_relation) {
            $old_bbox_ids[] = $old_relation->bbox_id_1;
            $old_bbox_ids[] = $old_relation->bbox_id_2;
        }

        /*  Remove duplicate id's  */
        $old_bbox_ids = array_unique($old_bbox_ids);


        //Delete OLD BBOX Entries
        $bboxRepository->newQuery()
            ->whereIn('id', $old_bbox_ids)
            ->delete();

        //Delete OLD LOCATION Entries
        $locationRepository->newQuery()
            ->whereIn('relation_id', $old_relations->pluck('id')->all())
            ->delete();

        //Delete OLD RELATION Entries
        $relationRepository->newQuery()
            ->where('feature_id', $feature_id)
            ->delete();
    }

    public function getDistrictReport()
    {
        $query = "SELECT CASE WHEN 
                    DISTRICT.ADI_NUMARA IS NULL 
                    THEN 'Diğer' ELSE DISTRICT.ADI_NUMARA || ' ' || 'Mahallesi'
                    END AS NAME,
                    round( CAST(COUNT(*) * 100.0 / SUM(COUNT(*)) OVER() as numeric), 2) as percentage
                    FROM DEFAULT_SMARTAX_FEATURE AS FEATURE
                    LEFT JOIN DEFAULT_BASAKSEHIR_MAHALLE AS DISTRICT ON FEATURE.DETECTED_DISTRICT_ID = DISTRICT.OBJECTID
                    GROUP BY DETECTED_DISTRICT_ID,DISTRICT.ADI_NUMARA ORDER BY DISTRICT.ADI_NUMARA;";

        $entries = DB::select($query);

        $chart_data = array();
        foreach (collect($entries) as $entry) {
            $chart_data[] = [
                $entry->name,
                floatval($entry->percentage)
            ];
        }

        return $this->response->json($chart_data);
    }
}
