<?php namespace Visiosoft\SmartaxModule\Http\Controller\Admin;

use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Visiosoft\SmartaxModule\Bbox\Contract\BboxRepositoryInterface;
use Visiosoft\SmartaxModule\Commands\getGeoDistricts;
use Visiosoft\SmartaxModule\Commands\getGeoDoors;
use Visiosoft\SmartaxModule\Commands\getGeoRoads;
use Visiosoft\SmartaxModule\Feature\Contract\FeatureRepositoryInterface;
use Visiosoft\SmartaxModule\Location\Contract\LocationRepositoryInterface;
use Visiosoft\SmartaxModule\Measurement\Contract\MeasurementRepositoryInterface;
use Visiosoft\SmartaxModule\Point\Contract\PointRepositoryInterface;
use Visiosoft\SmartaxModule\Relation\Contract\RelationRepositoryInterface;

class ApiController extends AdminController
{
    public function geoFeatures($type)
    {
        switch ($type) {
            case "mahalle":
                $geo = $this->dispatch(new getGeoDistricts($this->request->all()));
                break;
            case "yol":
                $geo = $this->dispatch(new getGeoRoads($this->request->all()));
                break;
            case "kapi":
                $geo = $this->dispatch(new getGeoDoors($this->request->all()));
                break;
        }

        return $this->response->json(json_decode($geo, true));
    }

    public function features(FeatureRepositoryInterface $featureRepository)
    {
        $page = request()->all();

        $features = $featureRepository->newQuery()
            ->leftJoin('basaksehir_mahalle', function ($leftJoin) {
                $leftJoin->on('smartax_feature.detected_district_id', '=', 'basaksehir_mahalle.objectid');
            })
            ->leftJoin('basaksehir_yol', function ($leftJoin) {
                $leftJoin->on('smartax_feature.detected_road_id', '=', 'basaksehir_yol.objectid');
            })
            ->leftJoin('basaksehir_kapi', function ($leftJoin) {
                $leftJoin->on('smartax_feature.detected_door_id', '=', 'basaksehir_kapi.objectid');
            })
            ->leftJoin('smartax_point', 'smartax_feature.thumbnail_photo_uuid', 'smartax_point.photo_uuid')
            ->select('smartax_feature.*',
                'basaksehir_mahalle.adi_numara as district_name',
                'basaksehir_yol.yol_adi as road_name',
                'basaksehir_kapi.kapi_no as door_id',
                'smartax_point.uploaded_hash',
                'smartax_point.resolution',
                'smartax_point.filename')
            ->where('class_code', 'inst-sign-store')
            ->where('area', '<', number_format(setting_value('visiosoft.module.smartax::max-area','20.0'),'1','.',''))
            ->where('area', '>', number_format(setting_value('visiosoft.module.smartax::min-area','3.0'),'1','.',''))
            ->whereNotNull('thumbnail_photo_uuid');

        // Filter Filterable Columns
        $filterables = array_filter($page['columns'], function ($column) {
            return $column['search']['value'];
        });

        // Add Filters in Query
        foreach ($filterables as $filterable) {
            if ($filterable['data'] === 'id' and is_int($filterable['search']['value'])) {
                $keyword = $filterable['search']['value'];
                $features = $features->where('id', "$keyword");
            } elseif ($filterable['data'] === 'formatted_created_at') {
                $dates = explode('|', $filterable['search']['value']);
                if ($dates[0]) {
                    $features = $features->whereDate('created_at', '>=', $dates[0]);
                }

                if ($dates[1]) {
                    $features = $features->whereDate('created_at', '<=', $dates[1]);
                }
            } elseif ($filterable['data'] === 'district_name') {
                $features = $features->where('basaksehir_mahalle.objectid', $filterable['search']['value']);
            } elseif ($filterable['data'] === 'road_name') {
                $features = $features->where('basaksehir_yol.objectid', $filterable['search']['value']);
            } elseif ($filterable['data'] === 'door_id') {
                $features = $features->where('basaksehir_kapi.objectid', $filterable['search']['value']);
            }
        }

        // Set Order in Query
        if (isset($page['order'])) {
            $order = $page['order'][0];

            $colExists = $featureRepository
                ->getModel()
                ->getConnection()
                ->getSchemaBuilder()
                ->hasColumn($featureRepository->getModel()->getTable(), $page['columnsDef'][$order['column']]);

            if ($colExists) {
                $features = $features->orderBy($page['columnsDef'][$order['column']], $order['dir']);
            }
        }

        $features = $features->paginate(
            $page['length'],
            ['*'],
            'page',
            $page['start'] / $page['length'] + 1
        );

        foreach ($features as $feature) {
            $feature->formatted_created_at = $feature->created_at->format('d.m.Y');
        }

        return [
            "meta" => [
                "page" => $features->currentPage(),
                "pages" => $features->lastPage(),
                "perpage" => $page['length'],
                "total" => $features->total(),
                "sort" => "desc",
                "field" => "id"
            ],
            "data" => $features->items(),
            "recordsFiltered" => $features->total(),
            "recordsTotal" => $featureRepository->count(),
        ];
    }

    public function pointDetail($id)
    {
        $featureRepository = app(FeatureRepositoryInterface::class);
        $bbox_repository = app(BboxRepositoryInterface::class);
        $relation_repository = app(RelationRepositoryInterface::class);
        $point_repository = app(PointRepositoryInterface::class);

        $feature = $featureRepository->newQuery()->find($id);

        $relations = $relation_repository->newQuery()
            ->where('feature_id', $feature->id)
            ->get();

        $bbox_id_entries = array();

        foreach ($relations as $relation) {
            $bbox_id_entries[] = $relation->bbox_id_1;
            $bbox_id_entries[] = $relation->bbox_id_2;
        }

        $bbox_id_entries = array_unique($bbox_id_entries);

        $bbox_entries = $bbox_repository->newQuery()->whereIn('id', $bbox_id_entries)->get();

        $bbox_entries = $bbox_entries->sortBy('score');

        foreach ($bbox_entries as $bbox_entry) {
            $bbox_entry->detail = $point_repository->newQuery()->where('photo_uuid', $bbox_entry->photo_uuid)->first();
        }

        $feature->bbox_entries = $bbox_entries;

        return response()->json($feature);
    }

    public function editPoint(FeatureRepositoryInterface $featureRepository,
                              BboxRepositoryInterface $bboxRepository,
                              RelationRepositoryInterface $relationRepository,
                              LocationRepositoryInterface $locationRepository,
                              MeasurementRepositoryInterface $measurementRepository,
                              $id)
    {
        $status = false;
        $required_fields = ['lat', 'lon', 'area', 'calculate_at', 'height', 'width'];

        try {
            if (request()->has($required_fields) and $feature = $featureRepository->find($id)) {

                $unique_key = Str::random(17);

                //Save Calculation
                $feature->update([
                    'lat' => $this->request->lat,
                    'lon' => $this->request->lon,
                    'area' => $this->request->area,
                    'confidence' => 1.0,
                ]);

                // Save Geom for Calculation
                DB::statement("UPDATE default_smartax_feature
	                                        SET  geom=ST_GeomFromGeoJSON('" . json_encode($this->request->geometry) . "')
	                                        WHERE id=" . $feature->id . ";");

                $measurementRepository->newQuery()
                    ->where('feature_id', $id)
                    ->update([
                        'avg_width' => $this->request->width,
                        'avg_height' => $this->request->height,
                        'avg_area' => $this->request->area,
                    ]);

                // DELETE OLD ENTRIES
                $this->deleteOldEntries($id);

                foreach ($this->request->matchedPoints as $item) {
                    $sequence_uuid = null;

                    $bbox_1_unique_key = $unique_key . $item['obj_1'];
                    $bbox_2_unique_key = $unique_key . $item['obj_2'];

                    // Save BBOX 1
                    if (!$bbox_1 = $bboxRepository->newQuery()->where('bbox_key', $bbox_1_unique_key)->first()) {

                        $old_bbox_1 = $bboxRepository->findWithTrashed($item['obj_1']);

                        $sequence_uuid = $old_bbox_1->sequence_uuid;

                        $bbox_1 = $bboxRepository->newQuery()
                            ->create([
                                'score' => 1.0,
                                'photo_uuid' => $old_bbox_1->photo_uuid,
                                'bbox_key' => $bbox_1_unique_key,
                                'sequence_uuid' => $old_bbox_1->sequence_uuid,
                                'x_min' => $item['bbox_1'][0],
                                'y_min' => $item['bbox_1'][1],
                                'x_max' => $item['bbox_1'][2],
                                'y_max' => $item['bbox_1'][3],
                            ]);
                    }

                    // Save BBOX 2
                    if (!$bbox_2 = $bboxRepository->newQuery()->where('bbox_key', $bbox_2_unique_key)->first()) {

                        $old_bbox_2 = $bboxRepository->findWithTrashed($item['obj_2']);

                        $bbox_2 = $bboxRepository->newQuery()
                            ->create([
                                'score' => 1.0,
                                'photo_uuid' => $old_bbox_2->photo_uuid,
                                'bbox_key' => $bbox_2_unique_key,
                                'sequence_uuid' => $old_bbox_2->sequence_uuid,
                                'x_min' => $item['bbox_2'][0],
                                'y_min' => $item['bbox_2'][1],
                                'x_max' => $item['bbox_2'][2],
                                'y_max' => $item['bbox_2'][3],
                            ]);
                    }

                    //Save RELATION

                    if (!$relation = $relationRepository->newQuery()
                        ->where('bbox_id_1', $bbox_1->id)
                        ->where('bbox_id_2', $bbox_2->id)
                        ->where('feature_id', $id)
                        ->first()) {
                        $relation = $relationRepository->newQuery()
                            ->create([
                                'bbox_id_1' => $bbox_1->id,
                                'bbox_id_2' => $bbox_2->id,
                                'feature_id' => $id,
                                'sequence_uuid' => $sequence_uuid
                            ]);
                    }

                    //Save LOCATION
                    if (!$location = $locationRepository->newQuery()
                        ->where('relation_id', $relation->id)
                        ->first()) {

                        $location = $locationRepository->newQuery()
                            ->create([
                                'sequence_uuid' => $sequence_uuid,
                                'relation_id' => $relation->id,
                                'score' => 1.0,
                                'lat' => $this->request->lat,
                                'lon' => $this->request->lon,
                            ]);

                        // Save Geom for Location
                        DB::statement("UPDATE default_smartax_location
	                                        SET  geom=ST_GeomFromGeoJSON('" . json_encode($this->request->geometry) . "')
	                                        WHERE id=" . $location->id . ";");
                    }
                }

                $status = true;
            }
        } catch (\Exception $exception) {
        }

        return response()->json(['status' => $status]);
    }

    public function deleteOldEntries($feature_id)
    {
        $bboxRepository = app(BboxRepositoryInterface::class);
        $relationRepository = app(RelationRepositoryInterface::class);
        $locationRepository = app(LocationRepositoryInterface::class);

        //Get OLD Relations
        $old_relations = $relationRepository->newQuery()
            ->where('feature_id', $feature_id)
            ->get();

        //Collect OLD BBOX ID's
        $old_bbox_ids = array();

        foreach ($old_relations as $old_relation) {
            $old_bbox_ids[] = $old_relation->bbox_id_1;
            $old_bbox_ids[] = $old_relation->bbox_id_2;
        }

        /*  Remove duplicate id's  */
        $old_bbox_ids = array_unique($old_bbox_ids);


        //Delete OLD BBOX Entries
        $bboxRepository->newQuery()
            ->whereIn('id', $old_bbox_ids)
            ->delete();

        //Delete OLD LOCATION Entries
        $locationRepository->newQuery()
            ->whereIn('relation_id', $old_relations->pluck('id')->all())
            ->delete();

        //Delete OLD RELATION Entries
        $relationRepository->newQuery()
            ->where('feature_id', $feature_id)
            ->delete();
    }

    public function getDistrictReport()
    {
        $query = "SELECT CASE WHEN 
                    DISTRICT.ADI_NUMARA IS NULL 
                    THEN 'Diğer' ELSE DISTRICT.ADI_NUMARA || ' ' || 'Mahallesi'
                    END AS NAME,
                    round( CAST(COUNT(*) * 100.0 / SUM(COUNT(*)) OVER() as numeric), 2) as percentage
                    FROM DEFAULT_SMARTAX_FEATURE AS FEATURE
                    LEFT JOIN DEFAULT_BASAKSEHIR_MAHALLE AS DISTRICT ON FEATURE.DETECTED_DISTRICT_ID = DISTRICT.OBJECTID
                    GROUP BY DETECTED_DISTRICT_ID,DISTRICT.ADI_NUMARA ORDER BY DISTRICT.ADI_NUMARA;";

        $entries = DB::select($query);

        $chart_data = array();
        foreach (collect($entries) as $entry) {
            $chart_data[] = [
                $entry->name,
                floatval($entry->percentage)
            ];
        }

        return $this->response->json($chart_data);
    }
}
