<?php namespace Visiosoft\SiteModule\Http\Controller;

use Anomaly\Streams\Platform\Field\Contract\FieldRepositoryInterface;
use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Anomaly\UsersModule\Role\Contract\RoleRepositoryInterface;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Anomaly\UsersModule\User\UserActivator;
use Anomaly\UsersModule\User\UserModel;
use EmailChecker\Laravel\EmailCheckerFacade;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SiteModule\Site\Jobs\UserActivatedBySms;
use Visiosoft\SpamDetectionExtension\Helper\SpamDetector;
use Visiosoft\SubscriptionsModule\Subscription\Command\RegisteredUserBySubscription;

class DemoFormController extends ResourceController
{
    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function validation()
    {
        $inputs = $this->request->all();
        $errors = array();

        //Convert Alphanumeric
        if (!empty($inputs['storename'])) {
            $inputs['storename'] = strtolower(preg_replace("/[^a-zA-Z0-9]+/", "", $inputs['storename']));
        }

        $validator = Validator::make($inputs, [
            'storename' => 'required|max:8|min:3|alpha_num',
            'email' => 'required|email',
            'name' => 'required',
            'surname' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors(), 'message' => null], 400);
        }

        if (Auth::check()) {
            if (Auth::user()->getEmail() != $this->request->email) {
                $errors['email'][] = trans('visiosoft.module.site::message.please_use_a_real_email');
            }
        } else {
            $isUserByEmail = $this->findUserByColumn('email', $this->request->email);

            if ($isUserByEmail && $isUserByEmail->isActivated()) {
                $errors['email'][] = trans('visiosoft.module.site::message.already_exists_email');
            }
        }

        $spamDetector = app(SpamDetector::class);
        $content = $inputs['storename'];
        if (is_array($content)) {
            $content = json_encode($content);
        }
        if ($spamDetector->detector($content)) {
            $errors['storename'][] = trans('visiosoft.module.site::message.contains_spam_word');
        }

        if (count($errors)) {
            return response()->json(['status' => false, 'errors' => $errors, 'message' => null], 400);
        }
        return response()->json(['status' => true, 'errors' => null, 'message' => null], 200);
    }

    /**
     * @param $column
     * @param $value
     * @return mixed|null
     */
    private function findUserByColumn($column, $value)
    {
        if (!is_null($user = UserModel::withTrashed()->where($column, $value)->get())) {
            return $user->first();
        }
        return null;
    }

    /**
     * @param $email
     * @return bool
     */
    private function isFakeEmail($email)
    {
        return !EmailCheckerFacade::isValid($email);
    }

    /**
     * @param UserRepositoryInterface $users
     * @param RoleRepositoryInterface $roles
     * @param SiteRepositoryInterface $sites
     * @param UserActivator $activator
     * @param FieldRepositoryInterface $fieldRepository
     * @return \Illuminate\Http\JsonResponse
     * @throws \Psr\Container\ContainerExceptionInterface
     * @throws \Psr\Container\NotFoundExceptionInterface
     */
    public function store(
        UserRepositoryInterface  $users,
        RoleRepositoryInterface  $roles,
        SiteRepositoryInterface  $sites,
        UserActivator            $activator,
        FieldRepositoryInterface $fieldRepository)
    {
        $request = request()->all();

        //Convert Alphanumeric
        if (!empty($request['storename'])) {
            $request['storename'] = strtolower(preg_replace("/[^a-zA-Z0-9]+/", "", $request['storename']));
        }

        // Validate Request
        $validator = Validator::make($request, [
            'storename' => 'required|max:8|min:3|alpha_num',
            'email' => 'required|email',
            'name' => 'required',
            'surname' => 'required',
        ]);

        // Response Errors
        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors(), 'message' => null], 400);
        }

        try {
            $user_step = false;
            $store_step = false;
            $user_activated = false;
            $errors = array();
            $password = rand(111111, 999999);

            /**
             * Step 1 | User Step
             * In this step, user operations are performed from the request.
             */

            if (Auth::check()) {
                if (Auth::user()->getEmail() != $this->request->email) {
                    $errors['email'][] = trans('visiosoft.module.site::message.please_use_a_real_email');
                } else {
                    $user_activated = true;
                    $user_step = true;
                    $user = Auth::user();
                }
            } else {
                // Check Email
                $user = $this->findUserByColumn('email', $this->request->email);
                if (!$user) {
                    //Check Fake Email
                    if ($this->isFakeEmail($this->request->email)) {
                        $errors['email'][] = trans('visiosoft.module.site::message.please_use_a_real_email');
                    } else {
                        //Create New User
                        $name_surname = $this->request->name . ' ' . $this->request->surname;
                        $userParameters = array();
                        $userParameters['email'] = $this->request->email;
                        $userParameters['display_name'] = $name_surname;
                        $userParameters['username'] = Str::slug($this->request->name . ' ' . strtotime("now"), '_');
                        $userParameters['first_name'] = $this->request->name;
                        $userParameters['last_name'] = $this->request->surname;
                        $userParameters['str_id'] = str_random(24);

                        $user = $users->create($userParameters);
                        $user->setAttribute('password', $password);
                        $user->setAttribute('activation_code', $password);
//                        $user->setAttribute('gsm_phone', $this->request->phone);
                        $user->setAttribute('roles', $roles->findBySlug('user'));

                        //Save UTM parameters
                        if ($fieldRepository->findBySlugAndNamespace('utm_source', 'users')) {
                            $user->setAttribute('utm_source', (isset($_COOKIE['utm_source'])) ? $_COOKIE['utm_source'] : null);
                            $user->setAttribute('utm_medium', (isset($_COOKIE['utm_medium'])) ? $_COOKIE['utm_medium'] : null);
                            $user->setAttribute('utm_campaign', (isset($_COOKIE['utm_campaign'])) ? $_COOKIE['utm_campaign'] : null);
                            $user->setAttribute('utm_term', (isset($_COOKIE['utm_term'])) ? $_COOKIE['utm_term'] : null);
                            $user->setAttribute('utm_content', (isset($_COOKIE['utm_content'])) ? $_COOKIE['utm_content'] : null);
                        }

                        //Save browser language
                        if ($fieldRepository->findBySlugAndNamespace('browser_lang', 'users') and isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
                            $user->setAttribute('browser_lang', substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2));
                        }

                        //Save ip location
                        if ($fieldRepository->findBySlugAndNamespace('location_for_ip', 'users')) {
                            try {
                                $ip = $this->request->ip();
                                $location = file_get_contents("https://ipinfo.io/$ip/json");
                                $location = json_decode($location, true);
                                $user->setAttribute('location_for_ip', $location['country']);
                            } catch (\Exception $exception) {
                            }
                        }
                        $user->save();

                        $user_step = true;
                    }
                } elseif (!is_null($user->deleted_at) || $user->isActivated()) {
                    $errors['email'][] = trans('visiosoft.module.site::message.already_exists_email');
                } else {
                    $user_step = true;
                }
            }

            /**
             * Step 2 | Store (Site) Step
             * In this step, store(site) transactions are performed from the request.
             */

            if ($user_step) {
                //Check SubDomain
                if (strlen($request['storename']) <= 8) {
                    $store_step = true;
                    $storename = $request['storename'];

                    $site = $sites->createSite($storename, $user, $password, [
                        'type' => setting_value('visiosoft.module.site::default_domain'),
                        'node' => setting_value('visiosoft.module.deploy::default_demo_server'),
                        'default_theme' => $this->request->theme,
                        'locale' => request()->get('locale', 'en')
                    ]);

                } else {
                    $errors['storename'][] = trans('visiosoft.module.site::message.storename_max_char');
                }
            }

            if (count($errors)) {
                return response()->json(['status' => false, 'errors' => $errors, 'message' => null], 400);
            }

//            if (!isset($this->request['noSMS'])) {
//                $this->sendVerifySms($this->request->phone ?? $user->gsm_phone, $password);
//            }

            if (!$user_activated) {
                $user->setAttribute('activation_code', $password);
                $user->save();
                $activator->send($user, route('visiosoft.module.site::installation.started'));
            } else {
                $this->dispatch(new RegisteredUserBySubscription($user));
            }

            $response_data = [
                'email' => $this->request->email,
                'storename' => $site->subdomain_name,
                'isActive' => $user_activated
            ];

            return response()->json([
                'status' => true,
                'errors' => null,
                'data' => $response_data,
                'vid' => $site->api_token,
                'message' => isset($this->request['noSMS']) ? trans('visiosoft.module.site::message.notification.salutation') : trans('visiosoft.module.site::message.register_success')
            ], 200);

        } catch (\Exception $exception) {
            return response()->json(['status' => false, 'errors' => null, 'message' => trans('streams::error.500.name')], 400);
        }
    }

    /**
     * @param SiteRepositoryInterface $sites
     * @return \Illuminate\Http\JsonResponse
     */
    public function validateVerificationCode(
        SiteRepositoryInterface $sites
    )
    {
        $validator = Validator::make(request()->all(), [
            'verification_code' => 'required',
            'pid' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 400);
        }

        try {
            $inputs = request()->all();

            $site = $sites->newQuery()->where('api_token', $inputs['pid'])
                ->where('password', $inputs['verification_code'])
                ->first();

            if (!$site) {
                return $this->response->json(['success' => false, 'errors' => [trans('visiosoft.module.site::error.site_not_found')]], 404);
            }

            $user = $site->assign;

            dispatch(new UserActivatedBySms($user));

            return $this->response->json(['success' => true, 'message' => trans('visiosoft.module.site::message.user_verified')]);
        } catch (\Exception $e) {
            return $this->response->json(['success' => false, 'errors' => [trans('streams::error.500.name')]], 500);
        }
    }


}
