<?php namespace Visiosoft\SiteModule\Http\Controller;

use Anomaly\Streams\Platform\Field\Contract\FieldRepositoryInterface;
use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Anomaly\UsersModule\Role\Contract\RoleRepositoryInterface;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Anomaly\UsersModule\User\User;
use Anomaly\UsersModule\User\UserActivator;
use Carbon\Carbon;
use EmailChecker\Laravel\EmailCheckerFacade;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Visiosoft\AdvsModule\Adv\AdvRepository;
use Visiosoft\CustomfieldsModule\CustomField\Contract\CustomFieldRepositoryInterface;
use Visiosoft\CwpModule\Server\Contract\ServerRepositoryInterface;
use Visiosoft\CwpModule\Task\Contract\TaskRepositoryInterface;
use Visiosoft\SiteModule\Addon\Contract\AddonRepositoryInterface;
use Visiosoft\SiteModule\AddonTrial\Contract\AddonTrialRepositoryInterface;
use Visiosoft\SiteModule\Bundle\Contract\BundleRepositoryInterface;
use Visiosoft\SiteModule\Command\CheckSite;
use Visiosoft\SiteModule\Domain\Contract\DomainRepositoryInterface;
use Visiosoft\SiteModule\Domain\Event\CreatedDomain;
use Visiosoft\SiteModule\Domain\Event\CreateDomain;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SiteModule\Site\Event\AddAddonSite;
use Visiosoft\SiteModule\Site\Event\AddMultipleAddonSite;
use Visiosoft\SiteModule\Site\Event\BuildSite;
use Visiosoft\SiteModule\Site\Event\CheckActivatedSite;
use Visiosoft\SiteModule\Site\Event\CheckSubdomain;
use Visiosoft\SiteModule\Site\Event\CompletedInstallationSite;
use Visiosoft\SiteModule\Site\Event\DeleteTrial;
use Visiosoft\SiteModule\Site\Event\FailedSite;
use Visiosoft\SiteModule\Site\Event\FailedSiteSSL;
use Visiosoft\SiteModule\Site\Event\GuideForYesterdayOpenedSites;
use Visiosoft\SiteModule\Site\Handler\CheckCreateSiteForSubscription;
use Visiosoft\SubscriptionsModule\Subscription\Command\RegisteredUserBySubscription;

class SiteController extends ResourceController
{
    private $site;
    private $domain;
    public $activator;
    public $user;
    public $adv;
    public $cf;
    private $addonRepository;
    private $addonTrialRepository;
    private $roleRepository;
    private $taskRepository;
    private $bundleRepository;

    public function __construct(
        SiteRepositoryInterface $site,
        DomainRepositoryInterface $domain,
        UserRepositoryInterface $user,
        UserActivator $activator,
        AdvRepository $adv,
        CustomFieldRepositoryInterface $cf,
        AddonRepositoryInterface $addonRepository,
        AddonTrialRepositoryInterface $addonTrialRepository,
        RoleRepositoryInterface $roleRepository,
        TaskRepositoryInterface $taskRepository,
        BundleRepositoryInterface $bundleRepository
    )
    {
        $this->site = $site;
        $this->domain = $domain;
        $this->user = $user;
        $this->activator = $activator;
        $this->adv = $adv;
        $this->cf = $cf;
        parent::__construct();
        $this->addonRepository = $addonRepository;
        $this->addonTrialRepository = $addonTrialRepository;
        $this->roleRepository = $roleRepository;
        $this->taskRepository = $taskRepository;
        $this->bundleRepository = $bundleRepository;
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     * Activate Site CRON Service
     */
    public function activateSites()
    {
        $activated = array();
        $pending_sites = $this->site->newQuery()->whereIn('status', ['incomplete', 'pending'])->get();

        foreach ($pending_sites as $site) {
            $result = array_first(event(new CheckActivatedSite($site)));
            if ($result) {
                $site->changeStatus('completed');

                $activated[] = "$site->subdomain_name";

                event(new CompletedInstallationSite($site));

                if (!$site->app) {
                    $install_addons = [
//                        ["type" => 'module', "name" => 'demodata'],
                        ["type" => 'extension', "name" => 'ads_widget'],
                        ["type" => 'extension', "name" => 'ads_chart_widget'],
                    ];

                    if ($site->default_theme && $bundle = $this->bundleRepository->find($site->default_theme)) {
                        foreach ($bundle->addons as $addon) {
                            $addon_cf_name = $this->cf->getAdValueByCustomFieldSlug('addon_composer_name', $addon->id);
                            $addon_array = explode('-', $addon_cf_name);
                            $install_addons[] = ['type' => $addon_array[1], 'name' => $addon_array[0]];
                        }
                    } else {
                        $install_addons[] = ['type' => 'theme', 'name' => 'sahibinden'];
                    }

                    event(new AddMultipleAddonSite($site, $install_addons));
                }
            }
        }
        return response()->json($activated);
    }

    public function registerValidate()
    {
        $validated = $this->request->validate([
            'subdomain' => 'required|max:8',
            'domain' => 'required',
            'email' => 'required|email',
            'name_surname' => 'required',
        ]);


        $isUserByEmail = $this->findUserByColumn('email', $this->request->email);
        $subdomain = $this->editSubdomain($this->request->subdomain, false);
        $isSubdomain = $this->site->findBySubdomain($subdomain);

        $message = [];
        !is_null($isUserByEmail) && Auth::user()->email !== $this->request->email
            ? $message['email'] = trans('visiosoft.module.site::message.already_exists_email') : null;
        !is_null($isSubdomain) ? $message['storename'] = trans('visiosoft.module.site::message.already_exists_storename') : null;

        if (count($message)) {
            return $this->response->json(['status' => 'error', 'msg' => $message]);
        } else {
            return $this->response->json(['status' => 'success']);
        }
    }

    /**
     * @param UserRepositoryInterface $users
     * @return \Illuminate\Http\JsonResponse
     * Register For Ajax Service
     */
    public function registerUser(UserRepositoryInterface $users, FieldRepositoryInterface $fieldRepository)
    {
        $required_field = ['name_surname', 'email', 'storename'];

        //Check Required Fields
        foreach ($required_field as $field) {
            if (!isset($this->request->$field) or $this->request->$field == "") {
                return $this->responseJSON('error', $field . ' field is required!');
            }
        }
        $name_surname = explode(' ', $this->request->name_surname);

        //New User Parameters
        $rand = rand(1111, 9999);
        $userParameters = array();
        $email = explode('@', $this->request->email);
        $userParameters['email'] = $this->request->email;
        $userParameters['display_name'] = $this->request->name_surname;
        $userParameters['username'] = array_first($email);
        $userParameters['username'] = preg_replace("/[^a-z0-9]/", "", $userParameters['username']) . $rand;
        $userParameters['first_name'] = count($name_surname) > 0 ? array_first($name_surname) : null;
        $userParameters['last_name'] = count($name_surname) > 1 ? end($name_surname) : null;
        $userParameters['str_id'] = str_random(24);

        $password = rand(000000, 999999);

        $isLoggedUser = Auth::check() && $userParameters['email'] === Auth::user()->email;

        //Check User
        $checkUser = true;
        $isUserByEmail = $this->findUserByColumn('email', $userParameters['email']);
        $isUserByUsername = $this->findUserByColumn('username', $userParameters['username']);
        if (!$isLoggedUser && (!is_null($isUserByEmail) || !is_null($isUserByUsername))) {
            $checkUser = false;
        }

        //Check SubDomain
        $checkSubdomain = true;
        if (strlen($this->request->storename) > 8) {
            $checkSubdomain = false;
        } else {
            $subdomain = $this->editSubdomain($this->request->storename);
        }

        if (!$checkSubdomain || !$checkUser) {
            return $this->responseJSON('false', '', ['errorEmail' => $checkUser, 'errorStorename' => $checkSubdomain]);
        }

        if ($isLoggedUser) {
            $user = $this->user->findByEmail($userParameters['email']);
        } else {
            if ($this->isFakeEmail($userParameters['email'])) {
                return $this->responseJSON('error', trans('visiosoft.module.site::message.please_use_a_real_email'));
            }

            //Create New User
            $user = $this->user->create($userParameters);
            $user->setAttribute('password', $password);
            $user->setAttribute('activation_code', Str::random(40));
            $user->setAttribute('gsm_phone', $this->request->phone);
            $user->setAttribute('roles', $this->roleRepository->findBySlug('user'));

            //Save UTM parameters
            if ($fieldRepository->findBySlugAndNamespace('utm_source', 'users')) {
                $user->setAttribute('utm_source', (isset($_COOKIE['utm_source'])) ? $_COOKIE['utm_source'] : null);
                $user->setAttribute('utm_medium', (isset($_COOKIE['utm_medium'])) ? $_COOKIE['utm_medium'] : null);
                $user->setAttribute('utm_campaign', (isset($_COOKIE['utm_campaign'])) ? $_COOKIE['utm_campaign'] : null);
                $user->setAttribute('utm_term', (isset($_COOKIE['utm_term'])) ? $_COOKIE['utm_term'] : null);
                $user->setAttribute('utm_content', (isset($_COOKIE['utm_content'])) ? $_COOKIE['utm_content'] : null);
            }

            //Save browser language
            if ($fieldRepository->findBySlugAndNamespace('browser_lang', 'users') and isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
                $user->setAttribute('browser_lang', substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2));
            }

            //Save ip location
            if ($fieldRepository->findBySlugAndNamespace('location_for_ip', 'users')) {
                try {
                    $location = file_get_contents("https://ipinfo.io/json");
                    $location = json_decode($location, true);
                    $user->setAttribute('location_for_ip', $location['country']);
                } catch (\Exception $exception) {
                }
            }

            $users->save($user);

            $this->activator->send($user, route('visiosoft.module.site::installation.started'));
        }
        $default_domain = setting_value('visiosoft.module.site::default_domain');

        $locale_lang = $this->request->locale ?? 'en';

        $site = $this->site->createSite($subdomain, $user, $password, [
            'type' => $default_domain,
            'node' => setting_value('visiosoft.module.site::default_node_server'),
            'default_theme' => $this->request->theme,
            'locale' => $locale_lang
        ]);

        $status = array_first(event(new CheckSubdomain($site)));

        if ($status) {
            $site = $this->updateSubdomain($site);
        }

        if ($isLoggedUser) {
            $this->dispatch(new RegisteredUserBySubscription($user));
        }

        return $this->responseJSON('success', trans('visiosoft.module.site::message.register_success',
            ['email' => $userParameters['email'], 'password' => $password]), '', ['user_id' => $user->getId()]);
    }

    protected function isFakeEmail($email)
    {
        return !EmailCheckerFacade::isValid($email);
    }


    public function responseJSON($status, $message, $type = "", $params = [])
    {
        return response()->json(array_merge(['status' => $status, 'type' => $type, 'message' => $message], $params));
    }

    /**
     * @param $column
     * @param $value
     * @return mixed|null
     */
    public function findUserByColumn($column, $value)
    {
        if (!is_null($user = User::withTrashed()->where($column, $value)->get())) {
            return $user->first();
        }
        return null;
    }

    /**
     * @param $id
     * @return \Illuminate\Contracts\View\View|\Illuminate\Http\RedirectResponse|mixed
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     * Add Domain For Site
     */
    public function addDomain($id)
    {
        if (!is_null($site = $this->site->find($id))) {
            if (isset($this->request->domain)) {
                $domain = $this->site->editDomain($this->request->domain);

                $isDomain = $this->domain->findDomain($domain);
                if (is_null($isDomain)) {
                    $response = array_first(event(new CreateDomain($site, $domain)));

                    if ($response) {
                        $domain = $this->domain->create(['subdomain_id' => $id, 'domain' => $domain]);

                        event(new CreatedDomain($site, $domain));

                        $this->messages->success(trans('visiosoft.module.site::message.success_create_domain'));
                    } else {
                        $this->messages->error($response['message']);
                    }
                    return $this->redirect->to('/subscription/' . $site->subscription_id . "/detail");
                }
            }
            $this->template->set('meta_title', trans('visiosoft.module.site::field.add_domain'));
            return $this->view->make("visiosoft.module.site::addDomain", compact('site'));
        }
    }


    /**
     * @return \Illuminate\Contracts\View\View|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|mixed
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     * Create New Site
     */
    public function createNew()
    {
        $subscription_id = $this->request->subscription;

        if (!empty($subscription_id) and $this->checkCreateSite($this->request->subscription)) {

            $domains = setting_value('visiosoft.module.site::demo_domains');
            $domains = array_combine($domains, $domains);

            if (isset($this->request->storename) and isset($this->request->type)) {

                $store_name = $this->request->storename;
                $store_type = $this->request->type;

                //Check SubDomain
                $subdomain = $this->editSubdomain($store_name);

                $isSubdomain = $this->site->findBySubdomain($subdomain);

                //Check StoreType
                $checkStoreType = $this->checkStoreType($store_type);

                if (!$isSubdomain and $checkStoreType) {

                    //Create Site Request
                    $site = $this->site
                        ->createSite($subdomain, Auth::user(), rand(000000, 999999), ['type' => $store_type,
                            'subscription_id' => $subscription_id, 'node' => setting_value('visiosoft.module.site::default_node_server')]);

                    $response = array_first(event(new BuildSite($site)));

                    if ($response['status']) {
                        $site->changeStatus('pending');
                        $this->messages->success(trans('visiosoft.module.site::message.success_create_store'));
                    } else {
                        $this->messages->error($response['message']);
                    }
                    return redirect('/subscription/' . $subscription_id . '/detail');
                }

                $this->messages->error(trans('visiosoft.module.site::message.already_exists_storename'));
            }

            $this->template->set(
                'meta_title',
                trans('visiosoft.module.site::field.create_new_store_site') . " | $subscription_id"
            );

            return $this->view->make("visiosoft.module.site::createNew", compact('domains'));
        }
        $this->messages->error(trans('visiosoft.module.site::message.error_create_limit'));
        return $this->redirect->route('visiosoft.module.subscriptions::my-subscriptions');
    }

    /**
     * @param $subdomain
     * @return bool|string
     * Edit Subdomain
     */
    public function editSubdomain($subdomain, $checkDuplicates = true)
    {
        $subdomain = strtolower($subdomain);
        $subdomain = preg_replace("/[^a-z0-9]/", "", $subdomain);
        $subdomain = substr($subdomain, 0, 8);

        if ($checkDuplicates) {
            $server_repo = app(ServerRepositoryInterface::class);
            if (!$node = $server_repo->findBy('node_name', setting_value('visiosoft.module.site::default_node_server'))) {
                throw new \Exception(trans('visiosoft.module.cwp::message.server_not_found'));
            }

            $duplicatedDomain = 0;
            $initialSubdomain = $subdomain;
            do {
                if ($duplicatedDomain) {
                    if (strlen($subdomain) === 8) {
                        $subdomain = substr($subdomain, 0, -(strlen($duplicatedDomain))) . $duplicatedDomain;
                    } else {
                        $subdomain = $initialSubdomain . $duplicatedDomain;
                    }
                }

                $account_params = [
                    'action' => 'list',
                    'user' => $subdomain,
                ];

                $account = $this->taskRepository->newRequest('accountdetail', $node->getApiKey(), $node->getApiUrl(), $account_params);

                $duplicatedDomain++;
            } while (!$account->hasUserDoesntExistError());
        }

        return $subdomain;
    }

    /**
     * Cron Guide(Helper) Mail
     */
    public function cronGuide()
    {

        $day = setting_value('visiosoft.module.site::send_guide_after_days');

        $sites = $this->site->getUnguidedSitesWithDay($day);

        foreach ($sites as $site) {
            $send_status = event(new GuideForYesterdayOpenedSites($site));

            if (array_first($send_status)) {
                $site->update(['guide_mail' => true]);
            }
        }

    }

    /**
     * @param $subscription_id
     * @return mixed
     */

    public function checkCreateSite($subscription_id)
    {
        return $this->dispatch(new CheckCreateSiteForSubscription($subscription_id, null));
    }

    /**
     * @param $storetype
     * @return bool
     */
    public function checkStoreType($storetype)
    {
        return in_array($storetype, ['openclassify.com', 'revisio.app']);
    }

    public function installationStarted()
    {
        $this->template->set('meta_title', trans('module::message.installation-title'));
        $this->messages->flush();
        return $this->view->make('visiosoft.module.site::installation-site');
    }

    public function autoRefresh(CustomFieldRepositoryInterface $cf)
    {
        $sites = $this->site->getAutoRefresh();

        foreach ($sites as $site) {
            $site->reinstall();

            foreach ($site->reinstall_addons as $addon) {
                if ($is_addon = $cf->getAdValueByCustomFieldSlug('addon_composer_name', $addon->id)) {
                    event(new AddAddonSite($site, $is_addon));
                }
            }

        }
    }

    public function tryAddon($subdomainName, $addonNamespace, $redirectURL)
    {
        try {
            if (!$customField = $this->cf->findBySlug('addon_composer_name')) {
                throw new \Exception(trans('visiosoft.module.site::message.custom_field_not_found'));
            }

            if (!$classified = $this->adv->findByCFJSON("cf$customField->id", $addonNamespace)) {
                throw new \Exception(trans('visiosoft.module.site::message.classified_not_found'));
            }

            if (!$site = $this->site->findBySubdomain($subdomainName)) {
                throw new \Exception(trans('visiosoft.module.site::message.site_not_found'));
            }

            $installAddons = array();

            if ($isAddon = $this->cf->getAdValueByCustomFieldSlug('addon_composer_name', $classified->id)) {
                if (!$this->addonRepository->isAddonInstalled($isAddon, $site->getId())) {
                    $addonArray = explode('-', $isAddon);
                    $installAddons[] = ["type" => $addonArray[1], "name" => $addonArray[0]];

                    $triedBefore = $this->addonTrialRepository->getModel()
                        ->where([
                            'site_id' => $site->id,
                            'trialed_addon_id' => $classified->id,
                        ])
                        ->first();
                    if ($triedBefore) {
                        throw new \Exception(trans('visiosoft.module.site::message.you_tried_this_addon_before'));
                    }

                    $this->addonTrialRepository->create([
                        'site_id' => $site->id,
                        'trialed_addon_id' => $classified->id,
                        'expires_at' => Carbon::now()->addDays(setting_value('visiosoft.module.site::trial_period') ?: 1),
                    ]);
                }

                $response = [
                    'success' => true,
                    'msg' => trans('visiosoft.module.site::message.addAddon'),
                ];
            } else {
                $response = [
                    'success' => false,
                    'msg' => trans('visiosoft.module.site::message.error_addon_slug'),
                ];
            }

            if (count($installAddons)) {
                event(new AddMultipleAddonSite($site, $installAddons));
            }
        } catch (\Exception $e) {
            $response = [
                'success' => false,
                'msg' => $e->getMessage(),
            ];
        }

        $redirectURL = decrypt($redirectURL);
        $redirectURL = str_replace(':status:', $response['success'] ? 'success' : 'error', $redirectURL);
        $redirectURL = str_replace(':msg:', $response['msg'], $redirectURL);
        return redirect($redirectURL);
    }

    public function removeExpiredAddons()
    {
        try {
            $expiredTrials = $this->addonTrialRepository->getExpiredTrials();

            foreach ($expiredTrials as $trial) {
                event(new DeleteTrial($trial));

                $trial->remove();
            }

            return [
                'success' => true,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'msg' => $e->getMessage()
            ];
        }
    }

    public function getSitesByAuthUser()
    {
        return $this->site->getSitesByAssign(Auth::id());
    }

    public function addAddon()
    {
        $addon_cf_name = $this->cf->getAdValueByCustomFieldSlug('addon_composer_name', $this->request->addon);
        $site = $this->site->find($this->request->site);
        return event(new AddAddonSite($site, $addon_cf_name));
    }

    public function checkSiteStatus()
    {
        try {
            $sites = $this->site->getSitestoCheck();

            $error_sites = array();
            $error_ssl_sites = array();

            foreach ($sites as $site) {

                $domains = $this->domain->getDomainsBySite([$site->id]);

                // Verify for Domains
                foreach ($domains as $domain) {
                    $url = "https://".$domain->getDomain();

                    $response = $this->dispatch(new CheckSite($url));

                    $verify = $response['verify'];
                    $ssl = $response['verify_ssl'];

                    if (!$verify) {
                        // Send Error Site Notification
                        $error_sites[] = $url;
                    }

                    if ($verify && !$ssl) {
                        // Send Error SSL Verify Notification
                        $error_ssl_sites[] = $url;
                    }

                    $domain->update([
                        'verify' => $verify,
                        'verify_ssl' => $ssl,
                    ]);
                }

                // Verify for Site base URL
                $url = "https://".$site->getUrl();

                $response = $this->dispatch(new CheckSite($url));

                $verify = $response['verify'];
                $ssl = $response['verify_ssl'];

                if (!$verify) {
                    // Send Error Site Notification
                    $error_sites[] = $url;
                }

                if ($verify && !$ssl) {
                    // Send Error SSL Verify Notification
                    $error_ssl_sites[] = $url;
                }

                $site->update([
                    'check_at' => Carbon::now(),
                    'verify' => $verify,
                    'verify_ssl' => $ssl,
                ]);

            }

            if (count($error_ssl_sites)) {
                event(new FailedSiteSSL($error_ssl_sites));
            }

            if (count($error_sites)) {
                event(new FailedSite($error_sites));
            }

        } catch (\Exception $exception) {
            return $this->response->json(['error' => true, 'message' => $exception->getMessage()]);
        }
    }

    public function updateSubdomain($entry)
    {
        for ($i = 1; $i <= 20; $i++) {
            $entry->setAttribute('subdomain_name', $entry->subdomain_name . $i);
            $entry->save();

            $status = array_first(event(new CheckSubdomain($entry)));

            if (!$status) {
                break;
            }
        }

        return $entry;
    }
}
