<?php namespace Visiosoft\SiteModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Anomaly\UsersModule\User\User;
use Illuminate\Support\Facades\Auth;
use Visiosoft\SiteModule\Addon\Contract\AddonRepositoryInterface;
use Visiosoft\SiteModule\Domain\Contract\DomainRepositoryInterface;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SiteModule\Site\Event\ActivatedSite;
use Visiosoft\SiteModule\Site\Event\CheckActivatedSite;
use Visiosoft\SiteModule\Site\Notification\CompletedSiteCreate;
use Visiosoft\SiteModule\Site\SiteModel;

class SiteController extends ResourceController
{
    private $site;
    private $domain;
    private $addon;

    public function __construct(SiteRepositoryInterface $site, DomainRepositoryInterface $domain, AddonRepositoryInterface $addon)
    {
        $this->site = $site;
        $this->domain = $domain;
        $this->addon = $addon;
        parent::__construct();
    }

    public function index()
    {
        if (!Auth::user()) {
            redirect('/login?redirect=' . url()->current())->send();
        }
        $sites = $this->site->findByAssing(Auth::id());
        $domains = $this->domain->getDomainsBySite($sites);
        $addons = $this->addon->getAddonsBySite($sites, true);
        return $this->view->make('visiosoft.module.site::sites/list', compact('sites', 'domains', 'addons'));
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     * Activate Site CRON Service
     */
    public function activateSites()
    {
        $activated = array();
        $pending = $this->site->newQuery()->where('enabled', false)->get();
        foreach ($pending as $site) {
            $result = $this->events->dispatch(new CheckActivatedSite($site));
            if (!is_null($result) and isset($result[0]['result']) and $result[0]['result'] == "SUCCESS") {
                $site->update(['enabled' => true]);
                $activated[] = "$site->subdomain_name";
                event(new ActivatedSite($site));
                $site->assign->notify(new CompletedSiteCreate($site));
            }
        }
        return response()->json($activated);
    }

    public function checkReqiredFields(array $required_field)
    {
        foreach ($required_field as $field) {
            if (!isset($this->request->$field) or $this->request->$field == "") {
                return $this->responseJSON('error', $field . ' field is required!');
                die;
            }
        }
    }

    /**
     * @param UserRepositoryInterface $users
     * @return \Illuminate\Http\JsonResponse
     * Register For Ajax Service
     */
    public function registerUser(UserRepositoryInterface $users)
    {
        $required_field = ['name_surname', 'email', 'subdomain'];
        $this->checkReqiredFields($required_field);

        $this->checkSubdomain($this->request->subdomain);

        $rand = rand(1111, 9999);
        $userParameters = array();
        $email = explode('@', $this->request->email);
        $userParameters['email'] = $this->request->email;
        $userParameters['display_name'] = $this->request->name_surname;
        $userParameters['username'] = array_first($email) . $rand;
        $userParameters['username'] = preg_replace("/[^a-zA-Z0-9]/", "", $userParameters['username']);
        $userParameters['activated'] = 1;
        $userParameters['subdomain'] = $this->request->subdomain . $rand;
        $userParameters['str_id'] = str_random(24);

        $password = str_random(8);

        $isUserByEmail = $this->findUserByColumn('email', $userParameters['email']);
        $isUserByUsername = $this->findUserByColumn('username', $userParameters['username']);

        if (!is_null($isUserByEmail) or !is_null($isUserByUsername)) {
            return $this->responseJSON('error', 'This Email Registered! Please Login');
        }

        $user = User::query()->create($userParameters);
        $user->setAttribute('password', $password);
        $users->save($user);

        return $this->responseJSON('success', "Register Success");
    }

    public function checkSubdomain($subdomain)
    {
        $isSubdomain = $this->site->findBySubdomain($subdomain);

        if (!is_null($isSubdomain)) {
            return $this->responseJSON('error', 'This subdomain is already exists!');
        }
    }


    public function responseJSON($type, $message)
    {
        return response()->json(['status' => $type, 'message' => $message]);
    }

    /**
     * @param $column
     * @param $value
     * @return mixed|null
     */
    public function findUserByColumn($column, $value)
    {
        if (!is_null($user = User::withTrashed()->where($column, $value)->get())) {
            return $user->first();
        }
        return null;
    }
}
