<?php namespace Visiosoft\SiteModule\Http\Controller\Admin;

use Anomaly\Streams\Platform\Entry\Contract\EntryInterface;
use Visiosoft\SiteModule\Addon\Contract\AddonRepositoryInterface;
use Visiosoft\SiteModule\Domain\Contract\DomainRepositoryInterface;
use Visiosoft\SiteModule\Site\AddAddonForm\AddAddonFormBuilder;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SiteModule\Site\Event\BuildSite;
use Visiosoft\SiteModule\Site\Event\CheckActivatedSite;
use Visiosoft\SiteModule\Site\Event\CompletedInstallationSite;
use Visiosoft\SiteModule\Site\Event\CreateSiteOnManuel;
use Visiosoft\SiteModule\Site\Form\SiteFormBuilder;
use Visiosoft\SiteModule\Site\Table\SiteTableBuilder;
use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Visiosoft\SubscriptionsModule\Subscription\Contract\SubscriptionRepositoryInterface;

class SiteController extends AdminController
{

    private $site;
    private $domain;
    private $addon;

    public function __construct(SiteRepositoryInterface $site, DomainRepositoryInterface $domain, AddonRepositoryInterface $addon)
    {
        $this->site = $site;
        $this->domain = $domain;
        $this->addon = $addon;
        parent::__construct();
    }

    /**
     * Display an index of existing entries.
     *
     * @param SiteTableBuilder $table
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function index(SiteTableBuilder $table)
    {
        $table->setColumns([
            'subdomain_name' => [
                'value' => function (EntryInterface $entry) {
                    $url = $entry->subdomain_name . "." . $entry->type;
                    $link = '<a href="https://' . $url . '" target="_blank">' . $url . '</a>';

                    if (!is_null($entry->suspend_at)) {
                        $link = $link . " <i class='fa fa-eye-slash'></i>";
                    }
                    if (!is_null($entry->deletedCwp)) {
                        $link = $link . ' <span class="tag tag-sm tag-danger">' . trans('streams::button.delete') . '</span>';
                    }
                    return $link;
                },
            ],
            'assign',
            'created_at' => [
                'value' => function (EntryInterface $entry) {
                    return (is_null($entry->created_at)) ? "-" : $entry->created_at->format('d/m/Y H:i');
                }
            ],
            'update_at' => [
                'value' => function (EntryInterface $entry) {
                    return (is_null($entry->updated_at)) ? "-" : $entry->updated_at->format('d/m/Y H:i');
                }
            ],
            'domain' => [
                'value' => function (EntryInterface $entry, DomainRepositoryInterface $domain) {
                    $domains = $domain->getDomainsBySite([$entry->id])->pluck('domain')->all();
                    return implode(' , ', $domains);
                },
            ],
            'remaining_time' => [
                'value' => function (EntryInterface $entry, SubscriptionRepositoryInterface $subscription) {
                    if ($subscription_detail = $subscription->find($entry->subscription_id)) {
                        return $subscription->getRemaining($subscription_detail->id);
                    }
                },
            ],
        ]);
        return $table->render();
    }

    /**
     * Create a new entry.
     *
     * @param SiteFormBuilder $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function create(SiteFormBuilder $form)
    {
        return $form->render();
    }

    /**
     * @param AddAddonFormBuilder $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function addAddon(AddAddonFormBuilder $form)
    {
        return $form->render();
    }

    public function check()
    {
        $activated = array();
        $pending_sites = $this->site->getPendingSites();
        foreach ($pending_sites as $site) {
            $result = array_first(event(new CheckActivatedSite($site)));
            if ($result) {
                $site->changeStatus('completed');

                $activated[] = $site->subdomain_name;

                event(new CompletedInstallationSite($site));
            }
        }
        return $this->redirect->to('/admin/site');
    }

    public function completeOnManuel($id)
    {
        //Check Site
        if ($site = $this->site->find($id)
            and $site->status == "incomplete") {

            $response = array_first(event(new BuildSite($site)));

            if (!$response['status']) {
                $site->updateErrorMsg($response['message']);
                $this->messages->error([$response['message']]);
            } else {
                $site->changeStatus('pending');

                $response_subscription = array_first(event(new CreateSiteOnManuel($site)));

                $this->messages->success(trans('visiosoft.module.site::message.started_on_manual'));

                if ($response_subscription) {
                    $site->setSubscription($response_subscription->id);
                    $this->messages->success(trans('visiosoft.module.site::message.started_subscription', ['id' => $response_subscription->id]));
                } else {
                    $this->messages->error(trans('visiosoft.module.site::message.failed_start_subscription'));
                }
            }
        }
        return back();
    }

    public function edit(SiteFormBuilder $builder, $id)
    {
        return $builder->render($id);
    }
}
