<?php namespace Visiosoft\SiteModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Anomaly\UsersModule\User\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Visiosoft\SiteModule\Addon\Contract\AddonRepositoryInterface;
use Visiosoft\SiteModule\Domain\Contract\DomainRepositoryInterface;
use Visiosoft\SiteModule\Domain\Event\CreateDomain;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SiteModule\Site\Event\ActivatedSite;
use Visiosoft\SiteModule\Site\Event\CheckActivatedSite;
use Visiosoft\SiteModule\Site\Event\CreateSiteForUser;
use Visiosoft\SiteModule\Site\Event\GuideForYesterdayOpenedSites;
use Visiosoft\SiteModule\Site\Notification\CompletedSiteCreate;
use Visiosoft\SiteModule\Site\SiteModel;

class SiteController extends ResourceController
{
    private $site;
    private $domain;
    private $addon;

    public function __construct(SiteRepositoryInterface $site, DomainRepositoryInterface $domain, AddonRepositoryInterface $addon)
    {
        $this->site = $site;
        $this->domain = $domain;
        $this->addon = $addon;
        parent::__construct();
    }

    public function index()
    {
        if (!Auth::user()) {
            redirect('/login?redirect=' . url()->current())->send();
        }
        $sites = $this->site->findByAssing(Auth::id());
        $domains = $this->domain->getDomainsBySite($sites);
        $addons = $this->addon->getAddonsBySite($sites, true);
        return $this->view->make('visiosoft.module.site::sites/list', compact('sites', 'domains', 'addons'));
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     * Activate Site CRON Service
     */
    public function activateSites()
    {
        $activated = array();
        $pending = $this->site->newQuery()->where('status', '!=', 'completed')->get();
        foreach ($pending as $site) {
            $result = $this->events->dispatch(new CheckActivatedSite($site));
            if (!is_null($result) and isset($result[0]['result']) and $result[0]['result'] == "SUCCESS") {
                $site->update(['enabled' => true, 'status' => 'completed']);
                $activated[] = "$site->subdomain_name";
                event(new ActivatedSite($site));
            }
        }
        return response()->json($activated);
    }

    /**
     * @param UserRepositoryInterface $users
     * @return \Illuminate\Http\JsonResponse
     * Register For Ajax Service
     */
    public function registerUser(UserRepositoryInterface $users)
    {
        $required_field = ['name_surname', 'email', 'storename'];

        //Check Required Fields
        foreach ($required_field as $field) {
            if (!isset($this->request->$field) or $this->request->$field == "") {
                return $this->responseJSON('error', $field . ' field is required!');
                die;
            }
        }

        //New User Parameters
        $rand = rand(1111, 9999);
        $userParameters = array();
        $email = explode('@', $this->request->email);
        $userParameters['email'] = $this->request->email;
        $userParameters['display_name'] = $this->request->name_surname;
        $userParameters['username'] = array_first($email);
        $userParameters['username'] = preg_replace("/[^a-z0-9]/", "", $userParameters['username']) . $rand;
        $userParameters['activated'] = 1;
        $userParameters['str_id'] = str_random(24);

        $password = rand(000000, 999999);

        //Check User
        $checkUser = true;
        $isUserByEmail = $this->findUserByColumn('email', $userParameters['email']);
        $isUserByUsername = $this->findUserByColumn('username', $userParameters['username']);
        if (!is_null($isUserByEmail) or !is_null($isUserByUsername)) {
            $checkUser = false;
        }

        //Check SubDomain
        $checkSubdomain = true;
        if (strlen($this->request->storename) > 8) {
            $checkSubdomain = false;
        } else {
            $subdomain = $this->editSubdomain($this->request->storename);
            $isSubdomain = $this->site->findBySubdomain($subdomain);
            if (!is_null($isSubdomain)) {
                $checkSubdomain = false;
            }
        }


        if (!$checkSubdomain || !$checkUser) {
            return $this->responseJSON('false', '', ['errorEmail' => $checkUser, 'errorStorename' => $checkSubdomain]);
        }

        //Create New User
        $user = User::query()->create($userParameters);
        $user->setAttribute('password', $password);
        $users->save($user);

        //Create Site Request
        $site = $this->site->create([
            'subdomain_name' => $subdomain,
            'assign' => $user,
            'password' => $password,
            'status' => 'incomplete'
        ]);

        event(new CreateSiteForUser($user, $site));

        return $this->responseJSON('success', trans('visiosoft.module.site::message.register_success',
            ['email' => $userParameters['email'], 'password' => $password]));
    }


    public function responseJSON($status, $message, $type = "")
    {
        return response()->json(['status' => $status, 'type' => $type, 'message' => $message]);
    }

    /**
     * @param $column
     * @param $value
     * @return mixed|null
     */
    public function findUserByColumn($column, $value)
    {
        if (!is_null($user = User::withTrashed()->where($column, $value)->get())) {
            return $user->first();
        }
        return null;
    }

    public function addDomain($id)
    {
        if (!is_null($site = $this->site->find($id))) {
            if (isset($this->request->domain)) {
                $domain = $this->site->editDomain($this->request->domain);

                $isDomain = $this->domain->findDomain($domain);
                if (is_null($isDomain)) {
                    $newSite = event(new CreateDomain($site, $domain));
                    if (array_first($newSite) === true) {
                        $this->domain->create([
                            'subdomain_id' => $id,
                            'domain' => $domain,
                        ]);
                        $this->messages->success(["Domain has been successfully added."]);
                    }
                    return redirect('/profile/mysites');
                }
            }
            return $this->view->make("visiosoft.module.site::addDomain", compact('site'));
        }
    }

    public function createNew()
    {
        if (isset($this->request->storename) and isset($this->request->type)) {
            $store_name = $this->request->storename;
            $store_type = $this->request->type;

            //Check SubDomain
            $subdomain = $this->editSubdomain($store_name);
            $isSubdomain = $this->site->findBySubdomain($subdomain);
            if (!is_null($isSubdomain)) {
                $this->messages->error(trans('visiosoft.module.site::message.already_exists_storename'));
                return $this->view->make("visiosoft.module.site::createNew");
            }

            $password = rand(000000, 999999);

            //Create Site Request
            $this->site->create([
                'subdomain_name' => $subdomain,
                'assign' => Auth::user(),
                'password' => $password,
                'status' => 'incomplete',
                'type' => $store_type
            ]);
            $this->messages->success(['Request to create a store has been received. When the installation is completed, information will be sent by e-mail.']);
            return redirect('/profile/mysites');

        }
        return $this->view->make("visiosoft.module.site::createNew");
    }

    public function editSubdomain($subdomain)
    {
        $subdomain = strtolower($subdomain);
        return preg_replace("/[^a-z0-9]/", "", $subdomain);
    }

    public function cronGuide()
    {

        $day = setting_value('visiosoft.module.site::send_guide_after_days');
        $sites = $this->site->newQuery()
            ->where('status', 'completed')
            ->whereDate('created_at', date('Y-m-d', strtotime("-" . $day . " days")))
            ->where('guide_mail', false)
            ->get();

        foreach ($sites as $site) {
            $send_status = event(new GuideForYesterdayOpenedSites($site));

            if (array_first($send_status)) {
                $site->update(['guide_mail' => true]);
            }
        }

    }
}
