<?php namespace Visiosoft\SiteModule\Http\Controller\Admin;

use Anomaly\Streams\Platform\Entry\Contract\EntryInterface;
use Visiosoft\SiteModule\Addon\Contract\AddonRepositoryInterface;
use Visiosoft\SiteModule\Domain\Contract\DomainRepositoryInterface;
use Visiosoft\SiteModule\Site\AddAddonForm\AddAddonFormBuilder;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SiteModule\Site\Event\ActivatedSite;
use Visiosoft\SiteModule\Site\Event\CheckActivatedSite;
use Visiosoft\SiteModule\Site\Form\SiteFormBuilder;
use Visiosoft\SiteModule\Site\Notification\CompletedSiteCreate;
use Visiosoft\SiteModule\Site\Table\SiteTableBuilder;
use Anomaly\Streams\Platform\Http\Controller\AdminController;

class SiteController extends AdminController
{

    private $site;
    private $domain;
    private $addon;

    public function __construct(SiteRepositoryInterface $site, DomainRepositoryInterface $domain, AddonRepositoryInterface $addon)
    {
        $this->site = $site;
        $this->domain = $domain;
        $this->addon = $addon;
        parent::__construct();
    }

    /**
     * Display an index of existing entries.
     *
     * @param SiteTableBuilder $table
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function index(SiteTableBuilder $table)
    {
        $table->setColumns([
            'subdomain_name' => [
                'value' => function (EntryInterface $entry) {
                    $url = $entry->subdomain_name . "." . $entry->type;
                    $link = '<a href="https://' . $url . '" target="_blank">' . $url . '</a>';

                    if (!is_null($entry->suspend_at)) {
                        $link = $link . " <i class='fa fa-eye-slash'></i>";
                    }
                    if (!is_null($entry->deletedCwp)) {
                        $link = $link . ' <span class="tag tag-sm tag-danger">' . trans('streams::button.delete') . '</span>';
                    }
                    return $link;
                },
            ],
            'assign',
            'created_at' => [
                'value' => function (EntryInterface $entry) {
                    return (is_null($entry->created_at)) ? "-" : $entry->created_at->format('d/m/Y H:i');
                }
            ],
            'update_at' => [
                'value' => function (EntryInterface $entry) {
                    return (is_null($entry->updated_at)) ? "-" : $entry->updated_at->format('d/m/Y H:i');
                }
            ],
            'domain' => [
                'value' => function (EntryInterface $entry, DomainRepositoryInterface $domain) {
                    $domains = $domain->getDomainsBySite([$entry->id])->pluck('domain')->all();
                    return implode(' , ', $domains);
                },
            ],
        ]);
        $table->addButtons([
            'showAddons' => [
                'type' => 'primary',
                'href' => '/admin/site/addon?limit=15&view=all&page=1&filter_subdomain={entry.id}'
            ],
            'addAddon' => [
                'text' => 'visiosoft.module.site::button.addAddon',
                'href' => route('site::addAddon') . '?site={entry.id}',
                'icon' => 'fa fa-plus',
                'type' => 'info'
            ],
            'status' => [
                'attributes' => [
                    'data-toggle' => 'tooltip',
                    'data-placement' => 'top',
                    'title' => function (EntryInterface $entry) {
                        return ($entry->status == "error") ? $entry->message : '';
                    },
                ],
                'text' => function (EntryInterface $entry) {
                    if ($entry->status == "incomplete") {
                        return "visiosoft.module.site::button.install_site";
                    } elseif ($entry->status == "completed") {
                        return "visiosoft.module.site::button.active";
                    } elseif ($entry->status == "pending") {
                        return "visiosoft.module.site::button.pending";
                    } else {
                        return "visiosoft.module.site::button.error";
                    }
                },
                'href' => function (EntryInterface $entry) {
                    if ($entry->status == "incomplete") {
                        return '/admin/subscriptions/completed/' . $entry->id;
                    } else
                        return "#";

                },
                'type' => function (EntryInterface $entry) {
                    if ($entry->status == "incomplete") {
                        return "warning";
                    } elseif ($entry->status == "completed") {
                        return "success";
                    } elseif ($entry->status == "pending") {
                        return "default";
                    } else {
                        return "warning";
                    }
                },
                'icon' => function (EntryInterface $entry) {
                    if ($entry->status == "incomplete") {
                        return "fa fa-credit-card-alt";
                    } elseif ($entry->status == "completed") {
                        return "fa fa-check";
                    } elseif ($entry->status == "pending") {
                        return "fa fa-clock-o";
                    } else {
                        return "fa fa-times";
                    }
                },],]);
        return $table->render();
    }

    /**
     * Create a new entry.
     *
     * @param SiteFormBuilder $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function create(SiteFormBuilder $form)
    {
        return $form->render();
    }

    /**
     * @param AddAddonFormBuilder $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function addAddon(AddAddonFormBuilder $form)
    {
        return $form->render();
    }

    public function check()
    {
        $activated = array();
        $pending = $this->site->newQuery()->where('status', '!=', 'completed')->get();
        foreach ($pending as $site) {
            $result = $this->events->dispatch(new CheckActivatedSite($site));
            if (!is_null($result) and isset($result[0]['result']) and $result[0]['result'] == "SUCCESS") {
                $site->update(['enabled' => true, 'status' => 'completed']);
                $activated[] = "$site->subdomain_name";
                event(new ActivatedSite($site));
            }
        }
        return $this->redirect->to('/admin/site');
    }
}
