<?php namespace Visiosoft\SiteModule\Domain;

use Carbon\Carbon;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\SiteModule\Domain\Event\CreatedDomain;
use Visiosoft\SiteModule\Domain\Event\CreateDomain;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SiteModule\Site\Event\InstallSSL;

class DomainApiCollection extends DomainRepository
{
    use DispatchesJobs;

    public function add(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['subdomain', 'token', 'domain'], $params));

        if (!$this->is_valid_domain_name($params['domain'])) {
            throw new \Exception(trans('visiosoft.module.site::message.error_domain_format'));
        }

        $site = $this->checkSite($params['token'], $params['subdomain']);

        $response = array_first(event(new CreateDomain($site, $params['domain'])));

        if ($response['status']) {
            $domain = $this->newQuery()
                ->create([
                    'created_at' => Carbon::now(),
                    'created_by_id' => $site->created_by_id,
                    'subdomain_id' => $site->id,
                    'domain' => $params['domain']
                ]);


            event(new CreatedDomain($site, $domain));

            return $domain;

        } else {
            throw new \Exception($response['message'], 500);
        }
    }

    public function is_valid_domain_name($domain_name)
    {
        return (preg_match("/^([a-z\d](-*[a-z\d])*)(\.([a-z\d](-*[a-z\d])*))*$/i", $domain_name) //valid chars check
            && preg_match("/^.{1,253}$/", $domain_name) //overall length check
            && preg_match("/^[^\.]{1,63}(\.[^\.]{1,63})*$/", $domain_name)); //length of each label
    }

    public function checkSite($token, $subdomain)
    {
        $site_repository = app(SiteRepositoryInterface::class);

        $site = $site_repository->newQuery()
            ->where('subdomain_name', $subdomain)
            ->where('api_token', $token)
            ->first();

        if (!$site) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Site']), 404);
        }

        return $site;
    }

    public function list(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['subdomain', 'token'], $params));

        $site = $this->checkSite($params['token'], $params['subdomain']);

        return $this->newQuery()
            ->where('subdomain_id', $site->id);
    }

    public function addSSL(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['subdomain', 'token', 'domain'], $params));

        $site = $this->checkSite($params['token'], $params['subdomain']);

        $domain = $this->newQuery()
            ->where('subdomain_id', $site->id)
            ->where('domain', $params['domain'])
            ->first();

        if (!$domain) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Domain']), 404);
        }

        $response = event(new InstallSSL($site, $domain));

        if (!array_first($response))
        {
            throw new \Exception(trans('visiosoft.module.site::message.error_adding_ssl'), 500);
        }

        return collect(['message' => trans('streams::message.edit_success', ['name' => $params['domain']])]);
    }
}
