<?php namespace Visiosoft\SiteModule\Site;

use Carbon\Carbon;
use Illuminate\Support\Str;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Visiosoft\SiteModule\Site\Event\SuspendSite;

class SiteRepository extends EntryRepository implements SiteRepositoryInterface
{
    protected $model;

    public function __construct(SiteModel $model)
    {
        $this->model = $model;
    }

    public function findBySubdomain($subdomain)
    {
        if ($sites = $this->model->withTrashed()->where('subdomain_name', $subdomain)->get()) {
            return $sites->first();
        }
        return null;
    }


    public function unsuspend($id)
    {
        if (!is_null($site = $this->newQuery()->withTrashed()->where('suspend_at', '!=', null)->find($id))) {
            $site->update
            ([
                'suspend_at' => null,
                'deleted_at' => null,
            ]);
            event(new SuspendSite($site, "false"));
        }
    }

    public function getSitesByAssign($user_id, $type = null)
    {
        $query = $this->newQuery()
            ->where('assign_id', $user_id);
        if ($type) {
            $query = $query->where('status', $type);
        }
        return $query->orderByDesc('id')->get();
    }

    public function editDomain($domain)
    {
        $domain = str_replace('http://', '', $domain);
        $domain = str_replace('https://', '', $domain);
        $domain = str_replace('www.', '', $domain);

        return $domain;
    }

    public function getSiteForSubscription($id)
    {
        return $this->newQuery()->where('subscription_id', $id)->get();
    }

    public function createSite($subdomain, $user, $password, $params = [])
    {
        if (setting_value('visiosoft.module.site::sub_installation')) {
            $params = array_merge($params, [
                'app' => setting_value('visiosoft.module.site::default_app'),
            ]);

            if ($app = $this->find(setting_value('visiosoft.module.site::default_app'))) {
                $params['type'] = $app->type;
            }
        }

        return $this->create(array_merge([
            'subdomain_name' => $subdomain,
            'assign' => $user,
            'locale' => ($user->browser_lang) ? $user->browser_lang : 'en',
            'password' => $password,
            'status' => 'incomplete',
            'api_token' => Str::random(24)
        ], $params));
    }

    public function getUnguidedSitesWithDay($day)
    {
        return $this->newQuery()
            ->where('status', 'completed')
            ->whereDate('created_at', date('Y-m-d', strtotime("-" . $day . " days")))
            ->where('guide_mail', false)
            ->get();
    }

    public function getPendingSites()
    {
        return $this->newQuery()
            ->whereRaw('DATE_ADD(created_at, INTERVAL 60 SECOND) < NOW()')
            ->whereIn('status', ['incomplete', 'pending'])
            ->get();
    }

    public function findUnsuspend($id)
    {
        return $this->newQuery()->where('suspend_at', null)->find($id);
    }

    public function findSuspend($id)
    {
        return $this->model->withTrashed()->whereNotNull('suspend_at')->find($id);
    }

    public function findTerminate($id)
    {
        return $this->model->withTrashed()->whereNull('deletedCwp')->find($id);
    }

    public function getAutoRefresh()
    {
        return $this->newQuery()->where('auto_refresh', true)->get();
    }

    public function getSitestoCheck()
    {
        $minute = (int)setting_value('visiosoft.module.site::verify_period', 1440); // one day
        $from = Carbon::now()->subMinutes($minute)->format('Y-m-d H:i:s');

        return $this->newQuery()
            ->where(function ($query) use ($from) {
                $query->where('check_at', null)
                    ->orWhereDate('check_at', '<', $from);
            })
            ->where('suspend_at', null)
            ->where('status', 'completed')
            ->get();
    }

    public function findByApiToken($token)
    {
        return $this->newQuery()
            ->where('api_token', $token)->first();
    }
}
