<?php namespace Visiosoft\SiteModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Anomaly\UsersModule\User\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Visiosoft\SiteModule\Addon\Contract\AddonRepositoryInterface;
use Visiosoft\SiteModule\Domain\Contract\DomainRepositoryInterface;
use Visiosoft\SiteModule\Domain\Event\CreatedDomain;
use Visiosoft\SiteModule\Domain\Event\CreateDomain;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SiteModule\Site\Event\ActivatedSite;
use Visiosoft\SiteModule\Site\Event\BuildSite;
use Visiosoft\SiteModule\Site\Event\CheckActivatedSite;
use Visiosoft\SiteModule\Site\Event\CompletedInstallationSite;
use Visiosoft\SiteModule\Site\Event\CreatedUserForDemo;
use Visiosoft\SiteModule\Site\Event\CreateSiteForUser;
use Visiosoft\SiteModule\Site\Event\GuideForYesterdayOpenedSites;
use Visiosoft\SiteModule\Site\Handler\CheckCreateSiteForSubscription;
use Visiosoft\SiteModule\Site\Notification\CompletedSiteCreate;
use Visiosoft\SiteModule\Site\SiteModel;
use Visiosoft\SubscriptionsModule\Subscription\SubscriptionRepository;

class SiteController extends ResourceController
{
    private $site;
    private $domain;
    private $addon;
    private $subscription;

    public function __construct(SiteRepositoryInterface $site, DomainRepositoryInterface $domain, AddonRepositoryInterface $addon, SubscriptionRepository $subscription)
    {
        $this->site = $site;
        $this->subscription = $subscription;
        $this->domain = $domain;
        $this->addon = $addon;
        parent::__construct();
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     * Activate Site CRON Service
     */
    public function activateSites()
    {
        $activated = array();
        $pending_sites = $this->site->newQuery()->whereIn('status', ['incomplete', 'pending'])->get();

        foreach ($pending_sites as $site) {
            $result = array_first(event(new CheckActivatedSite($site)));
            if ($result) {
                $site->changeStatus('completed');

                $activated[] = "$site->subdomain_name";

                event(new CompletedInstallationSite($site));
            }
        }
        return response()->json($activated);
    }

    /**
     * @param UserRepositoryInterface $users
     * @return \Illuminate\Http\JsonResponse
     * Register For Ajax Service
     */
    public function registerUser(UserRepositoryInterface $users)
    {
        $required_field = ['name_surname', 'email', 'storename'];

        //Check Required Fields
        foreach ($required_field as $field) {
            if (!isset($this->request->$field) or $this->request->$field == "") {
                return $this->responseJSON('error', $field . ' field is required!');
                die;
            }
        }

        //New User Parameters
        $rand = rand(1111, 9999);
        $userParameters = array();
        $email = explode('@', $this->request->email);
        $userParameters['email'] = $this->request->email;
        $userParameters['display_name'] = $this->request->name_surname;
        $userParameters['username'] = array_first($email);
        $userParameters['username'] = preg_replace("/[^a-z0-9]/", "", $userParameters['username']) . $rand;
        $userParameters['activated'] = 1;
        $userParameters['str_id'] = str_random(24);

        $password = rand(000000, 999999);

        //Check User
        $checkUser = true;
        $isUserByEmail = $this->findUserByColumn('email', $userParameters['email']);
        $isUserByUsername = $this->findUserByColumn('username', $userParameters['username']);
        if (!is_null($isUserByEmail) or !is_null($isUserByUsername)) {
            $checkUser = false;
        }

        //Check SubDomain
        $checkSubdomain = true;
        if (strlen($this->request->storename) > 8) {
            $checkSubdomain = false;
        } else {
            $subdomain = $this->editSubdomain($this->request->storename);
            $isSubdomain = $this->site->findBySubdomain($subdomain);
            if (!is_null($isSubdomain)) {
                $checkSubdomain = false;
            }
        }


        if (!$checkSubdomain || !$checkUser) {
            return $this->responseJSON('false', '', ['errorEmail' => $checkUser, 'errorStorename' => $checkSubdomain]);
        }

        //Create New User
        $user = User::query()->create($userParameters);
        $user->setAttribute('password', $password);
        $users->save($user);

        //Create Site Request
        $this->site->createSite($subdomain, $user, $password);

        event(new CreatedUserForDemo($user));

        return $this->responseJSON('success', trans('visiosoft.module.site::message.register_success',
            ['email' => $userParameters['email'], 'password' => $password]));
    }


    public function responseJSON($status, $message, $type = "")
    {
        return response()->json(['status' => $status, 'type' => $type, 'message' => $message]);
    }

    /**
     * @param $column
     * @param $value
     * @return mixed|null
     */
    public function findUserByColumn($column, $value)
    {
        if (!is_null($user = User::withTrashed()->where($column, $value)->get())) {
            return $user->first();
        }
        return null;
    }

    /**
     * @param $id
     * @return \Illuminate\Contracts\View\View|\Illuminate\Http\RedirectResponse|mixed
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     * Add Domain For Site
     */
    public function addDomain($id)
    {
        if (!is_null($site = $this->site->find($id))) {
            if (isset($this->request->domain)) {
                $domain = $this->site->editDomain($this->request->domain);

                $isDomain = $this->domain->findDomain($domain);
                if (is_null($isDomain)) {
                    $response = array_first(event(new CreateDomain($site, $domain)));

                    if ($response) {
                        $domain = $this->domain->create(['subdomain_id' => $id, 'domain' => $domain]);

                        event(new CreatedDomain($site, $domain));

                        $this->messages->success(trans('visiosoft.module.site::message.success_create_domain'));
                    } else {
                        $this->messages->error($response['message']);
                    }
                    return $this->redirect->to('/subscription/' . $site->subscription_id . "/detail");
                }
            }
            return $this->view->make("visiosoft.module.site::addDomain", compact('site'));
        }
    }


    /**
     * @return \Illuminate\Contracts\View\View|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|mixed
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     * Create New Site
     */
    public function createNew()
    {
        if (!empty($this->request->subscription) and $this->checkCreateSite($this->request->subscription)) {
            if (isset($this->request->storename) and isset($this->request->type)) {

                $subscription_id = $this->request->subscription;
                $store_name = $this->request->storename;
                $store_type = $this->request->type;

                //Check SubDomain
                $subdomain = $this->editSubdomain($store_name);

                $isSubdomain = $this->site->findBySubdomain($subdomain);

                //Check StoreType
                $checkStoreType = $this->checkStoreType($store_type);

                if (!$isSubdomain and $checkStoreType) {

                    //Create Site Request
                    $site = $this->site
                        ->createSite($subdomain, Auth::user(), rand(000000, 999999), ['type' => $store_type,
                        'subscription_id' => $subscription_id]);

                    $response = array_first(event(new BuildSite($site)));

                    if ($response['status']) {
                        $site->changeStatus('pending');
                        $this->messages->success(trans('visiosoft.module.site::message.success_create_store'));
                    } else {
                        $this->messages->error($response['message']);
                    }
                    return redirect('/subscription/' . $subscription_id . '/detail');

                }
                $this->messages->error(trans('visiosoft.module.site::message.already_exists_storename'));
                return $this->view->make("visiosoft.module.site::createNew");
            }
            return $this->view->make("visiosoft.module.site::createNew");
        }
        $this->messages->error(trans('visiosoft.module.site::message.error_create_limit'));
        return $this->redirect->to('/mySubscriptions');
    }

    /**
     * @param $subdomain
     * @return bool|string
     * Edit Subdomain
     */
    public function editSubdomain($subdomain)
    {
        $subdomain = strtolower($subdomain);
        $subdomain = preg_replace("/[^a-z0-9]/", "", $subdomain);
        return substr($subdomain, 0, 8);
    }

    /**
     * Cron Guide(Helper) Mail
     */
    public function cronGuide()
    {

        $day = setting_value('visiosoft.module.site::send_guide_after_days');

        $sites = $this->site->getUnguidedSitesWithDay($day);

        foreach ($sites as $site) {
            $send_status = event(new GuideForYesterdayOpenedSites($site));

            if (array_first($send_status)) {
                $site->update(['guide_mail' => true]);
            }
        }

    }

    /**
     * @param $subscription_id
     * @return mixed
     */

    public function checkCreateSite($subscription_id)
    {
        return $this->dispatch(new CheckCreateSiteForSubscription($subscription_id, null));
    }

    /**
     * @param $storetype
     * @return bool
     */
    public function checkStoreType($storetype)
    {
        return in_array($storetype, ['openclassify.com', 'revisio.app']);
    }
}
