<?php namespace Visiosoft\SiteModule\Http\Controller;

use Anomaly\Streams\Platform\Field\Contract\FieldRepositoryInterface;
use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Anomaly\UsersModule\Role\Contract\RoleRepositoryInterface;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Anomaly\UsersModule\User\UserActivator;
use Anomaly\UsersModule\User\UserModel;
use Carbon\Carbon;
use EmailChecker\Laravel\EmailCheckerFacade;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Visiosoft\AdvsModule\Adv\AdvRepository;
use Visiosoft\AdvsModule\Traits\ApiReturnResponseTrait;
use Visiosoft\CustomfieldsModule\CustomField\Contract\CustomFieldRepositoryInterface;
use Visiosoft\SiteModule\Addon\Contract\AddonRepositoryInterface;
use Visiosoft\SiteModule\AddonTrial\Contract\AddonTrialRepositoryInterface;
use Visiosoft\SiteModule\Bundle\Contract\BundleRepositoryInterface;
use Visiosoft\SiteModule\Command\CheckSite;
use Visiosoft\SiteModule\Domain\Contract\DomainRepositoryInterface;
use Visiosoft\SiteModule\Domain\Event\CreatedDomain;
use Visiosoft\SiteModule\Domain\Event\CreateDomain;
use Visiosoft\SiteModule\Site\Contract\SiteRepositoryInterface;
use Visiosoft\SiteModule\Site\Event\AddAddonSite;
use Visiosoft\SiteModule\Site\Event\AddMultipleAddonSite;
use Visiosoft\SiteModule\Site\Event\BuildSite;
use Visiosoft\SiteModule\Site\Event\DeleteAddonSite;
use Visiosoft\SiteModule\Site\Event\DeleteTrial;
use Visiosoft\SiteModule\Site\Event\FailedSite;
use Visiosoft\SiteModule\Site\Event\FailedSiteSSL;
use Visiosoft\SiteModule\Site\Event\GuideForYesterdayOpenedSites;
use Visiosoft\SiteModule\Site\Handler\CheckCreateSiteForSubscription;
use Visiosoft\SiteModule\Site\Jobs\UserActivatedBySms;
use Visiosoft\SmsModule\Commands\SendSms;
use Illuminate\Support\Facades\Validator;
use Visiosoft\SmsModule\Template\Contract\TemplateRepositoryInterface;
use Visiosoft\SpamDetectionExtension\Helper\SpamDetector;
use Visiosoft\SubscriptionsModule\Subscription\Command\RegisteredUserBySubscription;
use Visiosoft\SubscriptionsModule\Subscription\Contract\SubscriptionRepositoryInterface;

class SiteController extends ResourceController
{
    use ApiReturnResponseTrait;


    public $activator,
        $user,
        $adv,
        $cf;

    private $addonRepository,
        $addonTrialRepository,
        $roleRepository,
        $bundleRepository,
        $smsTemplateRepository,
        $subscriptionRepository,
        $site,
        $domain;

    public function __construct(
        SiteRepositoryInterface $site,
        DomainRepositoryInterface $domain,
        UserRepositoryInterface $user,
        UserActivator $activator,
        AdvRepository $adv,
        CustomFieldRepositoryInterface $cf,
        AddonRepositoryInterface $addonRepository,
        AddonTrialRepositoryInterface $addonTrialRepository,
        RoleRepositoryInterface $roleRepository,

        BundleRepositoryInterface $bundleRepository,
        TemplateRepositoryInterface $smsTemplateRepository,
        SubscriptionRepositoryInterface $subscriptionRepository
    )
    {
        $this->site = $site;
        $this->domain = $domain;
        $this->user = $user;
        $this->activator = $activator;
        $this->adv = $adv;
        $this->cf = $cf;
        $this->addonRepository = $addonRepository;
        $this->addonTrialRepository = $addonTrialRepository;
        $this->roleRepository = $roleRepository;

        $this->bundleRepository = $bundleRepository;
        $this->smsTemplateRepository = $smsTemplateRepository;
        $this->subscriptionRepository = $subscriptionRepository;
        parent::__construct();
    }

    public function sendVerifySms($phone, $code)
    {
        $smsTemplate = $this->smsTemplateRepository->findBySlug('otp');
        if (empty($smsTemplate)) {
            throw new \Exception('sms template not found');
        }
        $message = str_replace('{code}', $code, $smsTemplate->message);
        $this->dispatch(new SendSms($phone, $message));
    }

//    public function reSendVerifySms($id)
//    {
//        try {
//            $site = $this->site->findByApiToken($id);
//
//            if (!$site || $site->status != 'incomplete') {
//                throw new \Exception('Account not found');
//            }
//
//            $user = $site->assign;
//
//            $this->sendVerifySms($user->gsm_phone, $site->password);
//
//            return response()->json(['status' => true], 200);
//        } catch (\Exception $exception) {
//            return response()->json(['status' => false, 'errors' => null, 'message' => $exception->getMessage()], 400);
//        }
//
//
//    }

    protected function isFakeEmail($email)
    {
        return !EmailCheckerFacade::isValid($email);
    }


    /**
     * @param $column
     * @param $value
     * @return mixed|null
     */
    public function findUserByColumn($column, $value)
    {
        if (!is_null($user = UserModel::withTrashed()->where($column, $value)->get())) {
            return $user->first();
        }
        return null;
    }

    /**
     * @param $id
     * @return \Illuminate\Contracts\View\View|\Illuminate\Http\RedirectResponse|mixed
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     * Add Domain For Site
     */
    public function addDomain($id)
    {
        if (!is_null($site = $this->site->find($id))) {
            if (isset($this->request->domain)) {
                $domain = $this->site->editDomain($this->request->domain);

                $isDomain = $this->domain->findDomain($domain);
                if (is_null($isDomain)) {
                    $response = array_first(event(new CreateDomain($site, $domain)));

                    if ($response) {
                        $domain = $this->domain->create(['subdomain_id' => $id, 'domain' => $domain]);

                        event(new CreatedDomain($site, $domain));

                        $this->messages->success(trans('visiosoft.module.site::message.success_create_domain'));
                    } else {
                        $this->messages->error($response['message']);
                    }
                    return $this->redirect->to('/subscription/' . $site->subscription_id . "/detail");
                }
            }
            $this->template->set('meta_title', trans('visiosoft.module.site::field.add_domain'));
            return $this->view->make("visiosoft.module.site::addDomain", compact('site'));
        }
    }

    /**
     * Cron Guide(Helper) Mail
     */
    public function cronGuide()
    {

        $day = setting_value('visiosoft.module.site::send_guide_after_days');

        $sites = $this->site->getUnguidedSitesWithDay($day);

        foreach ($sites as $site) {
            $send_status = event(new GuideForYesterdayOpenedSites($site));

            if (array_first($send_status)) {
                $site->update(['guide_mail' => true]);
            }
        }

    }

    /**
     * @param $subscription_id
     * @return mixed
     */

    public function checkCreateSite($subscription_id)
    {
        return new CheckCreateSiteForSubscription($subscription_id, null);
    }

    public function installationStarted()
    {
        $this->template->set('meta_title', trans('module::message.installation-title'));
        $this->messages->flush();
        return $this->view->make('visiosoft.module.site::installation-site');
    }

    public function autoRefresh(CustomFieldRepositoryInterface $cf)
    {
        $sites = $this->site->getAutoRefresh();

        foreach ($sites as $site) {
            $site->reinstall();

            foreach ($site->reinstall_addons as $addon) {
                if ($is_addon = $cf->getAdValueByCustomFieldSlug('addon_composer_name', $addon->id)) {
                    event(new AddAddonSite($site, $is_addon));
                }
            }

        }
    }

    public function tryAddon($subdomainName, $addonNamespace, $redirectURL)
    {
        try {
            if (!$customField = $this->cf->findBySlug('addon_composer_name')) {
                throw new \Exception(trans('visiosoft.module.site::message.custom_field_not_found'));
            }

            if (!$classified = $this->adv->findByCFJSON("cf$customField->id", $addonNamespace)) {
                throw new \Exception(trans('visiosoft.module.site::message.classified_not_found'));
            }

            if (!$site = $this->site->findBySubdomain($subdomainName)) {
                throw new \Exception(trans('visiosoft.module.site::message.site_not_found'));
            }

            $installAddons = array();

            if ($isAddon = $this->cf->getAdValueByCustomFieldSlug('addon_composer_name', $classified->id)) {
                if (!$this->addonRepository->isAddonInstalled($isAddon, $site->getId())) {
                    $addonArray = explode('-', $isAddon);
                    $installAddons[] = ["type" => $addonArray[1], "name" => $addonArray[0]];

                    $triedBefore = $this->addonTrialRepository->getModel()
                        ->where([
                            'site_id' => $site->id,
                            'trialed_addon_id' => $classified->id,
                        ])
                        ->first();
                    if ($triedBefore) {
                        throw new \Exception(trans('visiosoft.module.site::message.you_tried_this_addon_before'));
                    }

                    $this->addonTrialRepository->create([
                        'site_id' => $site->id,
                        'trialed_addon_id' => $classified->id,
                        'expires_at' => Carbon::now()->addDays(setting_value('visiosoft.module.site::trial_period') ?: 1),
                    ]);
                }

                $response = [
                    'success' => true,
                    'msg' => trans('visiosoft.module.site::message.addAddon'),
                ];
            } else {
                $response = [
                    'success' => false,
                    'msg' => trans('visiosoft.module.site::message.error_addon_slug'),
                ];
            }

            if (count($installAddons)) {
                event(new AddMultipleAddonSite($site, $installAddons));
            }
        } catch (\Exception $e) {
            $response = [
                'success' => false,
                'msg' => $e->getMessage(),
            ];
        }

        $redirectURL = decrypt($redirectURL);
        $redirectURL = str_replace(':status:', $response['success'] ? 'success' : 'error', $redirectURL);
        $redirectURL = str_replace(':msg:', $response['msg'], $redirectURL);
        return redirect($redirectURL);
    }

    public function removeExpiredAddons()
    {
        try {
            $expiredTrials = $this->addonTrialRepository->getExpiredTrials();

            foreach ($expiredTrials as $trial) {
                event(new DeleteTrial($trial));

                $trial->remove();
            }

            return [
                'success' => true,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'msg' => $e->getMessage()
            ];
        }
    }

    public function getSitesByAuthUser()
    {
        return $this->site->getSitesByAssign(Auth::id());
    }

    public function addAddon()
    {
        $addon_cf_name = $this->cf->getAdValueByCustomFieldSlug('addon_composer_name', $this->request->addon);
        $site = $this->site->find($this->request->site);

        if($addon_cf_name){
            if (!$this->addonRepository->isAddonInstalled($addon_cf_name,$site->getId())){
                $this->addonRepository->create([
                    'subdomain' => $site,
                    'addon' => $addon_cf_name,
                    'payment' => false,
                    'installed' => 'pending',
                    'entry_id' => $this->request->addon,
                    'entry_type' => 'Visiosoft\AdvsModule\Adv\AdvModel'
                ]);
            }
        }

        return event(new AddAddonSite($site, $addon_cf_name));
    }

    public function removeAddon()
    {
        $addon_cf_name = $this->cf->getAdValueByCustomFieldSlug('addon_composer_name', $this->request->addon);
        $siteId = $this->site->find($this->request->site)->id;

        $addon = $this->addonRepository->newQuery()
            ->where('addon',$addon_cf_name)
            ->where('subdomain_id',$siteId)
            ->first();
        $this->addonRepository->delete($addon);

        return event(new DeleteAddonSite($addon));
    }

    public function checkSiteStatus()
    {
        try {
            $sites = $this->site->getSitestoCheck();

            $error_sites = array();
            $error_ssl_sites = array();

            foreach ($sites as $site) {

                $domains = $this->domain->getDomainsBySite([$site->id]);

                // Verify for Domains
                foreach ($domains as $domain) {
                    $url = "https://" . $domain->getDomain();

                    $response = $this->dispatch(new CheckSite($url));

                    $verify = $response['verify'];
                    $ssl = $response['verify_ssl'];

                    if (!$verify) {
                        // Send Error Site Notification
                        $error_sites[] = $url;
                    }

                    if ($verify && !$ssl) {
                        // Send Error SSL Verify Notification
                        $error_ssl_sites[] = $url;
                    }

                    $domain->update([
                        'verify' => $verify,
                        'verify_ssl' => $ssl,
                    ]);
                }

                // Verify for Site base URL
                $url = "https://" . $site->getUrl();

                $response = $this->dispatch(new CheckSite($url));

                $verify = $response['verify'];
                $ssl = $response['verify_ssl'];

                if (!$verify) {
                    // Send Error Site Notification
                    $error_sites[] = $url;
                }

                if ($verify && !$ssl) {
                    // Send Error SSL Verify Notification
                    $error_ssl_sites[] = $url;
                }

                $site->update([
                    'check_at' => Carbon::now(),
                    'verify' => $verify,
                    'verify_ssl' => $ssl,
                ]);

            }

            if (count($error_ssl_sites)) {
                event(new FailedSiteSSL($error_ssl_sites));
            }

            if (count($error_sites)) {
                event(new FailedSite($error_sites));
            }

        } catch (\Exception $exception) {
            return $this->response->json(['error' => true, 'message' => $exception->getMessage()]);
        }
    }

    public function getSiteStatus($id)
    {
        try {
            $status = true;
            $response_data = null;

            if (!$site = $this->site->findByApiToken($id)) {
                return $this->response->json(['success' => false, 'errors' => [trans('visiosoft.module.site::error.site_not_found')]], 404);
            }

            if ($site->status == 'completed') {

                $status = true;
                $response_data = [];
                $adminPanelLogin = 'https://' . $site->getUrl() . '/admin/login';
                $response_data['pid'] = $site->api_token;
                $response_data['address'] = 'https://' . $site->getUrl();
                $response_data['url'] = $site->getUrl();
                $response_data['email'] = ($site->assign) ? $site->assign->email : '';
                $response_data['password'] = $site->password;
                $response_data['admin_login'] = $adminPanelLogin;
//                event(new FreeTrialSubscriptionCreatedEvent($user, $activeSubscription, $password, $website, $adminPanelLogin));
            } else if ($site->status == 'error') {
                return $this->response->json(['success' => false, 'errors' => [trans('visiosoft.module.site::message.error_server')]], 500);
            }

            return $this->response->json(['success' => $status, 'result' => $response_data]);

        } catch (\Exception $e) {
            return $this->response->json(['success' => false, 'errors' => [$e->getMessage()]], 500);
        }
    }

    /**
     * Create New Site View
     */
    public function create()
    {

        // Check if user has any subscription
        $subscription = $this->subscriptionRepository->newQuery()
            ->where('assign_id', Auth::id())
            ->where('enabled', true)
            ->where('expires_at', '>', Carbon::now())
            ->first();

        if (!$subscription) {
            return $this->redirect->route('visiosoft.module.subscriptions::my-subscriptions');
        }
        $this->template->set('meta_title', trans('visiosoft.module.site::button.new_site'));

        return $this->view->make("visiosoft.module.site::create");
    }

    public function store()
    {
            $subscription_id = $this->request->subscription;

            if (!empty($subscription_id) and !$this->checkCreateSite($this->request->subscription)) {
                $this->messages->error(trans('visiosoft.module.site::message.error_create_limit'));
                return $this->redirect->route('visiosoft.module.subscriptions::my-subscriptions');
            }

            if (!empty($this->request->storename)) {

                if (empty($this->request->bundle_id)) {
                    $this->messages->error(trans('visiosoft.module.site::message.error_select_bundle'));
                    return $this->redirect->route('visiosoft.module.site::create-site');
                }

                $store_name = $this->request->storename;
                $password = rand(111111, 999999);

                $site = $this->site->createSite($store_name, Auth::user(), $password, [
                    'type' => setting_value('visiosoft.module.site::default_domain'),
                    'node' => setting_value('visiosoft.module.site::default_node_server'),
                    'default_theme' => $this->request->bundle_id,
                    'locale' => request()->get('locale', 'en'),
                    'subscription_id' => $subscription_id,
                ]);

                $response = array_first(event(new BuildSite($site)));

                if ($response['status']) {
                    $site->changeStatus('pending');
                    $this->messages->success(trans('visiosoft.module.site::message.success_create_store'));
                    return redirect('my-sites');
                } else {
                    $this->messages->error(trans('visiosoft.module.site::message.error_server'));
                    $site->updateErrorMsg($response['message']);
                    return redirect()->back();
                }

                return redirect('my-sites');
            }

    }
    public function index()
    {
        if (Auth::user()->hasRole('admin')){
            $sites = $this->site->all();
        }else{
            $sites = $this->site->findAllBy('assign_id', Auth::id());
        }

        $this->template->set('meta_title', trans('visiosoft.module.site::field.mySites'));

        return $this->view->make("visiosoft.module.site::sites", compact('sites'));
    }
}
