<?php namespace Visiosoft\SimpleClassifiedModule\Http\Controller\Admin\Integration;

use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;
use \Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Visiosoft\TrendyolModule\IntegratedProduct\IntegratedProductRepository;
use Visiosoft\TrendyolModule\Http\Controller\Admin\ApiRequestController;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Anomaly\FilesModule\File\FileModel;
use Anomaly\Streams\Platform\Message\MessageBag;
use Intervention\Image\Facades\Image;

class IntegrateProductToTrendyol extends ApiRequestController
{
    public $entry;

    public function integrate($createdAdv, $builder, $messages)
    {     
        $this->entry = $createdAdv->id;

        $this->getCredentialsForTrendyolApi();

        $this->detectEndpoint();

        // if this is handling of update product operation, we must check if some un-changable product attributes (for Trendyol) have changed. 
        // if yes, throw error

        if (stripos(url()->previous(), 'edit')) {
            $integratedProducts = app(IntegratedProductRepository::class);
            $integratedProductToBeUpdated = $integratedProducts->findBy('product_id', $this->entry);

            preg_match_all("/\d+/", $builder->getFormValue('brand'), $brandInputMatches);
            preg_match_all("/\d+/", $builder->getFormValue('category_trendyol'), $categoryInputMatches);
            preg_match_all("/\d+/", $builder->getFormValue('trendyol_cargoCompany'), $cargoCompanyInputMatches);
            
            $neverChangableformFields = array(
                'barcode' => $builder->getFormValue('barcode_number'),
                'categoryId' => intval($categoryInputMatches[0][0]),
            );

            $conditionallyChangableformFields = array(
                'productMainId' => $builder->getFormValue('productMainId'),
                'brandId' => intval($brandInputMatches[0][0]),
                'attributes' => $builder->getFormValue('categoryAttributes'),
            );

            $productOnTrendyol = json_decode($this->returnProductInformation('createdProductOnTrendyol', $integratedProductToBeUpdated['barcode'], $this->messages))->content[0];

            $neverChangableAttributes = array(
                'barcode' => $productOnTrendyol->barcode,
                'categoryId' => $productOnTrendyol->pimCategoryId,
            );

            $changableAttributesOnlyIfProductIsNotApproved = array(
                'productMainId' => $productOnTrendyol->productMainId,
                'brandId' => $productOnTrendyol->brandId,
                'attributes' => json_encode(json_decode($integratedProductToBeUpdated['trendyol_spesific_information'])->attributes),
            );

            $changedNeverChangableAttributes = array_diff_assoc($neverChangableformFields, $neverChangableAttributes);
            $changedConditionallyChangableAttributes = array_diff_assoc($conditionallyChangableformFields, $changableAttributesOnlyIfProductIsNotApproved);

            if (count($changedNeverChangableAttributes) > 0) {
                $changedAttributes = array();
                foreach($changedNeverChangableAttributes as $attr => $val) {
                    array_push($changedAttributes, $attr);
                }
                $messages->error('Your product\'s ' . implode(", ", $changedAttributes) . ' information cannot be updated from this site');
                return;
            } else if (count($changableAttributesOnlyIfProductIsNotApproved) > 0) {
                if ($productOnTrendyol->approved) {
                    $changedAttrs = array();
                    foreach($changableAttributesOnlyIfProductIsNotApproved as $attr => $val) {
                        array_push($changedAttrs, $attr);
                    }
                    $messages->error('Your product\'s ' . implode(", ", $changedAttrs) . ' information cannot be updated from this site');
                    return;
                }
            }
        }

        if ($messages->has('error')) {
            return;
        }

        $advModel = new AdvModel();
        $fileModel = new FileModel();

        $productInformation = $this->prepareProductForIntegration($createdAdv, $builder, $this->entry, $advModel, $fileModel, $messages);

        if ($messages->has('error')) {
            foreach ($messages->get('error') as $message) {
                return;
            }            
        }

        if (!(stripos(url()->previous(), 'edit'))) {
            $batchRequestId = $this->sendRequestToTrendyol($productInformation, $messages, "POST");
        } else if (stripos(url()->previous(), 'edit')) {
            $batchRequestId = $this->sendRequestToTrendyol($productInformation, $this->messages, "PUT");            
        }
        
        if ($messages->has('error')) {
            return;
        }

        $requestResult = $this->returnProductInformation('resultOfSendRequest', $batchRequestId, $messages);

        if ($messages->has('error')) {
            return;
        }

        $requestResultArr = (array) json_decode($requestResult);

        if ($requestResultArr['failedItemCount'] === 0) {
            if (stripos(url()->previous(), 'edit')) {
                $this->updateIntegrationDatabase($productInformation, $this->entry, $builder);
                $messages->success('Your product has been successfully updated on Trendyol. 
                You can access the page of your product on the Trendyol site from the button next to your product.');
            } else {
                $this->saveIntegrationToDatabase($productInformation, $this->entry, $builder);
                $messages->success('Your product has been successfully created on Trendyol. 
                You can access the page of your product on the Trendyol site from the button next to your product.');
            }
        } else {
            return ['error' => true];
        }
    }

    public function prepareProductForIntegration($createdAdv, $builder, $entry, $advModel, $fileModel, $messages)
    {
        // change other currencies to TRY

        $trendyolListPrice = $builder->getFormValue('trendyol_standardPrice');
        $trendyolPrice = $builder->getFormValue('trendyol_price');

        if ($createdAdv->currency !== 'TRY') {
            $enabledCurrencies = setting_value('visiosoft.module.advs::enabled_currencies');

            if (array_search("TRY", $enabledCurrencies) === false) {
                $messages->error('The operation failed. The foreign currency of a product to be created in Trendyol must be "TRY". 
                The price type of the product you created on this site is not "TRY". 
                However, if you select "TRY" as "enabled currencies" from the "Adv" module settings, we will do the conversion for you!');
                return;
            }

            $settings = app(\Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface::class);

            $currencyListForTrendyolPrice = $this->calculateForeignCurrency($createdAdv->currency, $trendyolPrice, $settings, false);
            $trendyolPrice = $currencyListForTrendyolPrice["TRY"];

            $currencyListForTrendyolListPrice = $this->calculateForeignCurrency($createdAdv->currency, $trendyolListPrice, $settings, false);
            $trendyolListPrice = $currencyListForTrendyolListPrice["TRY"];
        }

        if ($trendyolPrice >= $trendyolListPrice) {
            $messages->error('The operation failed. In order to send a product to Trendyol, the list price (standard price) of your product must be higher than the current price.');
            return;
        }

        // convert product images array into a format that can be sent to Trendyol

        $imagesArrToBeSentToTrendyol = array();

        $productImagesArr = $advModel->getAdvimage($entry);

        foreach ($productImagesArr as $image) {
            array_push($imagesArrToBeSentToTrendyol, (object)array(
                "url" => $image->make()->url()
            ));
        }

        // get Id's from input values

        preg_match_all("/\d+/", $builder->getFormValue('brand'), $brandInputMatches);
        preg_match_all("/\d+/", $builder->getFormValue('category_trendyol'), $categoryInputMatches);
        preg_match_all("/\d+/", $builder->getFormValue('trendyol_cargoCompany'), $cargoCompanyInputMatches);

        if (!(stripos(url()->previous(), 'edit')) && !($builder->getFormValue('variant'))) {
            $productInformation = (object)array(
                "items" => [
                    (object)[
                        'barcode' => $builder->getFormValue('barcode_number'),
                        'title' => $createdAdv->name,
                        'productMainId' => $builder->getFormValue('productMainId'),
                        'brandId' => intval($brandInputMatches[0][0]),
                        'categoryId' => intval($categoryInputMatches[0][0]),
                        'quantity' => $createdAdv->stock,
                        'stockCode' => $builder->getFormValue('stock_code'),
                        'dimensionalWeight' => intval($builder->getFormValue('dimensionalWeight')),
                        'description' => $createdAdv->advs_desc,
                        'currencyType' => 'TR',
                        'listPrice' => $trendyolListPrice,
                        'salePrice' => $trendyolPrice,
                        'cargoCompanyId' => intval($cargoCompanyInputMatches[0][0]),
                        'images' =>  $imagesArrToBeSentToTrendyol,
                        'vatRate' => intval($builder->getFormValue('tax')),
                        'attributes' => json_decode($builder->getFormValue('categoryAttributes')),
                    ],
                ],
            );            
        } else if (stripos(url()->previous(), 'edit')) {
            $productInformation = (object)array(
                "items" => [
                    (object)[
                        'barcode' => $builder->getFormValue('barcode_number'),
                        'title' => $createdAdv->name,
                        'productMainId' => $builder->getFormValue('productMainId'),
                        'brandId' => intval($brandInputMatches[0][0]),
                        'categoryId' => intval($categoryInputMatches[0][0]),
                        'stockCode' => $builder->getFormValue('stock_code'),
                        'dimensionalWeight' => intval($builder->getFormValue('dimensionalWeight')),
                        'description' => $createdAdv->advs_desc,
                        'cargoCompanyId' => intval($cargoCompanyInputMatches[0][0]),
                        'images' =>  $imagesArrToBeSentToTrendyol,
                        'vatRate' => intval($builder->getFormValue('tax')),
                        'attributes' => json_decode($builder->getFormValue('categoryAttributes')),
                    ],
                ],
            );            
        } else if (!(stripos(url()->previous(), 'edit')) && ($builder->getFormValue('variant'))) {
            $variations = array(
                "items" => [],
            );
            for ($x = 0; $x < $builder->getFormValue('variantCount'); $x++) {
                $information = [
                    'barcode' => json_decode($builder->getFormValue('variantBarcode'))[$x],
                    'title' => $createdAdv->name,
                    'productMainId' => $builder->getFormValue('productMainId'),
                    'brandId' => intval($brandInputMatches[0][0]),
                    'categoryId' => intval($categoryInputMatches[0][0]),
                    'quantity' => $createdAdv->stock,
                    'stockCode' => json_decode($builder->getFormValue('variantStockCode'))[$x],
                    'dimensionalWeight' => json_decode($builder->getFormValue('variantDimensionalWeight'))[$x],
                    'description' => $createdAdv->advs_desc,
                    'currencyType' => 'TR',
                    'listPrice' => $trendyolListPrice,
                    'salePrice' => $trendyolPrice,
                    'cargoCompanyId' => intval($cargoCompanyInputMatches[0][0]),
                    'images' =>  $imagesArrToBeSentToTrendyol,
                    'vatRate' => intval($builder->getFormValue('tax')),
                    'attributes' => json_decode($builder->getFormValue('categoryAttributes'))[$x],
                ];
                array_push($variations['items'], (object)$information);
            }
            $productInformation = (object)$variations;            
        }

        if ( is_null($builder->getFormValue('categoryAttributes')) ) {
            $messages->error( trans('visiosoft.module.trendyol::error.categoryAttributesMustFilledIn') );
            return;
        }

        return $productInformation;
    }

    public function saveIntegrationToDatabase($productInformation, $entry, $builder)
    {
        $integratedProducts = app(IntegratedProductRepository::class);

        $integratedProduct = $integratedProducts->create([
            'product_id' => $entry,
            'barcode' => $productInformation->items[0]->barcode,
            'trendyol_spesific_information' => json_encode([
                'listPrice' => $productInformation->items[0]->listPrice,
                'salePrice' => $productInformation->items[0]->salePrice,
                'productMainId' => $productInformation->items[0]->productMainId,
                'brandId' => $builder->getFormValue('brand'),
                'categoryId' => $builder->getFormValue('category_trendyol'),
                'stockCode' => $productInformation->items[0]->stockCode,
                'dimensionalWeight' => $productInformation->items[0]->dimensionalWeight,
                'cargoCompanyId' => $builder->getFormValue('trendyol_cargoCompany'),
                'vatRate' => $productInformation->items[0]->vatRate,
                'attributes' => $productInformation->items[0]->attributes,
            ]),
        ]);

        return redirect('/admin/trendyol');
    }

    public function updateIntegrationDatabase($productInformation, $entry, $builder)
    {
        $integratedProducts = app(IntegratedProductRepository::class);

        $integratedProductToBeUpdated = $integratedProducts->findBy('product_id', $entry);

        $integratedProductToBeUpdated->update([
            'product_id' => $entry,
            'barcode' => $productInformation->items[0]->barcode,
            'trendyol_spesific_information' => json_encode([
                'productMainId' => $productInformation->items[0]->productMainId,
                'brandId' => $builder->getFormValue('brand'),
                'categoryId' => $builder->getFormValue('category_trendyol'),
                'stockCode' => $productInformation->items[0]->stockCode,
                'dimensionalWeight' => $productInformation->items[0]->dimensionalWeight,
                'cargoCompanyId' => $builder->getFormValue('trendyol_cargoCompany'),
                'vatRate' => $productInformation->items[0]->vatRate,
                'attributes' => $productInformation->items[0]->attributes,
            ]),
        ]);

        return redirect('/admin/trendyol');
    }

    public function calculateForeignCurrency($currency, $price, $settings, $showMsg = true)
    {
        $currencies = setting_value('visiosoft.module.advs::enabled_currencies');
        $messages = app(MessageBag::class);
        $foreign_currency = array();

        foreach ($currencies as $currencyIn) {
            if ($currencyIn == $currency) {
                $foreign_currency[$currency] = (int)$price;
            } else {
                try {
                    $url = $currency . "_" . $currencyIn;
                    $freeCurrencyKey = $settings->value('visiosoft.module.advs::free_currencyconverterapi_key');

                    $client = new \GuzzleHttp\Client();
                    $response = $client->request('GET', 'http://free.currencyconverterapi.com/api/v6/convert', ['query' => [
                        'q' => $url,
                        'compact' => 'y',
                        'apiKey' => $freeCurrencyKey
                    ]]);

                    if ($response->getStatusCode() == '200') {
                        $response = (array)\GuzzleHttp\json_decode($response->getBody()->getContents());
                        if (!empty($response)) {
                            $rate = $response[$url]->val;
                            $foreign_currency[$currencyIn] = $price * $rate;
                            return $foreign_currency;
                        }
                    }
                } catch (\GuzzleHttp\Exception\ClientException $e) {
                    $response = $e->getResponse();
                    $responseBodyAsString = $response->getBody()->getContents();
                    $response = json_decode($responseBodyAsString, true);
                    if ($showMsg) {
                        $messages->error($response['error']);
                    }
                }
            }
        }
    }

}
