<?php namespace Visiosoft\SimpleClassifiedModule\Http\Controller\Admin\Integration;

use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\TrendyolModule\IntegratedProduct\IntegratedProductRepository;
use Visiosoft\TrendyolModule\IntegratedProduct\Form\CreateProductOnTrendyolFormHandler;
use \Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\FilesModule\File\FileModel;
use Anomaly\Streams\Platform\Message\MessageBag;

class UpdateProductOnTrendyol extends IntegrateProductToTrendyol
{
    public function updateProduct($productUpdatedOnOpenClassify, $builder, $messages)
    {
        $this->getCredentialsForTrendyolApi();
        $this->detectEndpoint();

        // First, let's compare the product information previously uploaded to Trendyol with the updated product information and detect the changed information.

        $integratedProducts = app(IntegratedProductRepository::class);
        $integratedProductToBeUpdated = $integratedProducts->findBy('product_id', $productUpdatedOnOpenClassify->id);
        $productOnTrendyol = json_decode($this->returnProductInformation('createdProductOnTrendyol', $integratedProductToBeUpdated['barcode'], $this->messages))->content[0];

        $imagesOfProductUpdatedOnOpenClassify = array();
        $advModel = new AdvModel();
        $fileModel = new FileModel();
        $productImagesArr = $advModel->getAdvimage($productUpdatedOnOpenClassify->id);
        foreach ($productImagesArr as $image) {
            array_push($imagesOfProductUpdatedOnOpenClassify, (object)array(
                "url" => $image->make()->url()
            ));
        }

        preg_match_all("/\d+/", $builder->getFormValue('brand'), $brandInputMatches);
        preg_match_all("/\d+/", $builder->getFormValue('category_trendyol'), $categoryInputMatches);
        preg_match_all("/\d+/", $builder->getFormValue('trendyol_cargoCompany'), $cargoCompanyInputMatches);
        preg_match_all("/\d+/", json_decode($integratedProductToBeUpdated['trendyol_spesific_information'])->cargoCompanyId, $savedCargoCompanyMatches);

        $productUpdatedOnOpenClassifyInfos = array(
            'title' => $productUpdatedOnOpenClassify->name,
            'description' => $productUpdatedOnOpenClassify->advs_desc,
            'images' => json_encode($imagesOfProductUpdatedOnOpenClassify),
            'barcode' => $builder->getFormValue('barcode_number'),
            'productMainId' => $builder->getFormValue('productMainId'),
            'brandId' => $brandInputMatches[0][0],
            'categoryId' => intval($categoryInputMatches[0][0]),
            'stockCode' => $builder->getFormValue('stock_code'),
            'dimensionalWeight' => $builder->getFormValue('dimensionalWeight'),
            'cargoCompanyId' => intval($cargoCompanyInputMatches[0][0]),
            'vatRate' => $builder->getFormValue('tax'),
            'attributes' => $builder->getFormValue('categoryAttributes'),
        );

        $trendyolProductInfos = array(
            'title' => $productOnTrendyol->title,
            'description' => $productOnTrendyol->description,
            'images' => json_encode($productOnTrendyol->images),
            'barcode' => $productOnTrendyol->barcode,
            'productMainId' => $productOnTrendyol->productMainId,
            'brandId' => $productOnTrendyol->brandId,
            'categoryId' => $productOnTrendyol->pimCategoryId,
            'stockCode' => $productOnTrendyol->stockCode,
            'dimensionalWeight' => $productOnTrendyol->dimensionalWeight,
            'cargoCompanyId' => $savedCargoCompanyMatches[0][0],
            'vatRate' => $productOnTrendyol->vatRate,
            'attributes' => json_encode(json_decode($integratedProductToBeUpdated['trendyol_spesific_information'])->attributes),
        );

        $changedProductInfos = array_diff_assoc($productUpdatedOnOpenClassifyInfos, $trendyolProductInfos);

        if (count($changedProductInfos) > 0) {
            // Ok, One or more of the (title, description, image, barcode, productMainId, 
            // brandId, categoryId, stockCode, dimensionalWeight,  
            // cargoCompanyId, vatRate, attributes) information has changed.
            // There is probability to use 'updateProducts' service.
            // But what if some un-changable (for Trendyol) product attributes have changed. 

            $neverChangableFormFields = array(
                'barcode' => $builder->getFormValue('barcode_number'),
                'categoryId' => intval($categoryInputMatches[0][0]),
            );

            $conditionallyChangableFormFields = array(
                'productMainId' => $builder->getFormValue('productMainId'),
                'brandId' => intval($brandInputMatches[0][0]),
                'attributes' => $builder->getFormValue('categoryAttributes'),
            );

            $neverChangableAttributes = array(
                'barcode' => $productOnTrendyol->barcode,
                'categoryId' => $productOnTrendyol->pimCategoryId,
            );

            $changableAttributesOnlyIfProductIsNotApproved = array(
                'productMainId' => $productOnTrendyol->productMainId,
                'brandId' => $productOnTrendyol->brandId,
                'attributes' => json_encode(json_decode($integratedProductToBeUpdated['trendyol_spesific_information'])->attributes),
            );

            $changedNeverChangableAttributes = array_diff_assoc($neverChangableFormFields, $neverChangableAttributes);
            $changedConditionallyChangableAttributes = array_diff_assoc($conditionallyChangableFormFields, $changableAttributesOnlyIfProductIsNotApproved);

            if (count($changedNeverChangableAttributes) > 0) {
                $changedAttributes = array();
                foreach($changedNeverChangableAttributes as $attr => $val) {
                    array_push($changedAttributes, $attr);
                }
                $messages->error('Your product\'s ' . implode(", ", $changedAttributes) . ' information cannot be updated from this site');
                return;
            } else if (count($changableAttributesOnlyIfProductIsNotApproved) > 0) {
                if ($productOnTrendyol->approved) {
                    $changedAttrs = array();
                    foreach($changableAttributesOnlyIfProductIsNotApproved as $attr => $val) {
                        array_push($changedAttrs, $attr);
                    }
                    $messages->error('Your product\'s ' . implode(", ", $changedAttrs) . ' information cannot be updated from this site');
                    return;
                }
            }
            
            // Ok, there is no problem! Let's use 'updateProducts' service.

            $productInformation = $this->prepareProductForInfoUpdateOnTrendyol($productUpdatedOnOpenClassify, $integratedProductToBeUpdated, $imagesOfProductUpdatedOnOpenClassify, $builder);

            $batchRequestId = $this->sendRequestToTrendyol($productInformation, $this->messages, "PUT");

            if ($this->messages->has('error')) {
                return;
            }

            $requestResult = $this->returnProductInformation('resultOfSendRequest', $batchRequestId, $this->messages);

            if ($this->messages->has('error')) {
                return;
            }

            $requestResultArr = (array) json_decode($requestResult);

            if ($requestResultArr['failedItemCount'] === 0) {
                $changedInfos = array();
                foreach($changedProductInfos as $info => $val) {
                    array_push($changedInfos, $info);
                }
                $this->messages->success('Your product\'s ' . implode(", ", $changedInfos) . ' information has been updated also on Trendyol.');

                $this->saveUpdatedInfosToDatabase($productInformation, $changedInfos, $integratedProductToBeUpdated);
            } else {
                return ['error' => true];
            }
        }

        // Updating price and stock information requires the use of a different service than updating other product information on Trendyol.

        $this->updateStockAndPrice($integratedProductToBeUpdated, $productUpdatedOnOpenClassify, $productOnTrendyol, $builder);
    }

    public function prepareProductForInfoUpdateOnTrendyol($productUpdatedOnOpenClassify, $integratedProductToBeUpdated, $imagesArrToBeSentToTrendyol, $builder)
    {
        $trendyolSpesificInformation = json_decode($integratedProductToBeUpdated['trendyol_spesific_information']);

        preg_match_all("/\d+/", $builder->getFormValue('brand'), $brandInputMatches);
        preg_match_all("/\d+/", $builder->getFormValue('category_trendyol'), $categoryInputMatches);
        preg_match_all("/\d+/", $builder->getFormValue('trendyol_cargoCompany'), $cargoCompanyInputMatches);

        $productInformation = (object)array(
                "items" => [
                    (object)[
                        'barcode' => $integratedProductToBeUpdated['barcode'],
                        'title' => $productUpdatedOnOpenClassify->name,
                        'productMainId' => $builder->getFormValue('productMainId'),
                        'brandId' => intval($brandInputMatches[0][0]),
                        'categoryId' => intval($categoryInputMatches[0][0]),
                        'stockCode' => $builder->getFormValue('stock_code'),
                        'dimensionalWeight' => $builder->getFormValue('dimensionalWeight'),
                        'description' => $productUpdatedOnOpenClassify->advs_desc,
                        'cargoCompanyId' => intval($cargoCompanyInputMatches[0][0]),
                        'images' =>  $imagesArrToBeSentToTrendyol,
                        'vatRate' => $productUpdatedOnOpenClassify->tax,
                        'attributes' => json_decode($builder->getFormValue('categoryAttributes')),
                    ],
                ],
        );
        
        return $productInformation;
    }

    public function updateStockAndPrice($integratedProductToBeUpdated, $productUpdatedOnOpenClassify, $productOnTrendyol, $builder)
    {
        // First, change operation if it is necessary.

        $trendyolListPrice = $builder->getFormValue('trendyol_standardPrice');
        $trendyolPrice = $builder->getFormValue('trendyol_price');

        if ($productUpdatedOnOpenClassify->currency !== 'TRY') {
            $enabledCurrencies = setting_value('visiosoft.module.advs::enabled_currencies');

            if (array_search("TRY", $enabledCurrencies) === false) {
                $messages->error('The operation failed. The foreign currency of a product to be created in Trendyol must be "TRY". 
                The price type of the product you created on this site is not "TRY". 
                However, if you select "TRY" as "enabled currencies" from the "Adv" module settings, we will do the conversion for you!');
                return;
            }

            $settings = app(\Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface::class);

            $currencyListForTrendyolPrice = $this->calculateForeignCurrency($createdAdv->currency, $trendyolPrice, $settings, false);
            $trendyolPrice = $currencyListForTrendyolPrice["TRY"];

            $currencyListForTrendyolListPrice = $this->calculateForeignCurrency($createdAdv->currency, $trendyolListPrice, $settings, false);
            $trendyolListPrice = $currencyListForTrendyolListPrice["TRY"];
        }

        if ($trendyolPrice >= $trendyolListPrice) {
            $messages->error('The operation failed. In order to send a product to Trendyol, the list price (standard price) of your product must be higher than the current price.');
            return;
        }

        // Detect if one or more of the (stock quantity, price, list price information) has changed.

        $productUpdatedOnOpenClassifyInfos = [
            'stock' => $builder->getFormValue('stock'),
            'listPrice' => $trendyolListPrice,
            'salePrice' => $trendyolPrice,
        ];

        $trendyolProductInfos = [
            'stock' => $productOnTrendyol->quantity,
            'listPrice' => $productOnTrendyol->listPrice,
            'salePrice' => $productOnTrendyol->salePrice,
        ];

        $changedProductInfos = array_diff($productUpdatedOnOpenClassifyInfos, $trendyolProductInfos);

        if (count($changedProductInfos) > 0) {
            // Yes, One or more of the (stock quantity, price, list price information) has changed.
            // Let's use 'updatePriceAndInventory' service.

            $productInformation = $this->prepareProductForStockAndPriceUpdate($productUpdatedOnOpenClassifyInfos, $integratedProductToBeUpdated);

            if ($this->messages->has('error')) {
                return;
            }

            $batchRequestId = $this->sendRequestToTrendyol($productInformation, $this->messages, "POST_UPDATE");

            if ($this->messages->has('error')) {
                return;
            }

            $requestResult = $this->returnProductInformation('resultOfSendRequest', $batchRequestId, $this->messages);

            if ($this->messages->has('error')) {
                return;
            }

            $requestResultArr = (array) json_decode($requestResult);

            if ($requestResultArr['failedItemCount'] === 0) {
                $changedInfos = array();
                foreach($changedProductInfos as $info => $val) {
                    array_push($changedInfos, $info);
                }
                $this->messages->success('Your product\'s ' . implode(", ", $changedInfos) . ' information has been updated also on Trendyol.                     
                If you have not changed the price information, there has been a change due to the exchange rate difference.');
                
                $this->saveUpdatedInfosToDatabase($productInformation, $changedInfos, $integratedProductToBeUpdated);
            } else {
                return ['error' => true];
            }            
        }
    }

    public function prepareProductForStockAndPriceUpdate($productUpdatedOnOpenClassifyInfos, $integratedProductToBeUpdated)
    {
        if ($productUpdatedOnOpenClassifyInfos['salePrice'] >= $productUpdatedOnOpenClassifyInfos['listPrice']) {
            $this->messages->error('The operation failed. In order to update a product on Trendyol, the list price (standard price) of your product must be higher than the current price.');
            return;
        }

        $productInformation = (object)array(
            "items" => [
                (object)[
                    'barcode' => $integratedProductToBeUpdated['barcode'],
                    "quantity" => $productUpdatedOnOpenClassifyInfos['stock'],
                    "salePrice" => $productUpdatedOnOpenClassifyInfos['salePrice'],
                    "listPrice" => $productUpdatedOnOpenClassifyInfos['listPrice']
                ],
            ],
        );

        return $productInformation;
    }

    public function calculateForeignCurrency($currency, $price, $settings, $showMsg = true)
    {
        $currencies = setting_value('visiosoft.module.advs::enabled_currencies');
        $messages = app(MessageBag::class);
        $foreign_currency = array();

        foreach ($currencies as $currencyIn) {
            if ($currencyIn == $currency) {
                $foreign_currency[$currency] = (int)$price;
            } else {
                try {
                    $url = $currency . "_" . $currencyIn;
                    $freeCurrencyKey = $settings->value('visiosoft.module.advs::free_currencyconverterapi_key');

                    $client = new \GuzzleHttp\Client();
                    $response = $client->request('GET', 'http://free.currencyconverterapi.com/api/v6/convert', ['query' => [
                        'q' => $url,
                        'compact' => 'y',
                        'apiKey' => $freeCurrencyKey
                    ]]);

                    if ($response->getStatusCode() == '200') {
                        $response = (array)\GuzzleHttp\json_decode($response->getBody()->getContents());
                        if (!empty($response)) {
                            $rate = $response[$url]->val;
                            $foreign_currency[$currencyIn] = $price * $rate;
                            return $foreign_currency;
                        }
                    }
                } catch (\GuzzleHttp\Exception\ClientException $e) {
                    $response = $e->getResponse();
                    $responseBodyAsString = $response->getBody()->getContents();
                    $response = json_decode($responseBodyAsString, true);
                    if ($showMsg) {
                        $messages->error($response['error']);
                    }
                }
            }
        }
    }

    public function saveUpdatedInfosToDatabase($productInformation, $changedInfos, $integratedProductToBeUpdated)
    {
        $obj = json_decode($integratedProductToBeUpdated['trendyol_spesific_information']);

        foreach ($changedInfos as $changedInfo) {
            if (property_exists($obj, $changedInfo)) {
                $obj->{$changedInfo} = $productInformation->items[0]->{$changedInfo};
            }
        }

        $integratedProductToBeUpdated->update([
            'product_id' => $integratedProductToBeUpdated->product_id,
            'barcode' => $integratedProductToBeUpdated->barcode,
            'trendyol_spesific_information' => json_encode($obj),
        ]);
    }
}