<?php namespace Visiosoft\SahibindenModule\Http\Controller\Admin;

use Anomaly\FilesModule\File\Contract\FileRepositoryInterface;
use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\Streams\Platform\Entry\Contract\EntryInterface;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;
use Visiosoft\BotModule\Http\Controller\ScrapesController;
use Visiosoft\CatsModule\Category\CategoryModel;
use Visiosoft\LocationModule\City\CityModel;
use Visiosoft\LocationModule\District\DistrictModel;
use Visiosoft\LocationModule\Neighborhood\NeighborhoodModel;
use Visiosoft\ProfileModule\Profile\Contract\ProfileRepositoryInterface;
use Visiosoft\SahibindenModule\Image\ImageModel;
use Visiosoft\SahibindenModule\Product\Form\ProductFormBuilder;
use Visiosoft\SahibindenModule\Product\ProductModel;
use Visiosoft\SahibindenModule\Product\Table\ProductTableBuilder;
use Anomaly\Streams\Platform\Http\Controller\AdminController;

class ProductController extends AdminController
{
    private $model;
    private $userRepository;
    private $postFiles;
    private $str;
    private $image;
    private $settings;
    private $scrapesController;
    private $profileRepository;
    private $advRepository;

    public function __construct(
        ProductModel $model,
        UserRepositoryInterface $userRepository,
        FileRepositoryInterface $postFiles,
        Str $str,
        ImageModel $image,
        SettingRepositoryInterface $settings,
        ScrapesController $scrapesController,
        ProfileRepositoryInterface $profileRepository,
        AdvRepositoryInterface $advRepository
    )
    {
        parent::__construct();
        $this->model = $model;
        $this->users = $userRepository;
        $this->postFiles = $postFiles;
        $this->str = $str;
        $this->image = $image;
        $this->settings = $settings;
        $this->scrapesController = $scrapesController;
        $this->profileRepository = $profileRepository;
        $this->advRepository = $advRepository;
    }

    /**
     * Display an index of existing entries.
     *
     * @param ProductTableBuilder $table
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function index(ProductTableBuilder $table)
    {
        $table->setButtons(array_merge($table->getButtons(), [
            'publish' => [
                'text' => 'visiosoft.module.sahibinden::button.publish.name',
                'href' => '/admin/sahibinden/save-ads/{entry.id}',
                'class' => function (EntryInterface $entry) {
                    if ($entry->saved == 1) {
                        return 'disabled';
                    }
                }
            ]
        ]));
        return $table->render();
    }

    /**
     * Create a new entry.
     *
     * @param ProductFormBuilder $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function create(ProductFormBuilder $form, Request $request)
    {
        set_time_limit(0);
        if ($request->action == "save") {
            if ($request->linkType === 'store') {
                $response = $this->getPaginationLinks($request->store_url);
                $response = $this->getAdsLinks($response, 'store');
                $response = $this->getAdDetails($response);
                if ($response == 'api_error') {
                    $this->messages->error(trans('visiosoft.module.sahibinden::message.api_error'));
                    return redirect('/admin/sahibinden');
                }
                $this->messages->success(trans('streams::message.create_success', ['name' => 'Products']));
                return redirect('/admin/sahibinden');
            } else {
                $response = $this->getPaginationLinks($request->store_url);
                $response = $this->getAdsLinks($response, 'category');
                $response = $this->getAdDetails($response);
                if ($response == 'api_error') {
                    $this->messages->error(trans('visiosoft.module.sahibinden::message.api_error'));
                    return redirect('/admin/sahibinden');
                }
                $this->messages->success(trans('streams::message.create_success', ['name' => 'Products']));
                return redirect('/admin/sahibinden');
            }
        }
        $form->setActions(['save']);
        $form->setFields([
            'store_url',
            'linkType' => [
                'type' => 'select',
                'options' => ['store' => 'Store Link', 'category' => 'Category Link'],
            ],
        ]);
        return $form->render();
    }

    /**
     * Edit an existing entry.
     *
     * @param ProductFormBuilder $form
     * @param        $id
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function edit(ProductFormBuilder $form, $id)
    {
        return $form->render($id);
    }

    public function getPaginationLinks($store_url)
    {
        $pageLinks = array($store_url);
        $next = $store_url;
        $host = parse_url($store_url);
        $domain = $host['scheme'] . '://' . $host['host'];
        do {
            $xpath = $this->scrapesController->curl($next);
            $doc = new \DOMDocument;
            libxml_use_internal_errors(true);
            $doc->loadHTML($xpath['response']);
            $xpath = new \DOMXPath($doc);
            $next = $xpath->query("//ul[contains(concat(' ', normalize-space(@class), ' '), ' pageNaviButtons ')]/li[last()]/a[contains(concat(' ', normalize-space(@class), ' '), ' prevNextBut ')]");
            $next = $next->item(0) ? $next->item(0)->getAttribute('href') : null;
            if (!is_null($next)) {
                $next = $domain . $next;
                $pageLinks[] = $next;
            }
        } while ($next);

        // Register Pagination Links in Bot Module
        foreach ($pageLinks as $pageLink) {
            $this->scrapesController->registerScrape($pageLink, 'pagination', 'sahibinden');
        }
    }

    public function curl($url)
    {
        $ch = curl_init();
        $api_key = $this->settings->value('visiosoft.module.sahibinden::scraperapi_key');
        curl_setopt($ch, CURLOPT_URL, "http://api.scraperapi.com/?api_key=" . $api_key ."&country_code=us&render=true&url=" . $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            "Accept: application/json"
        ));
        $response = curl_exec($ch);
        $status_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        $doc = new \DOMDocument;
        libxml_use_internal_errors(true);
        $doc->loadHTML($response);
        $xpath = new \DOMXPath($doc);
        return $xpath;
    }

    public function getAdsLinks($pageUrl, $urlType)
    {
        $usedXPath = $urlType === 'store' ?
            '//*[@id="container"]/div[1]/div[3]/div[2]/table/tbody/tr/td[1]/a' :
            '//table[@id="searchResultsTable"]/tbody/tr/td[1]/a';

        $adLinks = array();
        $xpath = $this->scrapesController->curl($pageUrl);
        $doc = new \DOMDocument;
        libxml_use_internal_errors(true);
        $doc->loadHTML($xpath['response']);
        $xpath = new \DOMXPath($doc);
        $adsLinks = $xpath->query($usedXPath);
        foreach ($adsLinks as $adLink) {
            $link = $adLink->getAttribute('href');
            if (!in_array($link, $adLinks)) {
                $link = strpos($link, "http") === 0 ? $link : 'https://www.sahibinden.com' . $link;
                $adLinks[] = $link;
                $this->scrapesController->registerScrape($link, 'page', 'sahibinden');
            }
        }

        return $adLinks;
    }

    function getAdDetails($adLink)
    {
        $adDetails = array();
        $xpath = $this->curl($adLink);
        $adId = $xpath->query('//*[@id="classifiedId"]');
        if (is_null($adId->item(0))) {
            return 'api_error';
        }
        $adId = $adId->item(0)->nodeValue;

        $username = $xpath->query("//div[contains(concat(' ', normalize-space(@class), ' '), ' username-info-area ')]/h5");
        $username = $username->item(0) ?
            $username->item(0)->nodeValue :
            $xpath->query("//div[contains(concat(' ', normalize-space(@class), ' '), ' Test ')]/p")->item(0)->nodeValue;

        $desc = $xpath->query('//*[@id="classifiedDescription"]');
        $desc = $desc->item(0)->nodeValue;

        $productName = $xpath->query('//*[@id="classifiedDetail"]/div[1]/div[1]/h1');
        $productName = $productName->item(0)->nodeValue;

        $price = $xpath->query('//*[@id="classifiedDetail"]/div[1]/div[2]/div[2]/h3/text()');
        $price = $price->item(0)->nodeValue;
        $price = array_filter(explode(' ', $price));
        unset($price[0]);
        sort($price);

        $locations = $xpath->query('//*[@id="classifiedDetail"]/div[1]/div[2]/div[2]/h2/a');
        $adLocation = array();
        foreach ($locations as $location) {
            $adLocation[] = preg_replace('/\s+/', '', $location->nodeValue);
        }

        $phone = $xpath->query('//*[@id="phoneInfoPart"]/li/span[1]');
        $phone = $phone->item($phone->length - 1)->nodeValue;

        $categories = $xpath->query('//*[@id="uiBreadCrumb"]/li/a');

        $adCategory = array();
        foreach ($categories as $key => $category) {
            $adCategory[] = preg_replace('/\s+/', '', $category->nodeValue);
        }

        $images = array();
        $photos = $xpath->query('//*[@id="classifiedDetail"]/div[1]/div[2]/div[1]/div[2]/ul/li/ul/li/label/img');
        foreach ($photos as $photo) {
            $newImage = array();
            $imagePath = $photo->getAttribute('src');
            $newImage['thmb'] = $imagePath;
            $newImage['org'] = substr($imagePath, 0, strpos($imagePath, "thmb")) . "x5" . substr($imagePath, strpos($imagePath, "thmb") + 4);
            $images[] = $newImage;
        }

        $adDetails['ad_id'] = $adId;
        $adDetails['price'] = $price;
        $adDetails['name'] = $productName;
        $adDetails['desc'] = $desc;
        $adDetails['city'] = $adLocation[0];
        if (count($adLocation) > 1) $adDetails['district'] = $adLocation[1];
        if (count($adLocation) > 2) $adDetails['neighborhood'] = $adLocation[2];
        $adDetails['phone'] = preg_replace("/[^0-9]/", "", $phone);
        $adDetails['images'] = $images;
        $adDetails['category'] = array_slice($adCategory, 0, count($adCategory) - 4);

        // Create a new user based on the user who posted the product
        $user = $this->userRepository->findBy('username', $this->str->slug($username, '_'));
        $user = is_null($user) ? $this->createUser($username, $phone) : $user;

        $response = $this->postAd($adDetails, $user);

        return $response;
    }

    public function postAd($adDetails, $user)
    {
        $advs_model = new AdvModel();
        $category_Model = new CategoryModel();
        $city_Model = new CityModel();
        $district_model = new DistrictModel();
        $neighborhood_model = new NeighborhoodModel();

        /* Looping through the categories and creating an array of categories.. ['cat1', 'cat2', etc..] */
        $category = array();
        for ($i = 0; $i < count($adDetails['category']); $i++) {
            $cat = $i + 1;
            $category['cat' . $cat] = $adDetails['category'][$i];
        }

        $cat1 = $category_Model->newQuery()->leftJoin('cats_category_translations', function ($join) {
            $join->on('cats_category.id', '=', 'cats_category_translations.entry_id');
        });
        $cat1 = $cat1->where('name', $category['cat1'])->first();
        $cat1 = is_null($cat1) ? $cat1 = 1 : $cat1->id;

        $city = $city_Model->where('name', $adDetails['city'])->first();
        $city = is_null($city) ? 1 : $city->id;

        $district = $district_model->where('name', $adDetails['district'])->first();
        $district = is_null($district) ? 947 : $district->id;

        $neighborhood = $neighborhood_model->where('name', $adDetails['neighborhood'])->first();
        $neighborhood = is_null($neighborhood) ? 2 : $neighborhood->id;

        $product = $this->advRepository->findBy('slug', $adDetails['name'] . '_' . $adDetails['ad_id']);
        $date = date('Y-m-d H:i:s');
        $data = [
            'en' => [
                'name' => $adDetails['name'],
                'advs_desc' => $adDetails['desc'],
            ],
            'tr' => [
                'name' => $adDetails['name'],
                'advs_desc' => $adDetails['desc'],
            ],
            'price' => $adDetails['price'][0],
            'cat1' => $cat1,
            'country_id' => 212,
            'city' => $city,
            'district' => $district,
            'neighborhood' => $neighborhood,
            'slug' => $adDetails['name'] . '_' . $adDetails['ad_id'],
            'currency' => $adDetails['price'][1],
            'cover_photo' => reset($adDetails['images'])['org'],
            'created_by' => $user,
            'status' => 'approved',
            'publish_at' => $date,
            'finish_at' => date('Y-m-d H:i:s', strtotime($date . ' + 60 day')),
        ];

        if (is_null($product)) {
            $saved = $advs_model->create($data);
        } else {
            $saved = $advs_model->update($data);
        }
        $this->savePhoto($adDetails['images'], $product->id);
    }

    public function savePhoto($photosUrl, $productId)
    {
        /* Looping through the photo's URLs and saving them */
        foreach ($photosUrl as $photoUrl) {
            $link = $photoUrl['org'];
            $name = basename($link);
            $contents = file_get_contents($link);
            $target = getcwd() . "/app/default/files-module/local/images/" . $name;
            /* Saving the photos in the local storage */
            file_put_contents($target, $contents);
            $file = $this->saveFileInModel($name, $productId);
            DB::table('advs_advs_files')->insert(
                ['file_id' => $file->id, 'entry_id' => $productId]
            );
        }
    }

    public function saveFileInModel($filename, $productId)
    {
        return $this->postFiles->newQuery()->create(
            [
                'folder_id' => 1,
                'name' => $filename,
                'entry_id' => $productId,
                'disk_id' => 1,
                'size' => 1,
                'mime_type' => "image/jpeg",
                'extension' => 1,
            ]
        );
    }

    public function createUser($username, $phoneNum)
    {
        $password = str_random(8);
        $user = $this->userRepository->create(
            [
                'username' => $username,
                'display_name' => $username,
                'email' => $this->str->slug($username, "_") . '@user.com',
                'activated' => 1,
            ]
        );
        $user->setAttribute('password', $password);
        $this->userRepository->save($user);

        // Create a profile for the user
        $this->profileRepository->create([
            'user_no' => $user,
            'gsm_phone' => $phoneNum,
        ]);

        return $user;
    }
}
