<?php namespace Visiosoft\RotaModule;

use Anomaly\Streams\Platform\Addon\Module\Module;
use Carbon\Carbon;

use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Visiosoft\RotaModule\Customer\Contract\CustomerRepositoryInterface;
use Visiosoft\RotaModule\Events\CustomerCreated;
use Visiosoft\RotaModule\Events\OrderCreated;
use Visiosoft\RotaModule\Events\ProductCreated;
use Visiosoft\RotaModule\Invoice\Contract\InvoiceRepositoryInterface;
use Visiosoft\RotaModule\Product\Contract\ProductRepositoryInterface;
use Visiosoft\RotaModule\Services\StockCard;

class RotaModule extends Module
{

    /**
     * @var string
     */
    protected $provide = 'invoice_provider';

    /**
     * The navigation display flag.
     *
     * @var bool
     */
    protected $navigation = false;

    /**
     * The addon icon.
     *
     * @var string
     */
    protected $icon = 'fa fa-puzzle-piece';

    /**
     * The module sections.
     *
     * @var array
     */
    protected $sections = [
        'invoice' => [
            'buttons' => [
                'new_invoice',
            ],
        ],
    ];


    public function createCustomer($params)
    {
        $customerRepository = app(CustomerRepositoryInterface::class);
        $customer = $customerRepository->findBy('customer_id', $params['customer_id']);
        if (empty($params['identity_number'])) {
            $params['identity_number'] = "11111111111";
        }

        if (!$customer && $customerRepository->create($params)) {
            $customer = $customerRepository->findBy('customer_id', $params['customer_id']);
        }

        $formatted_params = [
            "user_id" => setting_value('visiosoft.module.rota::user_id'),
            "account" => $customer->name,
            "code" => $customer->customer_id,
            "phone" => $customer->phone,
            "gsmno" => $customer->gsm,
            "vdairesi" => !empty($customer->tax_office) ? $customer->tax_office : $customer->identity_number,
            "vkn" => !empty($customer->tax_number) ? $customer->tax_number : $customer->identity_number,
            "tckn" => $customer->identity_number,
            "email" => $customer->email,
            "address" => $customer->address,
            "city" => $customer->city,
            "state" => $customer->state,
            "zip" => $customer->zipcode,
            "country" => $customer->country,
            "gname" => "Hiçbiri",
            "gid" => "0",
            "firmaid" => setting_value('visiosoft.module.rota::company_id'),
            "subeid" => [["subeid" => "1"]],
            "adresler" => [],
            "yetkililer" => []
        ];

        $create = $customerRepository->addToRota($formatted_params);
        if ($create->message == "Chk was created.") {
            $customer->rota_id = $create->result;
            $customer->sended = true;
            $customer->sended_at = Carbon::now();
            $customer->save();

            return $this->sendResponse($create->result, '');
        } else {
            throw new \Exception($create->message);
        }
    }

    public function getStocksByWarehouse($warehouse_provider_id)
    {
        $productRepository = app(ProductRepositoryInterface::class);

        $formatted_params = [
            "user_id" => setting_value('visiosoft.module.rota::user_id'),
            "company_id" => setting_value("visiosoft.module.rota::company_id"),
            "id" => 1,
            "first_date" => "1965-01-01",
            "last_date" => date("Y") . "-12-30"
        ];

        $list = $productRepository->stockListByWarehouse($formatted_params);
        if (is_array($list)) {
            return $this->sendResponse($list,'');
        }
    }

    public function createProduct($params)
    {
        $productRepository = app(ProductRepositoryInterface::class);
        $product = $productRepository->findBy('product_id', $params['product_id']);

        if (!$product && $productRepository->create($params)) {
            $product = $productRepository->findBy('product_id', $params['product_id']);
        }

        $formatted_params = [
            "user_id" => setting_value('visiosoft.module.rota::user_id'),
            "firmaid" => setting_value('visiosoft.module.rota::company_id'),
            "name" => $product->name,
            "unit" => $product->unit,
            "item_number" => $product->barcode,
            "barkod" => $product->barcode,
            "ozelkod" => $product->barcode
        ];

        $create = $productRepository->addToRota($formatted_params);
        if ($create->message == "Stk was created.") {
            $product->rota_id = $create->result;
            $product->sended = true;
            $product->sended_at = date("Y-m-d H:i:s");
            $product->save();

            return $this->sendResponse($create->result, '');
        } else {
            throw new \Exception($create->message);
        }
    }


    public function createInvoice($params)
    {
        $invoiceRepository = app(InvoiceRepositoryInterface::class);
        $customerRepository = app(CustomerRepositoryInterface::class);
        $productRepository = app(ProductRepositoryInterface::class);

        $invoice = $invoiceRepository->findBy('order_id', $params['order_id']);
        if (!$invoice) {
            $params['products'] = json_encode($params['products']);
            if ($invoiceRepository->create($params)) {
                $invoice = $invoiceRepository->findBy('order_id', $params['order_id']);
            }
        }
        $customer = $customerRepository->findBy('customer_id', $invoice->customer_id);
        $products = json_decode($invoice->products, true);

        foreach ($products as $key => $product) {
            $product_info = $productRepository->findBy('product_id', $product['id']);

            $temp_arr = [
                'urunid' => $product_info->rota_id,
                'itemcode' => $product_info->barcode,
                'description' => $product_info->description,
                'unit' => $product_info->unit,
                'qty' => $product['qty'],
                'amount' => $product['price'],
                'discount' => $product['discount'],
                'total' => $product['total'],
                "userid" => $customer->rota_id,
                'depo_id' => 1,
                'depo_ad' => ""

            ];

            $products[$key] = $temp_arr;
        }

        $formatted_params = [
            "user_id" => setting_value('visiosoft.module.rota::user_id'),
            "type" => "Sales",
            "userid" => $customer->rota_id,
            "invoicenum" => "B2B",
            "cn" => $invoice->order_id,
            "account" => $customer->name,
            "subtotal_dvz" => $invoice->subtotal,
            "discount_dvz" => $invoice->discount,
            "tax1_dvz" => $invoice->tax,
            "total_dvz" => $invoice->total,
            "subtotal" => $invoice->subtotal,
            "discount" => $invoice->discount,
            "tax1" => $invoice->tax,
            "total" => $invoice->total,
            "datecreated" => $invoice->created_at,
            "duedate" => $invoice->created_at,
            "teslimdate" => $invoice->created_at,
            "firmaid" => setting_value('visiosoft.module.rota::company_id'),
            "musteritemsilcisi" => setting_value('visiosoft.module.rota::user_id'),
            "depoid" => 1,
            "depoadi" => "",
            "subeid" => 1,
            "items" => $products,
            "plate" => $invoice->plate,
        ];

        $create = $invoiceRepository->addToRota($formatted_params);
        if ($create->message == "Order was created.") {
            $invoice->rota_id = $create->result;
            $invoice->sended = true;
            $invoice->sended_at = date("Y-m-d H:i:s");
            $invoice->save();

            return $this->sendResponse($create->result, '');
        } else {
            throw new \Exception($create->message);
        }
    }

    public function sendResponse($data, $message, $success = true)
    {
        $response = ['success' => $success, 'data' => $data, 'message' => $message];
        return $response;
    }

    public function storeInvoice($params)
    {

        if (is_null($params->user->id) || is_null($params->category) || is_null($params->id))
        {
            throw new \Exception("Missing parameters".json_encode($params));
        }

        // Dynamic parameters
        $tax = $params->payment_amount / (100 / $params->tax_percent);
        $subtotal = $params->payment_amount - $tax;
        $tax_number = "22222222222";

        // Check If Company Exists
        $is_company = !is_null($params->company_id);

        if ($is_company) {
            $tax_number = $params->company->organization_number;
        }

        $customerRepository = app(CustomerRepositoryInterface::class);
        if (!$customerRepository->findBy('customer_id', $params->user_id))
        {
            $customer_params = [
                'customer_id' => $params->user->id,
                'name' => $params->user->display_name ?: 'Unregistered User',
                'phone' => $params->user->phone ?: '+905555555555',
                'tax_office' => $params->user->city ?: 'İstanbul',
                'tax_number' => $tax_number,
                'identity_number' => "00000000000",
                'email' => $params->user->email ?: setting_value("streams::email"),
                'address' => $params->user->extended_address ?: 'No Address',
                'city' => $params->user->city ?: 'No City',
                'state' => "No State",
                'zipcode' => $params->user->zip_code ?: '34000',
                'country' => $params->user->country ?: 'No Country'
            ];

            try {
                event(new CustomerCreated($customer_params));
            }catch (\Exception $e){
                $this->logError($e);
                return false;
            }
        }


        $product_params = [
            'product_id' => $params->id,
            'name' => $params->plate." - ".$params->category->name." - ".$params->payment_amount." ".$params->currency,
            'barcode' => $params->category->slug,
            'unit' => "kwh",
            'description' => $params->plate." - ".$params->category->name." - ".$params->payment_amount." ".$params->currency
        ];
        $invoice_params = [
            'customer_id' => $params->user->id,
            'order_id' => $params->id,
            'products' => [["id" => $params->id, "qty" => 1, "price" => $params->payment_amount, 'discount' => 0, 'total' => $params->payment_amount]],
            'subtotal' => $subtotal,
            'discount' => 0,
            'tax' => $tax,
            'total' => $params->payment_amount,
            'plate' => $params->plate
        ];

        try {
            event(new ProductCreated($product_params));
            event(new OrderCreated($invoice_params));
        } catch (\Exception $e) {
            $this->logError($e);
            return false;
        }

        return true;
    }

    public function logError ($e) {
        $log = new Logger('name');
        $log->pushHandler(new StreamHandler(storage_path('logs/rota.log')), Logger::ERROR);
        $log->error($e);
    }


}
