<?php namespace Visiosoft\RotaModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Visiosoft\LocationModule\City\Contract\CityRepositoryInterface;
use Visiosoft\LocationModule\Country\Contract\CountryRepositoryInterface;
use Visiosoft\LocationModule\District\Contract\DistrictRepositoryInterface;
use Visiosoft\OrdersModule\Orderdetail\Contract\OrderdetailRepositoryInterface;
use Visiosoft\OrdersModule\Orderpayment\Contract\OrderpaymentRepositoryInterface;
use Visiosoft\ProfileModule\Adress\Contract\AdressRepositoryInterface;
use Visiosoft\RotaModule\Invoice\Contract\InvoiceRepositoryInterface;
use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;

class InvoiceController extends PublicController
{

    private $orderdetailRepository;
    private $orderpaymentRepository;
    private $advRepository;
    private $invoiceRepository;
    private $adressRepository;
    private $cityRepository;
    private $districtRepository;
    private $countryRepository;

    public function __construct(
        OrderdetailRepositoryInterface $orderdetailRepository,
        OrderpaymentRepositoryInterface $orderpaymentRepository,
        AdvRepositoryInterface $advRepository,
        InvoiceRepositoryInterface $invoiceRepository,
        AdressRepositoryInterface $adressRepository,
        CityRepositoryInterface $cityRepository,
        DistrictRepositoryInterface $districtRepository,
        CountryRepositoryInterface $countryRepository
    )
    {
        parent::__construct();
        $this->orderdetailRepository = $orderdetailRepository;
        $this->orderpaymentRepository = $orderpaymentRepository;
        $this->advRepository = $advRepository;
        $this->invoiceRepository = $invoiceRepository;
        $this->adressRepository = $adressRepository;
        $this->cityRepository = $cityRepository;
        $this->districtRepository = $districtRepository;
        $this->countryRepository = $countryRepository;
    }

    public function handlePaymentOrder($user, $order)
    {
        $requestData = $this->getRequestData($user, $order);
        $response = $this->createRotaInvoiceAPI($requestData);
        if (is_array($response) && isset($response['error']) && $response['error']) {
            $this->messages->error(trans('visiosoft.module.rota::message.create_invoice_error'));
        } else {
            $invoiceId = $this->createInvoice($response);

            $this->messages->success(trans('visiosoft.module.rota::message.create_invoice_success', [
                'id' => $invoiceId
            ]));
        }
    }

    public function getRequestData($user, $order)
    {
        $addressData = $this->getAddress($user->id);
        $ordersDetails = $this->orderdetailRepository->newQuery()->where('order_id', $order->id)->get();

        $taxRate = setting_value('visiosoft.module.rota::tax_rate');

        $productsData = array();
        $total = 0;
        foreach ($ordersDetails as $orderDetail) {
            $adv = $this->advRepository->find($orderDetail->item_id);

            $total = $total + $adv->price;

            $productsData[] = [
                'itemcode[]' => $adv->id,
                'description[]' => urlencode($adv->name),
                'qty[]' => $orderDetail->piece,
                'unit[]' => 'PIECE',
                'amount[]' => $adv->price,
                'tax[]' => $adv->price * $taxRate / 100,
                'taxrate[]' => $taxRate,
            ];
        }

        $requestData = [
            'api_key' => setting_value('visiosoft.module.rota::api_key'),
            'method' => 'invoices',
            'code' => $user->username,
            'name' => urlencode($user->first_name . ' ' . $user->last_name),
            'address' => urlencode($addressData['address']),
            'state' => urlencode($addressData['state']),
            'city' => urlencode($addressData['city']),
            'country' => urlencode($addressData['country']),
            'vkn' => $user->identification_number ?
                $user->identification_number :
                setting_value('visiosoft.module.rota::vkn'),
            'vdairesi' => setting_value('visiosoft.module.rota::vdairesi'),
            'email' => $user->email,
            'nationality' => $addressData['country'],
            'kdvdh' => setting_value('visiosoft.module.rota::including_item_tax') ? 1 : 0,
            'date' => urlencode(date('Y-m-d H:i:s')),
            'duedate' => date('Y-m-d'),
            'invdescription' => urlencode($adv->name),
            'total' => $total,
            'tax' => $total * $taxRate / 100,
            'discount' => setting_value('visiosoft.module.rota::discount'),
            'currency' => $orderDetail->currency,
            'currencyvalue' => setting_value('visiosoft.module.rota::currency_value'),
        ];

        return ['requestData' => $requestData, 'productsData' => $productsData];
    }

    public function createRotaInvoiceAPI($requestInfo)
    {
        $url = setting_value('visiosoft.module.rota::api_url');
        // Add general data
        foreach ($requestInfo['requestData'] as $key => $value) {
            $url .= "&$key=$value";
        }

        // Add products data
        foreach ($requestInfo['productsData'] as $productData) {
            foreach ($productData as $datumName => $datumValue) {
                $url .= "&$datumName=$datumValue";
            }
        }

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
        ));

        $response = curl_exec($curl);
        $err = curl_error($curl);

        curl_close($curl);

        if ($err) {
            return ['error' => true];
//            dd("cURL Error #:" . $err);
        } else {
            $response = json_decode($response);
            if (isset($response->error) && $response->error) {
                return ['error' => true];
            } else {
                return $response;
            }
        }
    }

    public function createInvoice($response)
    {
        $invoice = $this->invoiceRepository->create([
            'invoice_id' => $response->id
        ]);
        return $invoice->id;
    }

    public function getAddress($userId)
    {
        $addressData = array();

        $address = $this->adressRepository->findBy('user_id', $userId);

        $addressData['address'] = $address->adress_name . ' / ' . $address->content;
        $addressData['state'] = $this->cityRepository->find($address->city)->name;
        $addressData['city'] = $this->districtRepository->find($address->district)->name;
        $addressData['country'] = $this->countryRepository->find($address->country_id)->name;

        foreach ($addressData as $index => $value) {
            $addressData[$index] = str_replace(" ","%20", $value);
        }

        return $addressData;
    }
}
