<?php namespace Visiosoft\RotaModule;

use Anomaly\Streams\Platform\Addon\Module\Module;
use Carbon\Carbon;

use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Visiosoft\RotaModule\Customer\Contract\CustomerRepositoryInterface;
use Visiosoft\RotaModule\Events\CustomerCreated;
use Visiosoft\RotaModule\Events\OrderCreated;
use Visiosoft\RotaModule\Events\ProductCreated;
use Visiosoft\RotaModule\Events\ReceiptCreated;
use Visiosoft\RotaModule\Invoice\Contract\InvoiceRepositoryInterface;
use Visiosoft\RotaModule\Product\Contract\ProductRepositoryInterface;
use Visiosoft\RotaModule\Receipt\Contract\ReceiptRepositoryInterface;
use Visiosoft\RotaModule\Services\StockCard;

class RotaModule extends Module
{

    /**
     * @var string
     */
    protected $provide = 'invoice_provider';

    protected $tax_info_provider = 'true';

    /**
     * The navigation display flag.
     *
     * @var bool
     */
    protected $navigation = false;

    /**
     * The addon icon.
     *
     * @var string
     */
    protected $icon = 'fa fa-puzzle-piece';

    /**
     * The module sections.
     *
     * @var array
     */
    protected $sections = [
        'invoice' => [
            'buttons' => [
                'new_invoice',
            ],
        ],
    ];


    public function createCustomer($params)
    {
        $customerRepository = app(CustomerRepositoryInterface::class);
        $customer = $customerRepository->newQuery()->where('customer_id', $params['customer_id']);
        if ($params['id']) {
            $customer = $customer->where('id', $params['id']);
        }
        $customer = $customer->first();

        if (empty($params['identity_number'])) {
            $params['identity_number'] = "11111111111";
        }

        if (!$customer && $customerRepository->create($params)) {
            $customer = $customerRepository->findBy('customer_id', $params['customer_id']);
        }

        if (empty($customer->identity_number)) {
            $customer->identity_number = "11111111111";
        }

        $formatted_params = [
            "user_id" => setting_value('visiosoft.module.rota::user_id'),
            "account" => $customer->name,
            "code" => $customer->customer_id,
            "phone" => $customer->phone,
            "gsmno" => $customer->gsm,
            "vdairesi" => !empty($customer->tax_office) ? $customer->tax_office : $customer->identity_number,
            "vkn" => $customer->tax_number ?? '',
            "tckn" => !empty($customer->tax_number) ? '' : $customer->identity_number,
            "email" => $customer->email,
            "address" => $customer->address ?: 'Türkiye',
            "city" => $customer->city ?: 'Türkiye',
            "state" => $customer->state ?: 'Türkiye',
            "zip" => $customer->zipcode,
            "country" => $customer->country,
            "gname" => "Hiçbiri",
            "gid" => "0",
            "firmaid" => setting_value('visiosoft.module.rota::company_id'),
            "subeid" => [["subeid" => setting_value('visiosoft.module.rota::branch_id')]],
            "adresler" => [],
            "yetkililer" => []
        ];

        // Check is customer entered tax number and identity number
        // if not entered, create customer with default non identity group
        if (($customer->tax_number == "11111111111" || $customer->tax_number == "22222222222") &&
            $customer->identity_number == "00000000000") {
            $formatted_params['gname'] = setting_value('visiosoft.module.rota::default_non_identity_group_name', 'Şarj İstasyonu Kullanıcıları');
            $formatted_params['gid'] = setting_value('visiosoft.module.rota::default_non_identity_group_id', '2');
        }

        $create = $customerRepository->addToRota($formatted_params);
        if ($create->message == "Chk was created.") {
            $customer->rota_id = $create->result;
            $customer->sended = true;
            $customer->sended_at = Carbon::now();
            $customer->save();

            return $this->sendResponse($create->result, '');
        } else {
            throw new \Exception($create->message);
        }
    }

    public function getStocksByWarehouse($warehouse_provider_id)
    {
        $productRepository = app(ProductRepositoryInterface::class);

        $formatted_params = [
            "user_id" => setting_value('visiosoft.module.rota::user_id'),
            "company_id" => setting_value("visiosoft.module.rota::company_id"),
            "id" => 1,
            "first_date" => "1965-01-01",
            "last_date" => date("Y") . "-12-30"
        ];

        $list = $productRepository->stockListByWarehouse($formatted_params);
        if (is_array($list)) {
            return $this->sendResponse($list,'');
        }
    }

    public function createProduct($params)
    {
        $productRepository = app(ProductRepositoryInterface::class);
        $product = $productRepository->newQuery()->where('product_id', $params['product_id']);
        if ($params['id']) {
            $product = $product->where('id', $params['id']);
        }
        $product = $product->first();

        if (!$product && $productRepository->create($params)) {
            $product = $productRepository->findBy('product_id', $params['product_id']);
        }

        $formatted_params = [
            "user_id" => setting_value('visiosoft.module.rota::user_id'),
            "firmaid" => setting_value('visiosoft.module.rota::company_id'),
            "name" => $product->name,
            "unit" => $product->unit,
            "item_number" => $product->barcode,
            "barkod" => $product->barcode,
            "ozelkod" => $product->barcode,
            "description" => $product->description ?? ''
        ];

        $create = $productRepository->addToRota($formatted_params);
        if ($create->message == "Stk was created.") {
            $product->rota_id = $create->result;
            $product->sended = true;
            $product->sended_at = date("Y-m-d H:i:s");
            $product->save();

            return $this->sendResponse($create->result, '');
        } else {
            throw new \Exception($create->message);
        }
    }


    public function createInvoice($params)
    {
        $invoiceRepository = app(InvoiceRepositoryInterface::class);
        $customerRepository = app(CustomerRepositoryInterface::class);
        $productRepository = app(ProductRepositoryInterface::class);

        $invoice = $invoiceRepository->newQuery()->where('order_id', $params['order_id']);
        if ($params['id']) {
            $invoice = $invoice->where('id', $params['id']);
        }
        $invoice = $invoice->first();

        if (!$invoice) {
            $params['products'] = json_encode($params['products']);
            if ($invoiceRepository->create($params)) {
                $invoice = $invoiceRepository->findBy('order_id', $params['order_id']);
            }
        }
        $customer = $customerRepository->findBy('customer_id', $invoice->customer_id);
        $products = json_decode($invoice->products, true);

        foreach ($products as $key => $product) {
            $product_info = $productRepository->findBy('product_id', $product['id']);

            $temp_arr = [
                'urunid' => $product_info->rota_id,
                'itemcode' => $product_info->barcode,
                'description' => $product_info->description,
                'unit' => $product_info->unit,
                'qty' => $product['qty'],
                'amount' => $product['price'],
                'discount' => $product['discount'],
                'total' => $product['total'],
                "userid" => $customer->rota_id,
                'depo_id' => setting_value('visiosoft.module.rota::depot_id'),
                'depo_ad' => "",
                'taxrate' => $product["taxrate"] ?? "0",
                "kurkodu" => "TRY",
                'kurdegeri' => '1',
            ];

            $products[$key] = $temp_arr;
        }

        $formatted_params = [
            "user_id" => setting_value('visiosoft.module.rota::user_id'),
            "type" => "Sales",
            "userid" => $customer->rota_id,
            "invoicenum" => "B2B",
            "cn" => $invoice->order_id,
            "account" => $customer->name,
            "subtotal_dvz" => $invoice->subtotal,
            "discount_dvz" => $invoice->discount,
            "tax1_dvz" => $invoice->tax,
            "total_dvz" => $invoice->total,
            "subtotal" => $invoice->subtotal,
            "discount" => $invoice->discount,
            "tax1" => $invoice->tax,
            "total" => $invoice->total,
            "datecreated" => $invoice->created_at,
            "duedate" => $invoice->created_at,
            "teslimdate" => $invoice->created_at,
            "firmaid" => setting_value('visiosoft.module.rota::company_id'),
            "musteritemsilcisi" => setting_value('visiosoft.module.rota::user_id'),
            "depoid" => setting_value('visiosoft.module.rota::depot_id'),
            "depoadi" => "",
            "subeid" => setting_value('visiosoft.module.rota::branch_id'),
            "items" => $products,
            "plate" => $invoice->plate,
            "kurkodu" => "TRY",
            "kurdegeri" => "1",
            "kdvdh" => "1"
        ];

        $create = $invoiceRepository->addToRota($formatted_params);
        if ($create->message == "Order was created.") {
            $invoice->rota_id = $create->result;
            $invoice->sended = true;
            $invoice->sended_at = date("Y-m-d H:i:s");
            $invoice->save();

            return $this->sendResponse($create->result, '');
        } else {
            throw new \Exception($create->message);
        }
    }

    public function sendResponse($data, $message, $success = true)
    {
        $response = ['success' => $success, 'data' => $data, 'message' => $message];
        return $response;
    }

    public function storeInvoice($params)
    {

        if (is_null($params->user->id) || is_null($params->category) || is_null($params->id))
        {
            throw new \Exception("Missing parameters".json_encode($params));
        }

        // Dynamic parameters
        $tax = $params->payment_amount / (100 / $params->tax_percent);
        $subtotal = $params->payment_amount - $tax;
        $tax_number = "11111111111";

        // Check If Company Exists
        $is_company = !is_null($params->company_id);

        if ($is_company) {
            $tax_number = $params->company->organization_number;
        }

        $customerRepository = app(CustomerRepositoryInterface::class);
        if (!$customerRepository->findBy('customer_id', $params->user_id))
        {
            $customer_params = [
                'customer_id' => $params->user->id,
                'name' => ucwords($params->user->display_name) ?: 'Muhtelif Alıcı',
                'phone' => $params->user->phone ?: '+905555555555',
                'tax_office' => $params->user->city ?: 'İstanbul',
                'tax_number' => $tax_number,
                'identity_number' => "00000000000",
                'email' => $params->user->email ?: setting_value("streams::email"),
                'address' => $params->user->extended_address ?: '',
                'city' => $params->user->city ?: 'No City',
                'state' => "",
                'zipcode' => $params->user->zip_code ?: '34000',
                'country' => $params->user->country ?: 'Türkiye'
            ];

            try {
                event(new CustomerCreated($customer_params));
            }catch (\Exception $e){
                $this->logError($e);
                return false;
            }
        }

        // First check if default stock (product) exists in our database
        $defaultProductId = setting_value('visiosoft.module.rota::default_charging_stock_code');
        $defaultProduct = app(ProductRepositoryInterface::class)->findBy('rota_id', $defaultProductId);

        $product_params = [
            'product_id' => $params->id,
            'name' => setting_value("visiosoft.module.rota::default_stock_name","Şarj Hizmeti Satışı"),
            'barcode' => setting_value("visiosoft.module.rota::default_stock_code","600.02.001"),
            'unit' => "kwh",
            'description' => $params->plate." - ".$params->category->name." - ".$params->payment_amount." ".$params->currency
        ];
        $invoice_params = [
            'customer_id' => $params->user->id,
            'order_id' => $params->id,
            'products' => [
                [
                    "id" => $defaultProduct->product_id ?? $params->id,
                    "qty" => $params->usage,
                    "price" => $params->payment_amount / floatval($params->usage),
                    'discount' => 0,
                    'total' => $params->payment_amount,
                    'taxrate' => $params->tax_percent,
                    'kurdegeri' => '1',
                ]
            ],
            'subtotal' => $subtotal,
            'discount' => 0,
            'tax' => $tax,
            'total' => $params->payment_amount,
            'plate' => $params->plate
        ];

        try {
            if (!$defaultProduct) {
                event(new ProductCreated($product_params));
            }
            event(new OrderCreated($invoice_params));
        } catch (\Exception $e) {
            $this->logError($e);
            return false;
        }

        return true;
    }

    public function storeInvoiceWithParams ($id, $amount, $quantity, $tax_percent, $user_id, $display_name, $phone,
                                            $city, $email, $extended_address, $zip_code, $country,
                                            $organization_number = "11111111111") {

        if (is_null($user_id) || is_null($id))
        {
            throw new \Exception("Missing parameters ($user_id, $id)");
        }

        // Dynamic parameters
        $tax = $amount / (100 / $tax_percent);
        $subtotal = $amount - $tax;

        $customerRepository = app(CustomerRepositoryInterface::class);
        if (!$customerRepository->findBy('customer_id', $user_id))
        {
            $customer_params = [
                'customer_id' => $user_id,
                'name' => ucwords($display_name) ?: 'Muhtelif Alıcı',
                'phone' => $phone ?: '+905555555555',
                'tax_office' => $city ?: 'İstanbul',
                'tax_number' => $organization_number,
                'identity_number' => "00000000000",
                'email' => $email ?: setting_value("streams::email"),
                'address' => $extended_address ?: '',
                'city' => $city ?: 'No City',
                'state' => "",
                'zipcode' => $zip_code ?: '34000',
                'country' => $country ?: 'Türkiye'
            ];

            try {
                event(new CustomerCreated($customer_params));
            }catch (\Exception $e){
                $this->logError($e);
                return false;
            }
        }

        // First check if default stock (product) exists in our database
        $defaultProductId = setting_value('visiosoft.module.rota::default_charging_stock_code');
        $defaultProduct = app(ProductRepositoryInterface::class)->findBy('rota_id', $defaultProductId);

        $product_params = [
            'product_id' => $id,
            'name' => setting_value("visiosoft.module.rota::default_stock_name","Şarj Hizmeti Satışı"),
            'barcode' => setting_value("visiosoft.module.rota::default_stock_code","600.02.001"),
            'unit' => "kwh",
            'description' => setting_value("visiosoft.module.rota::default_stock_name","Şarj Hizmeti Satışı")
        ];
        $invoice_params = [
            'customer_id' => $user_id,
            'order_id' => $id,
            'products' => [
                [
                    "id" => $defaultProduct->product_id ?? $id,
                    "qty" => $quantity,
                    "price" => $amount / floatval($quantity),
                    'discount' => 0,
                    'total' => $amount,
                    'taxrate' => $tax_percent,
                    'kurdegeri' => '1',
                ]
            ],
            'subtotal' => $subtotal,
            'discount' => 0,
            'tax' => $tax,
            'total' => $amount,
        ];

        try {
            if (!$defaultProduct) {
                event(new ProductCreated($product_params));
            }
            event(new OrderCreated($invoice_params));
        } catch (\Exception $e) {
            $this->logError($e);
            return false;
        }

        return true;
    }

    public function storeReceipt($params){

        if (is_null($params['id']) || is_null($params['payment_amount']) || is_null($params->user->id))
        {
            throw new \Exception("Missing parameters".json_encode($params));
        }

        // find payers rota_id
        $customerRepository = app(CustomerRepositoryInterface::class);
        $customer = $customerRepository->findBy('customer_id', $params->user->id);

        if (!$customer && $customerRepository->rota_id) {
            throw new \Exception("Customer not found");
        }
        $data = [
            "amount" => $params['payment_amount'],
            "payer_id" => $customer->rota_id,
            "description" => $params['description'] ?? 'Virman Fişi',
            "order_id" => $params['id']
        ];


        try {
            event(new ReceiptCreated($data));
        } catch (\Exception $e) {
            $this->logError($e);
            return false;
        }

        return true;
    }

    public function createReceipt($params) {
        $receiptRepository = app(ReceiptRepositoryInterface::class);
        $customerRepository = app(CustomerRepositoryInterface::class);
        // this value is the information of the current record on rota
        $payee_const = [
            'id' => setting_value('visiosoft.module.rota::payee_id', 37641),
            'name' => setting_value('visiosoft.module.rota::payee_name','SİPAY ELEKTRONİK PARA VE ÖDEME HİZMETLERİ ANONİM ŞİRKETİ'),
        ];
        // get receipt information
        $receipt = $receiptRepository->newQuery()->where('order_id', $params['order_id'])->orWhere('id', $params['id'])->orderBy('id', 'asc')->first();

        // if receipt not found
        if (!$receipt) {
            throw new \Exception($params['id'] .' ID - receipt not found with this ID.');
        }
        // if receipt sended already
        if ($receipt->sended) {
            throw new \Exception($params['id'] .' ID - receipt already sended.');
        }

        // get customer information
        $customer = $customerRepository->findBy('customer_id', $receipt->payer_id);
        // to format date
        $receipt_date = $receipt->created_at;

        // format params
        $formatted_params = [
            "user_id" => setting_value('visiosoft.module.rota::user_id'),
            "account" => $customer->name ?? "Muhtelif Alıcı",
            "accname" => "",
            "accdes" => "",
            "type" => "Income",
            "code" => $receipt->id,
            "category" => "CH",
            "amount" => $receipt->amount,
            "payer" => $customer->name ?? "Muhtelif Alıcı",
            "payee" => $payee_const['name'],
            "payerid" => $receipt->payer_id,
            "payeeid" => $payee_const['id'],
            "description" => $receipt->description ?? "Virman Fişi",
            "tags" => "virman-fisi",
            "date" => $receipt_date->format('Y-m-d'),
            "time" => $receipt_date->format('Y-m-d H:i:s'),
            "dr" => $receipt->amount,
            "cr" => $receipt->amount,
            "firmaid" => setting_value('visiosoft.module.rota::company_id'),
            "kurkodu" => "TRY",
            "kurdegeri" => 1.00000,
            "amount_dvz" => $receipt->amount,
            "ozelkod" => "OZELKOD",
            "entegrasyon_turu" => "cari_virman",
            "virman" => 0,
            "fis_turu" => "cari",
            "alternatif_doviz_secimi" => "0",
            "alternatif_kurkodu" => "TRY",
            "alternatif_kurdegeri" => 1.00000,
            "alternatif_amount" => null,
            "subeid" => setting_value('visiosoft.module.rota::branch_id')
        ];

        $create = $receiptRepository->addToRota($formatted_params);

        if (str_contains(strtolower($create->message), 'finance receipt was created')) {
            $receipt->rota_id = $create->result;
            $receipt->sended = true;
            $receipt->sended_at = date("Y-m-d H:i:s");
            $receipt->save();

            return $this->sendResponse($receipt->result, '');
        } else {
            throw new \Exception($receipt->message);
        }

    }

    public function logError ($e) {
        $log = new Logger('rota');
        $log->pushHandler(new StreamHandler(storage_path('logs/rota.log')), Logger::ERROR);
        $log->error($e);
    }

    public function getTaxInformation($taxNumber)
    {
        $userId = setting_value('visiosoft.module.rota::user_id');
        $companyId = setting_value('visiosoft.module.rota::company_id');

        $customerRepository = app(CustomerRepositoryInterface::class);

        $query = [
            'user_id' => $userId,
            'company_id' => $companyId,
            'vkn' => $taxNumber
        ];

        return $customerRepository->getTaxInformation($query);
    }

}
