<?php namespace Visiosoft\ReportsModule\Report;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Visiosoft\ReportsModule\Helpers\FormattersHelper;
use Visiosoft\ReportsModule\Helpers\QueryHelper;
use Visiosoft\ReportsModule\Report\Contract\ReportInterface;
use Visiosoft\ReportsModule\Report\Contract\ReportRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use function GuzzleHttp\Psr7\str;

class ReportRepository extends EntryRepository implements ReportRepositoryInterface
{
    /**
     * The entry model.
     *
     * @var ReportModel
     */
    protected $model;

    private ReportInterface $report;

    /**
     * Create a new ReportRepository instance.
     *
     * @param ReportModel $model
     */
    public function __construct(ReportModel $model)
    {
        $this->model = $model;
    }

    /**
     * @return Builder[]|Collection
     */
    public function getReports(): Collection|array
    {
        return $this->model->newQuery()->get();
    }

    public function getReport(int $id)
    {
        return $this->model->find($id);
    }

    private function getDataWithPredefinedColumns(ReportInterface $report): array|\Illuminate\Support\Collection
    {
        $sub_duration = Carbon::now()->sub($report->getDuration() . $report->getDurationType());
        $query = DB::table($report->getReportTable());
        $db_connection = QueryHelper::getDbConnection();

        //TODO::It's hard to read, can something be done?
        if ($db_connection == 'pgsql') {
            $rawQuery = "WITH date_series AS (
                                            SELECT generate_series(
                                                       date_trunc('" . $report->getDurationType() . "', '" . $sub_duration . "'::date),
                                                       '" . Carbon::now() . "'::date,
                                                       '1 " . $report->getDurationType() . "'::interval
                                                   )::date AS timeLevel
                                            )
                                            SELECT 
                                                ds.timeLevel as time,  
                                                COALESCE(COUNT(duu.*), 0) AS total
                                            FROM 
                                                date_series ds
                                            LEFT JOIN 
                                                default_" . $report->getReportTable() . " duu
                                            ON 
                                                date_trunc('" . $report->getDurationType() . "', duu." . $report->getTimeColumn() . ")::date = ds.timeLevel";
            if ($report->getWhere()) {
                $rawQuery .= " WHERE " . $report->getWhere();
            }

            $rawQuery .= " GROUP BY ds.timeLevel
                            ORDER BY ds.timeLevel";
            $query = DB::select($rawQuery);
        } else {
            $query->selectRaw($report->getDurationType() . "(" . $report->getTimeColumn() . ") as time, " . $report->getOperation() . "(" . $report->getColumn() . ") as total");
            $query->whereBetween($report->getTimeColumn(), [$sub_duration, Carbon::now()])
                ->groupBy('time');

            if ($report->getWhere()) {
                $query->whereRaw($report->getWhere());
            }
            $query = $query->get();
        }
        return FormattersHelper::formatByOperation($query, $report->getOperation());
    }

    private function getDataWithRawQuery(ReportInterface $report)
    {
        return DB::select($report->getRawQuery());
    }

    public function getDataByLocal(ReportInterface $report)
    {
        return $report->getReportType() === "raw_query" ? $this->getDataWithRawQuery($report) : $this->getDataWithPredefinedColumns($report);
    }

    /**
     * @throws \Exception
     */
    public function getReportByLocal(int $id): array|\Illuminate\Support\Collection
    {
        $report = $this->model->find($id);
        if (!$report) {
            throw new \Exception("Report Not Found");
        }

        if ($report->getSource() != "localhost") {

            return $this->getDataByRemote($report);
        }

        return $this->getDataByLocal($report);
    }

    public function getDataByRemote(ReportInterface $report)
    {
        $url = FormattersHelper::formatSource($report->getSource()) . "/api/report/get/" . $report->getSourceAuthToken();
        $reportParams = $report->toArray();

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $reportParams);

        curl_setopt($ch, CURLOPT_FRESH_CONNECT, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);
        curl_setopt($ch, CURLOPT_HTTPHEADER,
            [
                'Accept:application/json'
            ]);

        $result = curl_exec($ch);
        curl_close($ch);
        return collect(json_decode($result));
    }
}
