<?php namespace Visiosoft\ReportsModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Visiosoft\ReportsModule\Helpers\QueryHelper;
use Visiosoft\ReportsModule\Report\Contract\ReportRepositoryInterface;
use Visiosoft\ReportsModule\Report\ReportModel;
use Visiosoft\ReportsModule\Token\Contract\TokenRepositoryInterface;
use Visiosoft\ReportsModule\Traits\ApiReturnResponseTrait;

class ReportController extends ResourceController
{
    use ApiReturnResponseTrait;

    private $reportsRepository;
    private $tokenRepository;

    public function __construct()
    {
        $this->reportsRepository = app(ReportRepositoryInterface::class);
        $this->tokenRepository = app(TokenRepositoryInterface::class);
    }

    public function getReports()
    {
        return $this->reportsRepository->getReports();
    }

    /**
     * @param $id
     * @return mixed
     */
    public function getReportByLocal($id)
    {
        $entry = $this->reportsRepository->getReportByLocal($id);
        $childs = $this->reportsRepository->getChilds($id);

        $response = ['entry' => $entry, 'childs' => $childs];

        return $response;
    }

    public function getReportByRemote($token)
    {
        $this->checkAuth($token);
        $validator = Validator::make(request()->all(), [
            'report_type' => ['required', 'max:255', Rule::in(['raw_query', 'predefined_columns'])],
            'report_table' => ['max:255', 'string'],
            'column' => ['max:255', 'string'],
            'time_column' => ['max:255', 'string'],
            'operation' => ['max:255', Rule::in(['sum', 'count', 'SUM', 'COUNT'])],
            'duration_type' => ['max:255', Rule::in('day', 'week', 'month', 'DAY', 'WEEK', 'MONTH')],
            'duration' => ['max:255', 'int'],
            'where' => ['string'],
            'raw_query' => ['string'],
        ]);

        if ($validator->fails()) {
            return $validator->errors();
        }

        $request = request()->post();
        $report = new ReportModel();
        $report->setReportTable($request['report_table']);
        $report->setReportType($request['report_type']);

        if ($request['report_type'] === "raw_query") {
            if (!empty($request['raw_query']) && QueryHelper::isBlacklisted($request['raw_query'])) {
                return $this->sendError('There are blacklisted statements in your query', [], 400);
            }

            $report->setRawQuery($request['raw_query']);
        } else {
            $report->setReportTable($request['report_table']);
            $report->setColumn($request['column']);
            $report->setTimeColumn($request['time_column']);
            $report->setOperation($request['operation']);
            $report->setDurationType($request['duration_type']);
            $report->setDuration($request['duration']);
            $report->setWhere($request['where']);
        }

        return $this->reportsRepository->getDataByLocal($report);
    }

    private function checkAuth($token)
    {
        if (!$token) {
            throw new \Exception('token-required');
        }

        if (!$token = $this->tokenRepository->findBy('token', $token)) {

            throw new \Exception('invalid-token');
        }

        if (!$token->status || $token->expire_date < Carbon::now()) {
            throw new \Exception('invalid-token');
        }


    }

}
