<?php namespace Visiosoft\ReportsModule\Report;

use Carbon\Carbon;
use DateTime;
use Illuminate\Support\Facades\DB;
use Visiosoft\PaymentPaytrExtension\Helpers\Settings;
use Visiosoft\ReportsModule\Report\Contract\ReportRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Visiosoft\ReportsModule\Token\Contract\TokenRepositoryInterface;
use function GuzzleHttp\Psr7\str;

class ReportRepository extends EntryRepository implements ReportRepositoryInterface
{

    /**
     * The entry model.
     *
     * @var ReportModel
     */
    protected $model;
    private Carbon $now;

    private object $report;

    /**
     * Create a new ReportRepository instance.
     *
     * @param ReportModel $model
     */
    public function __construct(ReportModel $model)
    {
        $this->model = $model;
        $this->now = Carbon::now();
    }

    /**
     * @param string $table
     * @param string $column
     * @param string $time_column
     * @param string $operation
     * @param string $duration_type
     * @param int $duration
     * @param string|null $where
     * @param string|null $source
     * @param string|null $source_auth_token
     * @return void
     */
    public function setReport(
        string $table,
        string $column,
        string $time_column,
        string $operation,
        string $duration_type,
        int    $duration,
        string $where = null,
        string $source = null,
        string $source_auth_token = null
    )
    {
        $this->report = (object)[
            'table' => $table,
            'column' => $column,
            'time_column' => $time_column,
            'operation' => $operation,
            'duration_type' => $duration_type,
            'duration' => $duration,
            'source' => $source,
            'source_auth_token' => $source_auth_token,
            'where' => $where,
        ];
    }

    /**
     * @return string
     */
    public function getTable(): string
    {
        return $this->report->table;
    }

    /**
     * @return string
     */
    public function getColumn(): string
    {
        return $this->report->column;
    }

    /**
     * @return string
     */
    public function getSource(): string
    {
        if (!str_contains($this->report->source, 'https://')) {
            $this->report->source = "https://" . $this->report->source;
        }

        if (substr($this->report->source, -1) != "/") {
            $this->report->source .= "/";
        }

        return $this->report->source;
    }

    /**
     * @return string|null
     */
    public function getSourceAuthToken(): string
    {
        return $this->report->source_auth_token;
    }

    /**
     * @return string|null
     */
    public function getWhere()
    {
        return $this->report->where;
    }

    /**
     * @return string
     */
    public function getTimeColumn(): string
    {
        return $this->report->time_column;
    }

    /**
     * @return string
     */
    public function getOperation(): string
    {
        return $this->report->operation;
    }

    /**
     * @return string
     */
    public function getDurationType(): string
    {
        return $this->report->duration_type;
    }

    /**
     * @return int
     */
    public function getDuration(): int
    {
        return $this->report->duration;
    }

    /**
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public function getReports()
    {
        return $this->model->newQuery()->get();
    }

    public function getReport(): object
    {
        return $this->report;
    }

    public function getDataByLocal()
    {
        $sub_duration = Carbon::now()->sub($this->getDuration() . $this->getDurationType());
        $query = DB::table($this->getTable());
        $db_connection = config('database.default');

        //Check db driver
        if ($db_connection == 'pgsql') {
            $rawQuery = "WITH date_series AS (
                                            SELECT generate_series(
                                                       date_trunc('" . $this->getDurationType() . "', '".$sub_duration."'::date),
                                                       '".$this->now."'::date,
                                                       '1 " . $this->getDurationType() . "'::interval
                                                   )::date AS timeLevel
                                            )
                                            SELECT 
                                                ds.timeLevel as time,  
                                                COALESCE(COUNT(duu.*), 0) AS total
                                            FROM 
                                                date_series ds
                                            LEFT JOIN 
                                                default_".$this->getTable()." duu
                                            ON 
                                                date_trunc('" . $this->getDurationType() . "', duu." . $this->getTimeColumn() . ")::date = ds.timeLevel";
            if ($this->getWhere()) {
                $rawQuery .= " WHERE " . $this->getWhere();
            }

            $rawQuery .= " GROUP BY ds.timeLevel
                            ORDER BY ds.timeLevel";                                           
            $query =  DB::select( $rawQuery );
        } else {
            $query->selectRaw($this->getDurationType() . "(" . $this->getTimeColumn() . ") as time, " . $this->getOperation() . "(" . $this->getColumn() . ") as total");
            $query->whereBetween($this->getTimeColumn(), [$sub_duration, $this->now])
                ->groupBy('time');

            if ($this->getWhere()) {
                $query->whereRaw($this->getWhere());
            }
            $query = $query->get();
        }
        return $this->format($query);
    }

    public function getReportByLocal($id)
    {
        $report = $this->model->find($id);
        if (!$report) {
            throw new \Exception("Report Not Found");
        }

        $this->setReport(
            $report->table,
            $report->column,
            $report->time_column,
            $report->operation,
            $report->duration_type,
            $report->duration,
            $report->where,
            $report->source,
            $report->source_auth_token,
        );

        if ($report->source != "localhost") {
            return $this->getDataByRemote();
        }

        return $this->getDataByLocal();
    }

    public function getDataByRemote()
    {
        $url = $this->getSource() . "/api/report/get/" . $this->getSourceAuthToken();
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, (array)$this->getReport());

        curl_setopt($ch, CURLOPT_FRESH_CONNECT, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);
        curl_setopt($ch, CURLOPT_HTTPHEADER,
            [
                'Accept:application/json'
            ]);

        $result = curl_exec($ch);
        curl_close($ch);
        return collect(json_decode($result));
    }

    private function format($query)
    {
        foreach ($query as $q) {
            if (strtolower($this->getOperation()) === "sum") {
                $q->total = number_format($q->total, 2, '.', '');
            }
        }
        return $query;
    }
}
