<?php namespace Visiosoft\ReferencesModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Anomaly\UsersModule\Role\Contract\RoleRepositoryInterface;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Anomaly\UsersModule\User\UserActivator;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Visiosoft\ReferencesModule\Events\ReferencedUserWasCreated;
use Visiosoft\ReferencesModule\Events\ReferenceWasCreated;
use Visiosoft\ReferencesModule\Reference\Contract\ReferenceRepositoryInterface;

class ApiController extends PublicController
{
    private $referenceRepository;

    public function __construct(ReferenceRepositoryInterface $referenceRepository)
    {
        parent::__construct();
        $this->referenceRepository = $referenceRepository;
    }

    public function getReferences()
    {
        $references = $this->referenceRepository->getUserReferences(request()->user_id);
        $pluckedReferences = array();
        foreach ($references as $reference) {
            $pluckedReferences[$reference->user_id] = $reference->user->name();
        }
        return $pluckedReferences;
    }

    public function register(
        UserRepositoryInterface $userRepository,
        UserActivator $activator,
        RoleRepositoryInterface $roleRepository
    )
    {
        try {
            $validator = Validator::make(request()->all(), [
                'first_name' => 'required|string|max:50',
                'last_name' => 'required|string|max:50',
                'email' => 'required|email|unique:users_users',
            ]);

            if ($validator->fails()) {
                $errMessages = $validator->errors()->getMessages();
                throw new \Exception(reset($errMessages)[0]);
            }

            $userRole = $roleRepository->findBySlug('user');
            $firstName = request()->first_name;
            $lastName = request()->last_name;
            $user = $userRepository->create([
                'username' => $firstName . Str::random(8),
                'first_name' => $firstName,
                'last_name' => $lastName,
                'display_name' => "$firstName $lastName",
                'email' => request()->email,
                'activated' => 0,
            ]);
            $password = Str::random(8);
            $user->setAttribute('password', $password);
            $user->setAttribute('roles', $userRole);
            $userRepository->save($user);

            $reference = $this->referenceRepository->createReference([
                'user' => $user,
                'referenced_user_id' => auth()->id(),
            ]);

            $activator->start($user);

            $activationLink = $user->route('activate', ['redirect' => route('profile::profile')]);

            event(new ReferenceWasCreated($reference));
            event(new ReferencedUserWasCreated($reference, $activationLink, $password));

            return [
                'success' => true,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}
