<?php

namespace Visiosoft\PushnotificationsModule\Notification\Command;

use Visiosoft\PushnotificationsModule\PushnotificationsModule;
use Visiosoft\PushnotificationsModule\Template\TemplateRepository;

class PushNotification
{
    protected object $template;
    protected array $titles;
    protected array $messages;
    protected bool $status = false;

    /**
     * @param $template string template name to use for notification message and title
     */
    public function __construct(string $template)
    {
        $this->setTemplate($template);
        $this->setContent();
        $this->setStatus($this->template->enabled);
    }

    /**
     * @return array returns array of users to send notifications to
     */
    public function getTitles(): array
    {
        return $this->titles;
    }

    /**
     * @param array $titles
     * @return object|$this sets $this->titles to $titles
     */
    public function setTitles(array $titles): object
    {
        $this->titles = $titles;
        return $this;
    }

    /**
     * @return array returns array of messages to send to users
     */
    public function getMessages(): array
    {
        return $this->messages;
    }

    /**
     * @param array $messages array of messages to send to users
     * @return object|$this
     */
    public function setMessages(array $messages): object
    {
        $this->messages = $messages;
        return $this;
    }

    /**
     * @return object returns template object
     */
    public function getTemplate(): object
    {
        return $this->template;
    }


    /**
     * @param string $template template slug to get from database (e.g. 'start_park_notification')
     * @return object|$this $this->template object with template from database by slug $template
     * @return void sets
     */
    private function setTemplate(string $template): void
    {
        $templateRepository = app(TemplateRepository::class);
        $this->template = $templateRepository->getTemplateBySlug($template);
        if ($this->template) {
            $template = $this->template->toArray();
            $this->setTemplateTitles($template);
            $this->setTemplateMessages($template);
        }
    }

    /**
     * @param array $template
     * @return void
     */
    private function setTemplateMessages(array $template): void
    {
        if (!empty($template)) {
            foreach ($this->template->translations as $translation) {
                $this->titles[$translation->locale] = $translation->message;
            }
        }
    }

    private function setTemplateTitles(array $template): void
    {
        if (!empty($template)) {
            foreach ($this->template->translations as $translation) {
                $this->messages[$translation->locale] = $translation->title;
            }
        }
    }

    /**
     * @return bool returns true if notification is enabled, false otherwise
     */
    public function getStatus(): bool
    {
        return $this->status;
    }

    /**
     * @param bool $status true to enable notification, false to disable notification
     * @return object|$this sets $this->status to $status
     */
    public function setStatus(bool $status): object
    {
        $this->status = $status;
        return $this;
    }

    /**
     * @return void sets $this->titles and $this->messages from $this->template
     */
    private function setContent()
    {
        $titles = [];
        $messages = [];

        foreach ($this->template->translations as $translation) {
            if ($translation->title != null) {
                $titles[$translation->locale] = $translation->title;
            }

            if ($translation->message != null) {
                $messages[$translation->locale] = $translation->message;
            }
        }

        $this->setTitles($titles);
        $this->setMessages($messages);
    }

    /**
     * @param string $key key to replace in template (e.g. 'parking_name')
     * @param string $value value to replace in template (e.g. 'My Parking')
     * @return $this replaces $key with $value in template title and message
     */
    public function setVariable(string $key, string $value)
    {
        foreach ($this->messages as $locale => $message) {
            if ($message != null) {
                $this->messages[$locale] = str_replace($key, $value, $this->messages[$locale]);
            }
        }
        foreach ($this->titles as $locale => $title) {
            if ($title != null) {
                $this->titles[$locale] = str_replace($key, $value, $this->titles[$locale]);
            }
        }
        return $this;
    }

    /**
     * @param array $users array of user ids to send notification to
     * @return bool returns true if notification is sent successfully, false otherwise
     * @throws \Exception
     */
    public function sendPushNotification(array $users): bool
    {
        if (!$this->getStatus()) {
            return false;
        }

        $users = array_map('strval', $users);

        $contents = [
            'titles' => $this->getTitles(),
            'messages' => $this->getMessages(),
        ];

        $pushNotificationsModule = new PushnotificationsModule();
        return $pushNotificationsModule->send($contents, $users);
    }

}