<?php namespace Visiosoft\PaymentStripeModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Visiosoft\CartsModule\Cart\Command\GetCart;
use Visiosoft\CartsModule\Cart\Command\TotalCart;
use Visiosoft\OrdersModule\Http\Traits\TransactionTrait;
use Visiosoft\OrdersModule\Order\Contract\OrderRepositoryInterface;
use Visiosoft\PaymentStripeModule\PaymentStripeModule;

class StripeController extends PublicController
{
    use TransactionTrait;

    private $orderRepository;

    public function __construct(OrderRepositoryInterface $orderRepository)
    {
        parent::__construct();
        $this->orderRepository = $orderRepository;
    }

    public function paymentForm()
    {
        try {
            $cart = $this->dispatch(new GetCart());
            $items = $cart->getItems();
            $publishableAPIKey = setting_value('visiosoft.module.payment_stripe::publishable_api_key');
            $secretAPIKey = setting_value('visiosoft.module.payment_stripe::secret_api_key');

            if (!$publishableAPIKey || !$secretAPIKey) {
                throw new \Exception(trans('visiosoft.module.payment_stripe::message.uncompleted_stripe_fields'));
            }

            if (count($items) > 1) {
                throw new \Exception(trans('visiosoft.module.payment_stripe::message.only_one_product_error'));
            }

            if ($items[0]->entry_type !== 'Visiosoft\PackagesModule\Package\PackageModel') {
                $model = app($items[0]->entry_type);
                $modelItem = $model->find($items[0]->entry_id);
                if (!$modelItem->plan_id) {
                    throw new \Exception(trans('visiosoft.module.payment_stripe::message.product_not_supported_error'));
                }
            } else {
                $model = app($items[0]->entry_type);
                $userModel = app('Visiosoft\PackagesModule\User\UserModel');
                $modelItem = $userModel->find($items[0]->entry_id);
                $modelItem = $model->find($modelItem->package_id);
                if (!$modelItem->plan_id) {
                    throw new \Exception(trans('visiosoft.module.payment_stripe::message.product_not_supported_error'));
                }
            }

            \Stripe\Stripe::setApiKey(setting_value('visiosoft.module.payment_stripe::secret_api_key'));

            $checkout_session = \Stripe\Checkout\Session::create([
                'line_items' => [[
                    'price' => $modelItem->plan_id,
                    'quantity' => 1,
                ]],
                'mode' => 'subscription',
                'success_url' => url('payment/stripe/success') . '?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => route('visiosoft.module.carts::carts.review'),
            ]);

            return redirect($checkout_session->url);
        } catch (\Exception $e) {
            // Log CC transaction
            $this->logTransaction([
                'payment_type' => PaymentStripeModule::TYPE,
                'message' => $e->getMessage(),
                'payment_order' => request()->cookie('order_id'),
            ]);

            $this->messages->error($e->getMessage());

            return redirect()->route('visiosoft.module.carts::carts.review');
        }
    }

    public function paymentSuccess()
    {
        try {
            $cart = $this->dispatch(new GetCart());
            $this->dispatch(new TotalCart($cart));
            $order_id = \request()->cookie('order_id');

            \Stripe\Stripe::setApiKey(setting_value('visiosoft.module.payment_stripe::secret_api_key'));

            $checkout_session = \Stripe\Checkout\Session::retrieve(request('session_id'));

            if ($checkout_session->payment_status !== 'paid' || $checkout_session->status !== 'complete') {
                throw new \Exception(trans('visiosoft.module.payment_stripe::message.an_error_occurred'));
            }

            // Log CC transaction
            $this->logTransaction([
                'payment_type' => PaymentStripeModule::TYPE,
                'success' => true,
                'payment_order' => $order_id,
            ]);

            $this->orderRepository->approveOrder($order_id);

            return redirect('payment/success/stripe');
        } catch (\Exception $e) {
            // Log CC transaction
            $this->logTransaction([
                'payment_type' => PaymentStripeModule::TYPE,
                'message' => $e->getMessage(),
                'payment_order' => $order_id,
            ]);

            $this->orderRepository->rejectOrder($order_id);

            $this->messages->error($e->getMessage());

            return redirect(route('visiosoft.module.carts::carts.review'));
        }
    }
}
