<?php namespace Visiosoft\PaymentStripeModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Visiosoft\CartsModule\Cart\Command\GetCart;
use Visiosoft\CartsModule\Cart\Command\TotalCart;
use Visiosoft\OrdersModule\Http\Traits\TransactionTrait;
use Visiosoft\OrdersModule\Order\Contract\OrderRepositoryInterface;
use Visiosoft\PaymentStripeModule\PaymentStripeModule;

class StripeController extends PublicController
{
    use TransactionTrait;

    private $orderRepository;

    public function __construct(OrderRepositoryInterface $orderRepository)
    {
        parent::__construct();
        $this->orderRepository = $orderRepository;
    }

    public function paymentForm()
    {
        $orderID = request()->cookie('order_id');
        $publishableAPIKey = setting_value('visiosoft.module.payment_stripe::publishable_api_key');
        $secretAPIKey = setting_value('visiosoft.module.payment_stripe::secret_api_key');

        if ($publishableAPIKey && $secretAPIKey) {
            return $this->view->make('visiosoft.module.payment_stripe::stripe/form');
        } else {
            $errMsg = trans('visiosoft.module.payment_stripe::message.uncompleted_stripe_fields');

            // Log CC transaction
            $this->logTransaction([
                'payment_type' => PaymentStripeModule::TYPE,
                'message' => $errMsg,
                'payment_order' => $orderID,
            ]);

            $this->messages->error($errMsg);
            return redirect()->route('visiosoft.module.carts::carts.review');
        }
    }

    public function paymentStripe()
    {
        try {
            $cart = $this->dispatch(new GetCart());
            $items = $cart->getItems();

            if (count($items) > 1) {
                throw new \Exception(trans('visiosoft.module.payment_stripe::message.only_one_product_error'));
            }

            if ($items[0]->entry_type !== 'Visiosoft\PackagesModule\Package\PackageModel') {
                $model = app($items[0]->entry_type);
                $modelItem = $model->find($items[0]->entry_id);
                if (!$modelItem->plan_id) {
                    throw new \Exception(trans('visiosoft.module.payment_stripe::message.product_not_supported_error'));
                }
            } else {
                $model = app($items[0]->entry_type);
                $userModel = app('Visiosoft\PackagesModule\User\UserModel');
                $modelItem = $userModel->find($items[0]->entry_id);
                $modelItem = $model->find($modelItem->package_id);
                if (!$modelItem->plan_id) {
                    throw new \Exception(trans('visiosoft.module.payment_stripe::message.product_not_supported_error'));
                }
            }

            \Stripe\Stripe::setApiKey(setting_value('visiosoft.module.payment_stripe::secret_api_key'));

            $customer = \Stripe\Customer::create([
                'payment_method' => request()->payment_method,
                'email' => auth()->user()->email,
                'invoice_settings' => [
                    'default_payment_method' => request()->payment_method
                ]
            ]);

            return \Stripe\Subscription::create([
                'customer' => $customer->id,
                'items' => [
                    [
                        'plan' => $modelItem->plan_id,
                    ],
                ],
                'expand' => ['latest_invoice.payment_intent'],
            ]);
        } catch (\Exception $e) {
            // Log CC transaction
            $this->logTransaction([
                'payment_type' => PaymentStripeModule::TYPE,
                'message' => $e->getMessage(),
                'payment_order' => request()->cookie('order_id'),
            ]);

            return ['error' => [
                'message' => 'Error: ' . $e->getMessage()
            ]];
        }
    }

    public function paymentSuccess()
    {
        try {
            $cart = $this->dispatch(new GetCart());
            $this->dispatch(new TotalCart($cart));

            $order_id = \request()->cookie('order_id');

            // Log CC transaction
            $this->logTransaction([
                'payment_type' => PaymentStripeModule::TYPE,
                'success' => true,
                'payment_order' => $order_id,
            ]);

            $this->orderRepository->approveOrder($order_id);

            return ['success' => true];
        } catch (\Exception $e) {
            // Log CC transaction
            $this->logTransaction([
                'payment_type' => PaymentStripeModule::TYPE,
                'message' => $e->getMessage(),
                'payment_order' => $order_id,
            ]);

            $this->orderRepository->rejectOrder($order_id);

            return ['error' => [
                'message' => 'Error: ' . $e->getMessage()
            ]];
        }
    }
}
