<?php namespace Visiosoft\PaymentPaypalModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use PayPal\Api\Amount;
use PayPal\Api\Details;
use PayPal\Api\Item;
use PayPal\Api\ItemList;
use PayPal\Api\Payer;
use PayPal\Api\Payment;
use PayPal\Api\PaymentExecution;
use PayPal\Api\RedirectUrls;
use PayPal\Api\Transaction;
use PayPal\Auth\OAuthTokenCredential;
use PayPal\Exception\PayPalConnectionException;
use PayPal\Rest\ApiContext;
use Visiosoft\CartsModule\Cart\Command\GetCart;
use Visiosoft\CartsModule\Cart\Traits\calculateTotal;
use Visiosoft\OrdersModule\Order\Contract\OrderRepositoryInterface;

class PaypalController extends PublicController
{

    use calculateTotal;
    public $orderRepository;

    public $settings;

    public function __construct(OrderRepositoryInterface $orderRepository)
    {
        $this->orderRepository = $orderRepository;

        //set Settings Value
        $this->settings = [
            'mode' => setting_value('visiosoft.module.payment_paypal::mode')
        ];

        $settings_key = [
            'settings',
            'paypal_client_secret',
            'paypal_client_id',
            'paypal_profile_id',
        ];

        foreach ($settings_key as $key) {
            $suffix = ($this->settings['mode'] === "live") ? "_live" : '';
            $this->settings[$key] = setting_value('visiosoft.module.payment_paypal::' . $key . $suffix);
        }
        parent::__construct();
    }

    public function getData()
    {
        //Set Carts items
        $cart = $this->dispatch(new GetCart());

        $data = array();
        foreach ($cart->items as $cart_item) {
            $item = new Item();
            $item->setName($cart_item->name)
                ->setCurrency(substr(setting_value('streams::currency'), 0, 3))
                ->setQuantity($cart_item->quantity)
                ->setSku($cart_item->id)
                ->setPrice($cart_item->price);
            array_push($data, $item);
        }

        return $data;
    }

    public function getConfig()
    {
        //set paypal config
        $apiContext = new ApiContext(new OAuthTokenCredential($this->settings['paypal_client_id'], $this->settings['paypal_client_secret']));

        $apiContext->setConfig(
            array(
                'mode' => $this->settings['mode'],
                'log.LogEnabled' => false,
                'cache.enabled' => false,
            )
        );

        return $apiContext;
    }


    public function payment()
    {

        $order_id = $this->request->cookie('order_id');

        //Find Order
        if (!$this->orderRepository->find($order_id)) {
            $this->messages->error(trans('visiosoft.module.payment_paypal::message.order_not_found'));
            return $this->redirect->to(route('visiosoft.module.carts::cart'));
        }

        $cart = $this->dispatch(new GetCart());

        $items = $this->getData();

        //SubTotal INT |/Tax INT |/Shipping INT |/Total INT |Currency STRING|/Payment Description STRING
        $subtotal = $cart->subtotal;
        $tax = $cart->tax;
        $shipping = $cart->shipping;
        $commission = $cart->commission;
        $total = $cart->total;
        $discounts = 0;

        if (is_module_installed('visiosoft.module.coupon')) {
            $calculate = $this->calculateTotal();
            $total = $calculate['total'];
            $discounts = $calculate['discount'];
        }

        $currency = substr(setting_value('streams::currency'), 0, 3);

        //Set Payment Method
        $payer = new Payer();
        $payer->setPaymentMethod("paypal");

        //Set Items
        $itemList = new ItemList();
        $itemList->setItems($items);

        //Set Details
        $details = new Details();
        $details
            ->setShipping($shipping)
            ->setTax($tax + $commission)
            ->setSubtotal($subtotal)
            ->setShippingDiscount($discounts);

        //Set Amount
        $amount = new Amount();
        $amount
            ->setCurrency($currency)
            ->setTotal($total)
            ->setDetails($details);


        //Set Transaction
        $transaction = new Transaction();
        $transaction
            ->setAmount($amount)
            ->setItemList($itemList)
            ->setDescription(url('/'))
            ->setInvoiceNumber(uniqid());


        //URL Settings
        $return_url = route('visiosoft.module.payment_paypal::result') . "?success=true&order=" . $order_id;
        $cancel_url = route('visiosoft.module.payment_paypal::result') . "?success=false&order=" . $order_id;

        $redirectUrls = new RedirectUrls();
        $redirectUrls
            ->setReturnUrl($return_url)
            ->setCancelUrl($cancel_url);


        $payment = new Payment();
        $payment->setIntent("sale")
            ->setPayer($payer)
            ->setRedirectUrls($redirectUrls)
            ->setExperienceProfileId($this->settings['paypal_profile_id'])
            ->setTransactions(array($transaction));

        try {
            $payment->create($this->getConfig());

        } catch (PayPalConnectionException $ex) {
            $this->messages->error([$ex->getMessage()]);
            return $this->redirect->to($cancel_url);
        }

        return redirect($payment->getApprovalLink());
    }

    public function result()
    {
        if (request()->has(['order', 'paymentId', 'PayerID', 'success'])) {

            $order_id = request()->get('order');
            $payment_id = request()->get('paymentId');
            $payer_id = request()->get('PayerID');

            if (request()->get('success') === "true") {
                $apiContext = $this->getConfig();
                $payment = Payment::get($payment_id, $apiContext);

                $execution = new PaymentExecution();
                $execution->setPayerId($payer_id);

                try {
                    $result = $payment->execute($execution, $apiContext);
                    if ($result->getState() === "approved") {
                        $this->orderRepository->approveOrder($order_id);
                        return redirect(route('visiosoft.module.carts::payment_success', ['type' => 'paypal']));
                    }

                } catch (\Exception $ex) {
                    $this->messages->error($ex->getMessage());
                }
            }
            $this->orderRepository->rejectOrder($order_id);
            return redirect(route('visiosoft.module.carts::payment_failed'));
        } else {
            $this->messages->error(trans('module::message.error_transaction'));
            return $this->redirect->route('visiosoft.module.carts::cart');
        }
    }
}
