<?php namespace Visiosoft\PaymentIyzicoModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\AdvsModule\Adv\Command\ConvertCurrency;
use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;
use Visiosoft\AdvsModule\OptionConfiguration\Contract\OptionConfigurationRepositoryInterface;
use Visiosoft\CartsModule\Cart\Command\GetCart;
use Visiosoft\CartsModule\Cart\Command\TotalCart;
use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Visiosoft\LocationModule\City\Contract\CityRepositoryInterface;
use Visiosoft\LocationModule\Country\Contract\CountryRepositoryInterface;
use Visiosoft\OrdersModule\CcLog\Contract\CcLogRepositoryInterface;
use Visiosoft\OrdersModule\Order\Contract\OrderRepositoryInterface;
use Visiosoft\OrdersModule\Orderdetail\OrderdetailModel;
use Visiosoft\PaymentIyzicoModule\Events\IyziciPaymentCreated;
use Visiosoft\PaymentIyzicoModule\Events\IyzicoErrors;
use Visiosoft\PaymentIyzicoModule\PaymentIyzicoModule;
use Visiosoft\ProfileModule\Adress\AdressModel;

/**
 * Class CartsController
 *
 * @link          http://openclassify.com/
 * @author        OpenClassify, Inc. <support@openclassify.com>
 * @author        Visiosoft Inc <support@openclassify.com>
 * @package       Visiosoft\CartsModule\Http\Controller
 */
class IyzicoController extends PublicController
{
    private $address_model;
    private $countryRepository;
    private $userRepository;
    private $cityRepository;
    private $settings;
    private $advModel;
    private $advRepository;
    private $optionConfigurationRepository;
    private $orderdetailModel;
    protected $ccLogRepository;

    public function __construct(
        AdressModel $adressModel,
        CountryRepositoryInterface $countryRepository,
        UserRepositoryInterface $userRepository,
        CityRepositoryInterface $cityRepository,
        SettingRepositoryInterface $settingRepository,
        AdvModel $advModel,
        OptionConfigurationRepositoryInterface $optionConfigurationRepository,
        AdvRepositoryInterface $advRepository,
        OrderdetailModel $orderdetailModel,
        CcLogRepositoryInterface $ccLogRepository
    )
    {
        parent::__construct();
        $this->address_model = $adressModel;
        $this->countryRepository = $countryRepository;
        $this->userRepository = $userRepository;
        $this->cityRepository = $cityRepository;
        $this->settings = $settingRepository;
        $this->advModel = $advModel;
        $this->advRepository = $advRepository;
        $this->optionConfigurationRepository = $optionConfigurationRepository;
        $this->orderdetailModel = $orderdetailModel;
        $this->ccLogRepository = $ccLogRepository;
    }

    public function paymentForm($contractPage = null)
    {
        $delivery_id = $this->request->cookie('delivery');
        return $this->view->make('visiosoft.module.payment_iyzico::iyzico/form', compact('delivery_id', 'contractPage'));
    }

    public function isCardFromTurkey($options, $price, $card_number)
    {
        $card_number = substr(str_replace(' ', '', $card_number), 0, 6);

        $request = new \Iyzipay\Request\RetrieveInstallmentInfoRequest();
        $request->setLocale(\Iyzipay\Model\Locale::TR);
        $request->setBinNumber($card_number);
        $request->setPrice($price);

        $installmentInfo = \Iyzipay\Model\InstallmentInfo::retrieve($request, $options);

        $detail = array_first($installmentInfo->getInstallmentDetails());

        if (count($detail->getInstallmentPrices()) > 1) {
            return true;
        }

        return false;
    }

    public function paymentIyzico()
    {
        $bill_id = $this->request->cookie('bill');
        $orderId = $this->request->cookie('order_id');
        $payment_status = $this->request->payment_status;

        if (!$bill_id) {
            $this->messages->error(trans('visiosoft.module.payment_iyzico::message.required_bill_address'));
        }

        $address = $this->address_model->getAdress($bill_id)->first();
        $country = $this->countryRepository->find($address->country_id);
        $city = $this->cityRepository->find($address->city);

        $expiration = explode('/', $this->request->expiration);

        $cart = $this->dispatch(new GetCart());
        $this->dispatch(new TotalCart($cart));

        $options = new \Iyzipay\Options();
        $options->setApiKey($this->settings->value('visiosoft.module.payment_iyzico::iyzico_setApiKey'));
        $options->setSecretKey($this->settings->value('visiosoft.module.payment_iyzico::iyzico_setSecretKey'));
        $options->setBaseUrl($this->settings->value('visiosoft.module.payment_iyzico::iyzico_setBaseUrl'));

        $card_from_turkey = $this->isCardFromTurkey($options, $cart->total, $this->request->ccno);

        $total = $cart->total;
        $currency = setting_value('streams::currency');

        if (setting_value('streams::currency') != "TRY" and $card_from_turkey) {

            $convert = $this->dispatch(new ConvertCurrency($total, setting_value('streams::currency'), "TRY"));

            $total = (float)number_format($convert['price'], 2);
            $currency = $convert['currency'];

        }

        $request = new \Iyzipay\Request\CreatePaymentRequest();
        $request->setLocale(setting_value('streams::default_locale'));
        $request->setPrice($total);
        $request->setPaidPrice($total);
        $request->setCurrency($currency);
        $request->setInstallment(1);
        $request->setCallbackUrl(url('payment/return'));

        $conversation_parameters = [
            'order_id' => $orderId,
            'payment_status' => $payment_status ?? null,
        ];

        if (Auth::check()) {
            //Remember Auth
            $auth = $this->userRepository->find(Auth::id());
            $remember_token_code = Str::random(20);
            $auth->setAttribute('remember_token', $remember_token_code);
            $auth->save();
            $conversation_parameters['remember_token'] = $remember_token_code;
        }

        $request->setConversationId(str_replace('"', '\'', json_encode($conversation_parameters)));

        $paymentCard = new \Iyzipay\Model\PaymentCard();
        $paymentCard->setCardHolderName($address->adress_first_name . ' ' . $address->adress_last_name);
        $paymentCard->setCardNumber(str_replace(' ', '', $this->request->ccno));
        $paymentCard->setExpireMonth(array_first($expiration));
        $paymentCard->setExpireYear(end($expiration));
        $paymentCard->setCvc($this->request->ccv);
        $paymentCard->setRegisterCard(0);
        $request->setPaymentCard($paymentCard);

        $buyer = new \Iyzipay\Model\Buyer();
        $buyer->setId($address->user_id);
        $buyer->setName($address->adress_first_name);
        $buyer->setSurname($address->adress_last_name);
        $buyer->setEmail($address->user->email);
        $buyer->setIdentityNumber("12312312312");
        $buyer->setRegistrationAddress($address->adress_content);
        $buyer->setCity($city->name);
        $buyer->setCountry($country->name);
        $request->setBuyer($buyer);


        //Create Billing Address
        $billingAddress = new \Iyzipay\Model\Address();
        $billingAddress->setContactName($address->adress_first_name . ' ' . $address->adress_last_name);
        $billingAddress->setCity($city->name);
        $billingAddress->setCountry($country->name);
        $billingAddress->setAddress($address->adress_content);
        $request->setBillingAddress($billingAddress);

        //Create Shipping Address
        if ($shipping = $this->request->cookie('delivery') and $shipping = $this->address_model->getAdress($shipping)->first()) {
            $shipping_country = $this->countryRepository->find($shipping->country_id);
            $shipping_city = $this->cityRepository->find($shipping->city);

            $shipping_address = new \Iyzipay\Model\Address();
            $shipping_address->setContactName($shipping->adress_first_name . ' ' . $shipping->adress_last_name);
            $shipping_address->setCity($shipping_city->name);
            $shipping_address->setCountry($shipping_country->name);
            $shipping_address->setAddress($shipping->adress_content);
            $request->setShippingAddress($shipping_address);
        }

        $basketItems = array();
        $orderItems = $this->orderdetailModel->getDetail($orderId);
        foreach ($orderItems as $item) {
            $basketItem = new \Iyzipay\Model\BasketItem();
            $basketItem->setId($item->id);
            $basketItem->setName('Item');
            $basketItem->setCategory1("VIRTUAL");
            $basketItem->setItemType(\Iyzipay\Model\BasketItemType::VIRTUAL);

            $item_total = $item->total;

            if (setting_value('streams::currency') != "TRY" and $card_from_turkey) {

                $convert = $this->dispatch(new ConvertCurrency($item_total, setting_value('streams::currency'), "TRY"));

                $item_total = (float)number_format($convert['price'], 2);
            }

            $basketItem->setPrice($item_total);

            if ($this->advModel->is_enabled('payment_iyzicomp') && ($item->item_type == 'adv' or $item->item_type == 'ads-configuration')) {
                try {
                    if (!$this->advModel->is_enabled('store')) {
                        throw new \Exception('Store module is required!');
                    }

                    if ($item->item_type == 'ads-configuration') {
                        $configuration = $this->optionConfigurationRepository->find($item->item_id);
                        if (!$adv = $configuration->parent_adv) {
                            throw new \Exception(trans('visiosoft.module.payment_iyzico::message.ad_not_found'));
                        }
                    } else {
                        $adv = $this->advRepository->find($item->item_id);
                    }

                    $store = app('Visiosoft\StoreModule\Store\Contract\StoreRepositoryInterface')
                        ->findByUser($adv->created_by_id);

                    if (!$store) {
                        throw new \Exception(trans('visiosoft.module.payment_iyzico::message.seller_doesnt_have_store'));
                    }
                    $commission = setting_value('visiosoft.module.orders::commission');
                    if ($this->advModel->is_enabled('packages')) {
                        $commission = app('Visiosoft\PackagesModule\User\UserModel')
                            ->getUserCategoryCommission($adv->cat1, $adv->created_by_id);
                    }
                    $subMerchantPrice = $item->total - ($item->total * ($commission / 100));
                    $subMerchant = app('Visiosoft\PaymentIyzicompModule\Http\Controller\SubmerchantsController')
                        ->getSubmerchant($store, $options);

                    $basketItem->setSubMerchantPrice($subMerchantPrice);
                    $basketItem->setSubMerchantKey($subMerchant->key);
                } catch (\Exception $e) {
                    event(new IyzicoErrors($e->getMessage(), 'ad', $adv ?? null));
                    $this->messages->error(
                        trans('visiosoft.module.payment_iyzico::message.something_went_wrong')
                        . ' ' . $e->getMessage() . '. '
                        . trans('visiosoft.module.payment_iyzico::message.please_report_this_issue_to_the_site_owner'));
                    return redirect('payment/iyzico');
                }
            }

            $basketItems[] = $basketItem;
        }

        //Add Shipping
        if ($cart->getShipping() > 0) {
            $basketItem = new \Iyzipay\Model\BasketItem();
            $basketItem->setName('Shipping');
            $basketItem->setId("00000");
            $basketItem->setCategory1("Shipping");
            $basketItem->setItemType(\Iyzipay\Model\BasketItemType::VIRTUAL);

            $shipping_price = $cart->getShipping();

            if (setting_value('streams::currency') != "TRY" and $card_from_turkey) {

                $convert = $this->dispatch(new ConvertCurrency($shipping_price, setting_value('streams::currency'), "TRY"));

                $shipping_price = (float)number_format($convert['price'], 2);
            }

            $basketItem->setPrice($shipping_price);

            array_push($basketItems, $basketItem);
        }

        $request->setBasketItems($basketItems);

        $threedsInitialize = \Iyzipay\Model\ThreedsInitialize::create($request, $options);

        if ($threedsInitialize->getStatus() == "failure") {
            // Log CC transaction
            $this->ccLogRepository->create([
                'payment_type' => PaymentIyzicoModule::TYPE,
                'message' => $threedsInitialize->getErrorMessage(),
                'response' => $threedsInitialize->getRawResult(),
                'success' => false,
                'payment_order' => $orderId,
            ]);

            return redirect()
                ->route('visiosoft.module.payment_iyzico::payment.iyzico')
                ->with('error', [$threedsInitialize->getErrorMessage()]);
        } elseif (is_null($threedsInitialize->getStatus())) {
            // Log CC transaction
            $this->ccLogRepository->create([
                'payment_type' => PaymentIyzicoModule::TYPE,
                'message' => trans('visiosoft.module.payment_iyzico::message.please_check_your_iyzico_settings'),
                'success' => false,
                'payment_order' => $orderId,
            ]);

            return redirect()
                ->route('visiosoft.module.payment_iyzico::payment.iyzico')
                ->with('error', [trans('visiosoft.module.payment_iyzico::message.please_check_your_iyzico_settings')]);
        }

        // Log CC transaction
        $this->ccLogRepository->create([
            'payment_type' => PaymentIyzicoModule::TYPE,
            'response' => $threedsInitialize->getRawResult(),
            'success' => true,
            'payment_order' => $orderId,
        ]);

        $htmli = $threedsInitialize->getHtmlContent();
        return $this->view->make('visiosoft.module.payment_iyzico::iyzico/redirect', compact('htmli'));
    }

    public function returnIyzico(OrderRepositoryInterface $orderRepository)
    {
        $conversation_parameters = $this->request->get('conversationId');
        $conversation_parameters = json_decode(str_replace('\'', '"', $conversation_parameters), true);

        $order_id = $conversation_parameters['order_id'];
        $options = new \Iyzipay\Options();
        $options->setApiKey($this->settings->value('visiosoft.module.payment_iyzico::iyzico_setApiKey'));
        $options->setSecretKey($this->settings->value('visiosoft.module.payment_iyzico::iyzico_setSecretKey'));
        $options->setBaseUrl($this->settings->value('visiosoft.module.payment_iyzico::iyzico_setBaseUrl'));
        $pID = $this->request->paymentId;

        if (isset($conversation_parameters['remember_token']) and !empty($conversation_parameters['remember_token'])
            and $auth = $this->userRepository->findBy('remember_token', $conversation_parameters['remember_token'])) {

            $last_activity = strtotime($auth->last_activity_at);
            $now = strtotime('now');

            //diff 3 minutes
            if (($now - $last_activity) < 180) {
                Auth::login($auth);
            }

            //remove remember token
            $auth->setAttribute('remember_token', null);
            $auth->save();
        }

        $request = new \Iyzipay\Request\CreateThreedsPaymentRequest();
        $request->setLocale(\Iyzipay\Model\Locale::TR);
        $request->setPaymentId($pID);
        $threedsPayment = \Iyzipay\Model\ThreedsPayment::create($request, $options);
        if ($threedsPayment->getStatus() == 'success') {
            // Log CC transaction
            $this->ccLogRepository->create([
                'payment_type' => PaymentIyzicoModule::TYPE,
                'response' => $threedsPayment->getRawResult(),
                'success' => true,
                'payment_order' => $order_id,
            ]);

            event(new IyziciPaymentCreated($threedsPayment));

            $orderRepository->approveOrder($order_id);

            if (!$order = $orderRepository->newQuery()->find($order_id)) {
                $this->messages->error(trans('visiosoft.module.payment_iyzico::message.error_approving'));
                return $this->redirect->to('/purchase');
            }

            if (!Auth::check()) {
                return redirect(route('visiosoft.module.orders::customer.payment_success'));
            }

            if (isset($conversation_parameters['payment_status'])) {
                return $this->view->make($conversation_parameters['payment_status']);
            }

            return redirect('payment/success/iyzico');
        } else {
            // Log CC transaction
            $this->ccLogRepository->create([
                'payment_type' => PaymentIyzicoModule::TYPE,
                'message' => $threedsPayment->getErrorMessage(),
                'response' => $threedsPayment->getRawResult(),
                'success' => false,
                'payment_order' => $order_id,
            ]);

            $errorMsg = $threedsPayment->getErrorCode() == "5002"
                ? trans('visiosoft.module.payment_iyzico::message.payment_failed')
                : $threedsPayment->getErrorMessage();

            $this->messages->error($errorMsg);

            $orderRepository->rejectOrder($order_id, $errorMsg);

            if (isset($conversation_parameters['payment_status'])) {
                return $this->view->make($conversation_parameters['payment_status']);
            }

            return redirect('payment/fail');
        }
    }
}
