<?php namespace Visiosoft\PaymentModule\Payment;

use Exception;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\CurrencyModule\CurrencyModule;
use Visiosoft\ParkingModule\Park\Contract\PricingCostCalculateHelperInterface;
use Visiosoft\PaymentModule\Helper\CommonHelper;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;

class PaymentRepository extends EntryRepository implements PaymentRepositoryInterface
{

    public string $currency;

    protected $model;

    public function __construct(PaymentModel $model)
    {
        $currencyModule = app(CurrencyModule::class);
        $this->currency = $currencyModule->getDefault()->symbol;
        $this->model = $model;
    }


    /**
     * @param array $params (['payment_id' => int])
     * @return object
     * @throws Exception
     */
    public function getDetail(array $params): object
    {
        $output = [
            'payment_id' => null,
            'location_name' => null,
            'created_at' => null,
            'ended_at' => null,
            'plate' => null,
            'usage_type' => null,
            'is_drifterbox' => null,
            'type' => '',
            'category_id' => null,
            'status_id' => null,
            'park_session_cost' => null,
            'parking_price' => null,
            'park_session_note' => null,
            'charging_price_type' => null,
            'charging_session_cost' => null,
            'digital_code' => null,
            'charging_price' => null,
            'vat' => 0,
            'charging_electricity_consumption' => null,
            'parking_duration' => null,
            'total_amount' => 0,
            'administration_fee' => 0,
            'park_session_status' => null
        ];
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        $payment = $this->newQuery()
            ->whereNull('deleted_at')
            ->find($params['payment_id']);
        if (!$payment) {
            throw new Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }
        $output['payment_id'] = $payment->id;
        $output['status_id'] = $payment->status_id;
        $output['category_id'] = $payment->category_id;
        $locationName = '';
        $output['usage_type'] = !empty($payment->company_id) ? 'business' : 'personal';
        if (!empty($payment->charging_transaction)) {
            $chargingPricingType = $payment->charging_transaction->station->pricing_type;
            if (!$chargingPrice = $payment->charging_transaction->transaction_station->charge_price) {
                $chargingPrice = 0;
            }
            $output['charging_price_type'] = $chargingPricingType;
            $output['charging_session_cost'] = $payment->charging_transaction->payment_amount;
            $output['charging_price'] = $chargingPrice;
            $output['charging_electricity_consumption'] = $payment->charging_transaction->total_consumption;
            $locationName = $payment->charging_transaction->station->title;
            $output['created_at'] = $payment->charging_transaction->created_at;
            $output['ended_at'] = $payment->charging_transaction->ended_at;
            $output['plate'] = $payment->charging_transaction->plate_number;
            $output['is_drifterbox'] = !empty($payment->charging_transaction->drifterbox_id) ? 1 : 0;
        }
        if (!empty($payment->park_session)) {
            $parkingCostCalculator = app(PricingCostCalculateHelperInterface::class);
            //todo parking price should submit to the session table. realtime calculation is wrong
            $duration = strtotime($payment->park_session->ended_at) - strtotime($payment->park_session->started_at);
            $costs = $parkingCostCalculator->calculate($payment->park_session, $duration);
            $output['park_session_cost'] = $payment->park_session->payment_amount;
            $output['parking_price'] = $costs['current_price'];
            $output['parking_duration'] = $duration;
            $locationName = $payment->park_session->park->name;
            $output['created_at'] = $payment->park_session->created_at;
            $output['ended_at'] = $payment->park_session->ended_at;
            $output['plate'] = $payment->park_session->plate_number;
            $output['is_drifterbox'] = $payment->park_session->auto_parking;
            $output['park_session_status'] = $payment->park_session->session_status_id == 1 ? 'normal' : 'offense';
            if ($payment->park_session->is_digital_code_active) {
                $output['park_session_note'] = trans('visiosoft.module.payment.message.you_have_used_digital_code');
                $output['digital_code'] = $payment->park_session->digital_code_text;
                $output['park_session_cost'] = 0;
                $output['parking_price'] = 0;
            }
        }
        if ($payment->category_id == 4) {
            $output['type'] = 'park_and_charge';
        }
        if ($payment->category_id == 1) {
            $output['type'] = 'charge';
        }
        if ($payment->category_id == 2) {
            $output['type'] = 'park';
        }
        $output['location_name'] = $locationName;
        $fees = $payment->fees;
        $output['vat'] = $fees['total_vat_amount'];
        $output['total_amount'] = $fees['payment_amount'];
        $output['administration_fee'] = $fees['administration_fee'];
        return collect($output);
    }


    /**
     * @param string $plate
     * @param int $park_id
     * @return object
     */
    public function listByPlateAndPark(string $plate, int $park_id): object
    {

        $list = [];
        $payments = $this->newQuery()
            ->where('amount', '>', 0)
            ->where('plate', $plate)
            ->where('park_id', $park_id)
            ->where('status_id', '!=', 2)
            ->whereNull('deleted_at')
            ->orderBy('created_at', 'DESC')
            ->each(function ($payment) use (&$list) {
                $output = [
                    'payment_id' => $payment->id,
                    'usage_type' => !empty($payment->company_id) ? 'business' : 'personal',
                    'category_id' => $payment->category_id,
                ];

                if (!empty($payment->charging_transaction)) {
                    $chargingPricingType = $payment->charging_transaction->station->pricing_type;
                    if (!$chargingPrice = $payment->charging_transaction->transaction_station->charge_price) {
                        $chargingPrice = 0;
                    }
                    $output['charging_price_type'] = $chargingPricingType;
                    $output['charging_session_cost'] = $payment->charging_transaction->payment_amount;
                    $output['charging_price'] = $chargingPrice;
                    $output['charging_electricity_consumption'] = $payment->charging_transaction->total_consumption;
                    $output['created_at'] = $payment->charging_transaction->created_at;
                    $output['ended_at'] = $payment->charging_transaction->ended_at;
                }
                if (!empty($payment->park_session)) {
                    $parkingCostCalculator = app(PricingCostCalculateHelperInterface::class);
                    //todo parking price should submit to the session table. realtime calculation is wrong
                    if($payment->park_session->ended_at == null) {
                        $payment->park_session->ended_at = $payment->park_session->started_at;
                    }
                    $duration = strtotime($payment->park_session->ended_at) - strtotime($payment->park_session->started_at);
                    $costs = $parkingCostCalculator->calculate($payment->park_session, $duration);
                    $output['park_session_cost'] = $payment->park_session->payment_amount;
                    $output['park_session_status'] = $payment->park_session->session_status_id == 1 ? 'normal' : 'offense';
                    $output['parking_price'] = $costs['current_price'];
                    if ($duration) {
                        $output['parking_duration'] = CommonHelper::format_duration($duration);
                    } else {
                        $output['parking_duration'] = 0;
                    }
                    $output['created_at'] = $payment->park_session->created_at->format("Y-m-d H:i:s");
                    $output['ended_at'] = $payment->park_session->ended_at->format("Y-m-d H:i:s");
                    if ($payment->park_session->is_digital_code_active) {
                        $output['park_session_note'] = trans('visiosoft.module.payment.message.you_have_used_digital_code');
                        $output['digital_code'] = $payment->park_session->digital_code_text;
                        $output['park_session_cost'] = 0;
                        $output['parking_price'] = 0;
                    }
                }
                if ($payment->category_id == 4) {
                    $output['type'] = 'park_and_charge';
                }
                if ($payment->category_id == 1) {
                    $output['type'] = 'charge';
                }
                if ($payment->category_id == 2) {
                    $output['type'] = 'park';
                }
                $fees = $payment->fees;
                $output['vat'] = $fees['total_vat_amount'];
                $output['total_amount'] = $fees['payment_amount'];
                $output['administration_fee'] = $fees['administration_fee'];

                $list[] = $output;

            }, 100);

        return collect($list);
    }

}
