<?php namespace Visiosoft\PaymentModule\Http\Controller;

use \Anomaly\Streams\Platform\Http\Controller\PublicController;
use Anomaly\Streams\Platform\Image\Command\MakeImageInstance;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Visiosoft\ConnectModule\Traits\ApiReturnResponseTrait;
use Visiosoft\ParkingModule\Park\Contract\PricingCostCalculateHelperInterface;
use Visiosoft\ParkingModule\ParkingSession\Contract\ParkingSessionRepositoryInterface;
use Visiosoft\PaymentModule\Event\InvoiceDownloadEvent;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;
use Visiosoft\PaymentModule\Payment\PaymentModel;
use Visiosoft\PaymentModule\PaymentModule;
use Visiosoft\PaymentStripeExtension\PaymentStripeExtension;

class PaymentController extends PublicController
{
    use ApiReturnResponseTrait;

    protected ParkingSessionRepositoryInterface $parkingSessionRepository;
    protected PaymentRepositoryInterface $paymentRepository;
    protected PricingCostCalculateHelperInterface $pricingCostCalculateHelper;

    public function __construct(
        ParkingSessionRepositoryInterface   $parkingSessionRepository,
        PaymentRepositoryInterface          $paymentRepository,
        PricingCostCalculateHelperInterface $pricingCostCalculateHelper

    )
    {
        $this->parkingSessionRepository = $parkingSessionRepository;
        $this->paymentRepository = $paymentRepository;
        $this->pricingCostCalculateHelper = $pricingCostCalculateHelper;
        parent::__construct();
    }

    public function successPage(Request $request)
    {
        $inputs = $request->all();
        $sessionId = $inputs['session_id'];

        $stripeSuccessPageData = PaymentStripeExtension::successPageSession($sessionId);

        $session = $this->parkingSessionRepository->newQuery()->find($stripeSuccessPageData['session']['client_reference_id']);
        $payment = $this->paymentRepository->newQuery()->find($session->payment_transaction_id);
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        if ($stripeSuccessPageData['session']['payment_status'] == 'paid') {
            $payment->update(['status_id' => 2]);
        }
        $customerName = $stripeSuccessPageData['session']['customer_details']['name'];
        $customerEmail = $stripeSuccessPageData['session']['customer_details']['email'];
        if (!empty($payment->invoice_file)) {
            $invoiceUrl = PaymentModule::retrieveInvocieUrl($payment);
        } else {
            $invoiceUrl = PaymentModule::createInvoice($payment, ['customer_email' => $customerEmail, 'customer_name' => $customerName]);
        }
        return $this->view->make('visiosoft.module.payment::success-page', [
            'session' => $session,
            'payment' => $payment,
            'customer_name' => $customerName,
            'customer_email' => $customerEmail,
            'invoice_download_url' => $invoiceUrl
        ]);
    }

    public function cancelPage(Request $request)
    {
        $inputs = $request->all();
        $sessionId = $inputs['session_id'];
        $cancelData = PaymentStripeExtension::cancelPageSession($sessionId);
        $session = $this->parkingSessionRepository->newQuery()->where('id', $cancelData['session']['client_reference_id'])->first();
        if (!$session) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Park Session']));
        }
        $this->paymentRepository->newQuery()->where('park_session_id', $session->id)->delete();
        $session->update(['ended_at' => null]);
        return $this->view->make('visiosoft.module.payment::cancel-page');
    }

    public function invoice($stripePaymentIntent)
    {
        $payment = $this->paymentRepository->newQuery()->where('stripe_intent_id', $stripePaymentIntent)->where('status_id', 2)->first();
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $invoiceUrl = PaymentModule::retrieveInvocieUrl($payment);
        return $this->sendResponse($invoiceUrl);
    }

    public function invoiceEmail(Request $request, $stripePaymentIntent)
    {
        $inputs = $request->all();
        $customerName = $inputs['customer_name'];
        $email = $inputs['customer_email'];
        $link = config('app.url') . '/invoice/download/' . $stripePaymentIntent;
        event(new InvoiceDownloadEvent($customerName, $email, $link));
        return $this->sendResponse([], trans('visiosoft.module.payment::message.email_sent'));
    }
}