<?php namespace Visiosoft\PaymentModule\Http\Controller;

use \Anomaly\Streams\Platform\Http\Controller\PublicController;
use Anomaly\Streams\Platform\Image\Command\MakeImageInstance;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;
use Visiosoft\ConnectModule\Traits\ApiReturnResponseTrait;
use Visiosoft\ParkingModule\Park\Contract\ParkRepositoryInterface;
use Visiosoft\ParkingModule\Park\Contract\PricingCostCalculateHelperInterface;
use Visiosoft\ParkingModule\ParkingSession\Contract\ParkingSessionRepositoryInterface;
use Visiosoft\PaymentModule\Event\InvoiceDownloadEvent;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;
use Visiosoft\PaymentModule\Payment\PaymentModel;
use Visiosoft\PaymentModule\PaymentMethod\PaymentMethodRepository;
use Visiosoft\PaymentModule\PaymentModule;
use Visiosoft\PaymentModule\Status\Contract\StatusRepositoryInterface;
use Visiosoft\PaymentStripeExtension\PaymentStripeExtension;

class PaymentController extends PublicController
{
    use ApiReturnResponseTrait;

    protected ParkingSessionRepositoryInterface $parkingSessionRepository;
    protected PaymentRepositoryInterface $paymentRepository;
    protected PricingCostCalculateHelperInterface $pricingCostCalculateHelper;
    protected ParkRepositoryInterface $parkRepository;

    public function __construct(
        ParkingSessionRepositoryInterface   $parkingSessionRepository,
        PaymentRepositoryInterface          $paymentRepository,
        PricingCostCalculateHelperInterface $pricingCostCalculateHelper,
        ParkRepositoryInterface $parkRepository

    )
    {
        $this->parkingSessionRepository = $parkingSessionRepository;
        $this->paymentRepository = $paymentRepository;
        $this->pricingCostCalculateHelper = $pricingCostCalculateHelper;
        $this->parkRepository =$parkRepository;
        parent::__construct();
    }

    public function successPage(Request $request)
    {
        $inputs = $request->all();
        $sessionId = $inputs['session_id'];

        $stripeSuccessPageData = PaymentStripeExtension::successPageSession($sessionId);

        $session = $this->parkingSessionRepository->newQuery()->find($stripeSuccessPageData['session']['client_reference_id']);
        $park = $this->parkRepository->find($session->park_id);
        $payment = $this->paymentRepository->newQuery()->find($session->payment_transaction_id);
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        if ($stripeSuccessPageData['session']['payment_status'] == 'paid') {
            $payment->update(['status_id' => 2]);
        }
        $customerName = $stripeSuccessPageData['session']['customer_details']['name'];
        $customerEmail = $stripeSuccessPageData['session']['customer_details']['email'];
        if (!empty($payment->invoice_file)) {
            $invoiceUrl = PaymentModule::retrieveInvocieUrl($payment);
        } else {
            $invoiceUrl = PaymentModule::createInvoice($payment, ['customer_email' => $customerEmail, 'customer_name' => $customerName]);
        }
        return $this->view->make('visiosoft.module.payment::success-page', [
            'session' => $session,
            'parking' => $park,
            'payment' => $payment,
            'customer_name' => $customerName,
            'customer_email' => $customerEmail,
            'invoice_download_url' => $invoiceUrl
        ]);
    }

    public function cancelPage(Request $request)
    {
        $inputs = $request->all();
        $sessionId = $inputs['session_id'];
        $cancelData = PaymentStripeExtension::cancelPageSession($sessionId);
        $session = $this->parkingSessionRepository->newQuery()->where('id', $cancelData['session']['client_reference_id'])->first();
        if (!$session) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Park Session']));
        }
        $this->paymentRepository->newQuery()->where('park_session_id', $session->id)->delete();
        $session->update(['ended_at' => null]);
        return $this->view->make('visiosoft.module.payment::cancel-page',['park_id'=>$session->park_id]);
    }

    public function invoice($stripePaymentIntent)
    {
        $payment = $this->paymentRepository->newQuery()->where('stripe_intent_id', $stripePaymentIntent)->where('status_id', 2)->first();
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $invoiceUrl = PaymentModule::retrieveInvocieUrl($payment);
        return $this->sendResponse($invoiceUrl);
    }

    public function invoiceEmail(Request $request, $stripePaymentIntent)
    {
        $inputs = $request->all();
        $customerName = $inputs['customer_name'];
        $email = $inputs['customer_email'];
        $link = config('app.url') . '/invoice/download/' . $stripePaymentIntent;
        event(new InvoiceDownloadEvent($customerName, $email, $link));
        return $this->sendResponse([], trans('visiosoft.module.payment::message.email_sent'));
    }

    public function checkPendingPayment()
    {
        $control_range = setting_value('visiosoft.module.payment::payment_control_range');
        $query_time = Carbon::now()->subMinute($control_range);
        $statusRepository = app(StatusRepositoryInterface::class);
        $status_id = $statusRepository->findBySlug('unpaid')->getId();

        $payments = $this->paymentRepository->newQuery()
            ->where('created_at','<',$query_time->toDateTimeString())
            ->whereHas('status', function($q)
            {
                $q->whereIn('slug', ['pending']);
            })->each(function ($payment) use ($status_id){
                $payment->status = $status_id;
                $payment->save();
            },1000);
    }


    public function checkBalanceOverlimit()
    {
        $paymentMethodRepository = app(PaymentMethodRepository::class);
        $statusRepository = app(StatusRepositoryInterface::class);
        $status_id = $statusRepository->findBySlug('paid')->getId();
        $balance_limit = setting_value('visiosoft.module.payment::balance_limit');

        $overlimits = $this->paymentRepository->newQuery()
                ->selectRaw("sum(amount) as amount, user_id as user_id,group_concat(id) as ids,currency")
                ->havingRaw('(SUM(amount)/100) >= ?', [$balance_limit])
                ->groupBy(['user_id'])
                ->whereHas('status', function($q)
                {
                    $q->whereIn('slug', ['unpaid']);
                })->each(function ($record) use ($status_id,$paymentMethodRepository){
                    $payment_ids =explode(",",$record->ids);

                    $payment_method = $paymentMethodRepository->newQuery()
                        ->where('owner_id',$record->user_id)
                        ->where('default',1)
                        ->whereNull('deleted_at')
                        ->first();

                    if ($payment_method)
                    {
                        $payment_params =
                            [
                                'currency' => $record->currency,
                                'amount' => $record->amount,
                                'payment_method'=> $payment_method->service_payment_method_id,
                                'payment_method_type' => 'card',
                                'customer' => $payment_method->customer
                            ];

                        $create_payment = PaymentStripeExtension::createPaymentIntent($payment_params);
                        if ($create_payment)
                        {
                            $confirm_payment = PaymentStripeExtension::confirmPaymentIntent(['stripe_intent_id' => $create_payment['id']]);
                            if ($confirm_payment && $confirm_payment['status']=="succeeded")
                            {
                                foreach ($payment_ids as $payment_id){
                                    $payment = $this->paymentRepository->find($payment_id);
                                    $payment->payment_method_id=$payment_method->id;
                                    $payment->stripe_intent_id = $create_payment['id'];
                                    $payment->status = $status_id;
                                    $payment->save();
                                }
                            }
                        }

                    }
                },1000);
    }
}