<?php namespace Visiosoft\PaymentModule\Payment;

use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Facades\Auth;
use Visiosoft\ChargingModule\Transaction\Contract\TransactionRepositoryInterface;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\CurrencyModule\CurrencyModule;
use Visiosoft\PaymentModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\PaymentModule\Helper\PaymentChannel;
use Visiosoft\PaymentModule\Status\Contract\StatusRepositoryInterface;

class PaymentApiCollection extends PaymentRepository
{
    use DispatchesJobs;

    public function createCard(array $params)
    {
        $user = Auth::user();
        try {
            $cardUserKey = $this->paymentMethodRepository->getCustomerIdByService($user->id, $this->card_provider->getProviderSlug());
            $providerResponse = $this->card_provider->createCard(
                $params['name'],
                $params['card']['card_number'],
                $params['card']['expire_year'],
                $params['card']['expire_month'],
                $params['card']['cvc'],
                $user->email,
                $cardUserKey
            );

            if ($providerResponse['status']) {
                $createCard = [
                    'owner_id' => Auth::id(),
                    'service' => $providerResponse['data']['service'],
                    'service_payment_method_id' => $providerResponse['data']['service_payment_method_id'],
                    'customer' => $providerResponse['data']['customer'],
                    'card' => $providerResponse['data']['card'],
                    'holder_name' => $params['name']
                ];

                $this->paymentMethodRepository->addPaymentMethod($createCard);

            } else {
                throw new \Exception($providerResponse["message"] ?? trans('visiosoft.module.payment::message.cant_add_card'));
            }

        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());
        }
        return true;
    }

    public function cards()
    {
        return $this->paymentMethodRepository->getCardsByOwner(Auth::id(), ['id', 'card', 'default', 'holder_name', 'created_at']);
    }

    public function listV2()
    {
        $userId = Auth::id();

        $paymentCategoryRepository = app(CategoryRepositoryInterface::class);
        $categoryVerification = $paymentCategoryRepository->findBy('slug', 'verification');
        $payments = PaymentWithoutAttrModel::with(['category', 'status'])
            ->select(['id', 'category_id', 'status_id', 'currency', 'amount', 'created_at', 'company_id', 'transaction_id'])
            ->where('user_id', $userId)
            ->orderBy('created_at', 'DESC');



        if ($categoryVerification) {
            $payments = $payments->where('category_id', '!=', $categoryVerification->id);
        }

        if (!empty($params['category_id'])) {
            $payments = $payments->where('category_id', $params['category_id']);
        }
        if (!empty($params['status_id'])) {
            $payments = $payments->where('status_id', $params['status_id']);
        }
        if (!empty($params['start_date'])) {
            $payments = $payments->where('created_at', '>=', $params['start_date']);
        }
        if (!empty($params['end_date'])) {
            $payments = $payments->where('created_at', '<=', $params['end_date']);
        }
        if (!empty($params['account_type_id'])) {
            if ($params['account_type_id'] == 1) {
                $payments = $payments->whereNull('company_id');
            }
            if ($params['account_type_id'] == 2) {
                $payments = $payments->whereNotNull('company_id');
            }
        }
        return $payments;
    }

    /**
     * @deprecated use listV2
     */
    public function list(array $params)
    {
        $userId = Auth::id();
        $payments = $this->newQuery()
            ->where('user_id', $userId)
            ->orderBy('created_at', 'DESC');
        if (!empty($params['category_id'])) {
            $payments = $payments->where('category_id', $params['category_id']);
        }
        if (!empty($params['status_id'])) {
            $payments = $payments->where('status_id', $params['status_id']);
        }
        if (!empty($params['start_date'])) {
            $payments = $payments->where('created_at', '>=', $params['start_date']);
        }
        if (!empty($params['end_date'])) {
            $payments = $payments->where('created_at', '<=', $params['end_date']);
        }
        if (!empty($params['account_type_id'])) {
            if ($params['account_type_id'] == 1) {
                $payments = $payments->whereNull('company_id');
            }
            if ($params['account_type_id'] == 2) {
                $payments = $payments->whereNotNull('company_id');
            }
        }
        return $payments;
    }

    public function references()
    {
        $categoryRepository = app(CategoryRepositoryInterface::class);
        $statusRepository = app(StatusRepositoryInterface::class);
        return collect([
            [
                'statuses' => $statusRepository->newQuery()->get(),
                'category' => $categoryRepository->newQuery()->get(),
                'account_types' => [['name' => 'Personal', 'id' => 1], ['name' => 'Business', 'id' => 2]]
            ]
        ]);
    }

    public function removeCard(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['card_id'], $params));
        if (!$paymentMethod = $this->paymentMethodRepository->find($params['card_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment_method.name')]));
        }

        if ($paymentMethod->delete()) {
            $deleted_card = $this->card_provider->removeCard($paymentMethod->service_payment_method_id);
            if (!$deleted_card) {
                throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.card')]));
            }
        }
        return true;
    }

    public function pay(array $params)
    {
        return $this->doPay($params['payment_id'], PaymentChannel::USER);
    }

    public function setDefaultCard(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_method_id'], $params));
        if (!$paymentMethod = $this->paymentMethodRepository->find($params['payment_method_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.card')]));
        }
        $this->paymentMethodRepository->newQuery()->where('owner_id', Auth::id())->update(['default' => 0]);

        if ($paymentMethod->update(['default' => 1])) {
            return true;
        };
        return false;
    }

    public function getDefaultCard()
    {
        return $this->paymentMethodRepository->getDefaultCardByOwner(Auth::id(), ['id', 'card', 'default', 'holder_name', 'created_at']);
    }

    public function getPaymentV2(array $params)
    {
        return $this->getDetail($params);
    }

    public function getPayment(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        $payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id']);
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }
        $paymentCollection = $payment;
        $payment = $payment->toArray();
        $details = [];
        if (!empty($payment['charging_transaction'])) {
            $transaction = $payment['charging_transaction'];
            $duration = strtotime($transaction['ended_at']) - strtotime($transaction['started_at']);
            if ($duration < 60) {
                $duration = 60;
            }
            $consumedAmount = round($duration / 60, 0);
            $paymentAmount = $consumedAmount;
            $transactionRepository = app(TransactionRepositoryInterface::class);
            $details = $transactionRepository->getDetail($paymentAmount, $duration);
        }

        return collect(['payment' => $payment, 'details' => $details]);
    }

    public function getUserDept($userID = null)
    {
        $debt = parent::getUserDept(Auth::id());
        $currencyModule = app(CurrencyModule::class);
        $currency = $currencyModule->getDefault()->symbol;
        $message = $debt ? trans('visiosoft.module.charging::message.you_have_overdue_payment_short', ['amount' => $debt, 'currency' => $currency]) : null;
        return collect(['debt' => parent::getUserDept(Auth::id()), 'message' => $message]); // TODO: Change the autogenerated stub
    }

}
