<?php namespace Visiosoft\PaymentModule\Payment;

use Carbon\Carbon;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Facades\Auth;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\PaymentModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\PaymentModule\PaymentMethod\Contract\PaymentMethodRepositoryInterface;
use Visiosoft\PaymentModule\Status\Contract\StatusRepositoryInterface;
use Visiosoft\PaymentStripeExtension\PaymentStripeExtension;

class PaymentApiCollection extends PaymentRepository
{
    use DispatchesJobs;

    public function createCard(array $params)
    {
        if (!$create_card = PaymentStripeExtension::createCard($params)) {
            throw new \Exception(trans('visiosoft.model.payment::message.cant_add_card'));
        }
        $create_card = $create_card->toArray();
        if (!empty($create_card['id'])) {
            $create_card['service_payment_method_id'] = $create_card['id'];
            $create_card['service'] = 'stripe';
        }
        unset($create_card['id']);
        unset($create_card['object']);
        $create_card['owner_id'] = Auth::id();
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        return $paymentMethodRepository->addPaymentMethod($create_card);
    }

    public function cards()
    {
        $cards = PaymentStripeExtension::cards();
        $paymentMethodIds = [];
        if (!empty($cards)) {
            foreach ($cards as $card) {
                $paymentMethodIds[$card['id']] = $card['id'];
            }
        }
        $results = [];
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        $paymentMethods = $paymentMethodRepository->newQuery()
            ->where('owner_id', Auth::id())->whereNull('deleted_at')->get();
        foreach ($paymentMethods as $paymentMethod) {
            if (!empty($paymentMethodIds[$paymentMethod->service_payment_method_id])) {
                $results[] = $paymentMethod;
                unset($paymentMethodIds[$paymentMethod->service_payment_method_id]);
            } else {
                $paymentMethod->delete();
            }
        }
        return $paymentMethodRepository->newQuery()
            ->where('owner_id', Auth::id())->whereNull('deleted_at')->get();
    }

    public function createDraftPayment(array $params)
    {
        $paymentData = [
            'user_id' => $params['user_id'],
            'amount' => $params['amount'],
            'usage' => $params['usage'],
            'status' => 1,
            'category_id' => $params['category_id'],
            'currency' => "SEK"
        ];
        if (!empty($params['transaction_id'])) {
            $paymentData['transaction_id'] = $params['transaction_id'];
        }
        if (!empty($params['park_session_id'])) {
            $paymentData['park_session_id'] = $params['park_session_id'];
        }
        $create = $this->newQuery()->create($paymentData);

        if (!$create) {
            throw new \Exception(trans('visiosoft.module.payment::message.transaction_create_failed'), 400);
        }
        return $create;
    }


    public function createPayment(array $params)
    {

        $payment = $this->newQuery()->where('transaction_id', $params['transaction_id'])->first();

        $payment_params = ['currency' => "SEK", 'amount' => $payment['amount'], 'method_id' => $params['method_id']];

        $create_payment = PaymentStripeExtension::createPaymentIntent($payment_params);
        $payment->update(['payment_intent_id' => $create_payment['id']]);
        return $this->confirmPayment(['payment_id' => $create_payment['id']]);
    }

    public function confirmPayment(array $params)
    {
        $confirm_payment = PaymentStripeExtension::confirmPaymentIntent($params);
        $payment = $this->newQuery()->where('payment_intent_id', $params['payment_id'])->first();
        if ($confirm_payment['status'] == "succeeded") {
            $payment->update(['status_id' => 2]);
        } else throw new \Exception("Payment Failed", 404);
        return $payment;
    }

    public function list(array $params)
    {
        $userId = Auth::id();
        $payments = $this->newQuery()
            ->where('user_id', $userId)
            ->orderBy('created_at', 'DESC');
        if (!empty($params['category_id'])) {
            $payments = $payments->where('category_id', $params['category_id']);
        }
        if (!empty($params['status_id'])) {
            $payments = $payments->where('status_id', $params['status_id']);
        }
        if (!empty($params['start_date'])) {
            $payments = $payments->where('created_at', '>=', $params['start_date']);
        }
        if (!empty($params['end_date'])) {
            $payments = $payments->where('created_at', '<=', $params['end_date']);
        }
        return $payments->get();
    }

    public function references()
    {
        $categoryRepository = app(CategoryRepositoryInterface::class);
        $statusRepository = app(StatusRepositoryInterface::class);
        return collect([
            [
                'statuses' => $statusRepository->newQuery()->get(),
                'category' => $categoryRepository->newQuery()->get()
            ]
        ]);
    }

    public function removeCard(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['card_id'], $params));
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        $user = Auth::user();
        if (!$paymentMethod = $paymentMethodRepository->newQuery('owner_id', $user->id)->find($params['card_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment Method']));
        }
        return $detached = PaymentStripeExtension::detachPaymentMethod($paymentMethod->service_payment_method_id, $user->stripe_id);
    }

    public function createPaymentIntent(array $params) {
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if(!$payment = $this->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $amount = $payment->amount;
        $currency = $payment->currency;
        return $detached = PaymentStripeExtension::createPaymentIntent(['amount' => $amount, 'currency' => $currency]);
    }
}
