<?php namespace Visiosoft\PaymentModule\Payment;

use Anomaly\UsersModule\User\Contract\UserInterface;
use Anomaly\UsersModule\User\UserModel;
use Visiosoft\ChargingModule\Transaction\TransactionModel;
use Visiosoft\CompanyModule\Company\Contract\CompanyInterface;
use Visiosoft\PaymentModule\Payment\Contract\PaymentInterface;
use Anomaly\Streams\Platform\Model\Payment\PaymentPaymentEntryModel;
use Visiosoft\PaymentModule\PaymentModule;
use Visiosoft\VehicleModule\Vehicle\VehicleModel;

class PaymentModel extends PaymentPaymentEntryModel implements PaymentInterface
{

    protected $appends = [
        'charging_transaction',
        'vehicle',
        'payment_category',
        'payment_status',
        'user',
        'invoice_download',
        'name',
        'fees'
    ];

    public function getFeesAttribute()
    {
        return app(PaymentModule::class)->getFees($this->amount);
    }

    public function getInvoiceDownloadAttribute()
    {
        $file = $this->invoiceFile()->first();
        if ($file) {
            return config('app.url') . '/files/' . $file->path();
        }
        return null;
    }

    public function getUserAttribute()
    {
        return $this->user()->first();
    }

    public function getNameAttribute()
    {
        if ($transaction = $this->transaction()->first()) {
            $station = $transaction->station()->first();
            if ($station && isset($station->title)) {
                return $station->title;
            } else {
                return "";
            }
        }
    }

    public function getChargingTransactionAttribute()
    {
        return $this->chargingTransaction()->first();
    }

    public function getPaymentCategoryAttribute()
    {
        return $this->category()->first();
    }

    public function chargingTransaction()
    {
        return $this->belongsTo(TransactionModel::class, 'transaction_id');
    }

    public function getVehicleAttribute()
    {
        return $this->vehicle()->first();
    }

    public function vehicle()
    {
        return $this->belongsTo(VehicleModel::class, 'vehicle_id');
    }

    public function getPaymentStatusAttribute()
    {
        return $this->status()->first();
    }

    public function user()
    {
        return $this->belongsTo(UserModel::class)->withTrashed();
    }

    public function setPaymentMethod($type, $info)
    {
        return json_encode(['type' => $type, 'info' => $info]);
    }

    public function setPaymentChannel($channel)
    {
        $this->setAttribute('channel', $channel);
        $this->save();
    }


    public function getPaymentMethod()
    {
        return json_decode($this->payment_method);
    }

    /**
     * @return \Carbon\Carbon
     */

    public function getCreatedAt(): \Carbon\Carbon
    {
        return $this->created_at;
    }

    /**
     * @return UserInterface
     */
    public function getUser(): ?UserInterface
    {
        return $this->user()->first() ?? null;
    }

    /**
     * @return int
     */
    public function getUserId(): int
    {
        return $this->user_id;
    }

    /**
     * @return CompanyInterface|null
     */
    public function getCompany(): ?CompanyInterface
    {
        return $this->company()->first() ?? null;
    }

    public function getCompanyId(): ?int
    {
        return $this->company_id;
    }

    /**
     * @return float
     */
    public function getAmountIncludingVat(): float
    {
        return max(0, number_format($this->payment_amount, 2));
    }

    /**
     * @return float
     */
    public function getAmountExcludingVat(): float
    {
        //TODO:: Find where the amount field is registered and save it as excluding VAT and delete the calculation.
        return max(0, $this->getAmountIncludingVat() - $this->getVatAmount());
    }

    /**
     * @return float
     */
    public function getVatAmount(): float
    {
        return number_format($this->tax_amount, 2);
    }

    public function getVatPercent(): int
    {
        return $this->tax_percent ?? 0;
    }


    public function getPlate(): ?string
    {
        return $this->plate;
    }

    public function getUsage(): float
    {
        return $this->usage;
    }

    public function getUnitPrice(): float
    {
        if (round($this->getAmountExcludingVat()) > 0 && $this->getUsage() > 0) {
            return number_format($this->getAmountIncludingVat() / $this->getUsage());
        }

        return 0;
    }

    public function getTransactionId(): ?int
    {
        return $this->transaction_id;
    }

    public function getStationId(): ?int
    {
        return $this->station_id;
    }

    public function getLastTry()
    {
        return $this->last_try;
    }

    public function getServiceMessage()
    {
        return $this->service_message;
    }

}
