<?php namespace Visiosoft\PaymentModule\PaymentMethod;

use Illuminate\Support\Facades\Auth;
use Visiosoft\PaymentModule\PaymentMethod\Contract\PaymentMethodRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Visiosoft\PaymentModule\PaymentMethod\Event\CardCreatedEvent;

class PaymentMethodRepository extends EntryRepository implements PaymentMethodRepositoryInterface
{

    protected $model;

    public function __construct(PaymentMethodModel $model)
    {
        $this->model = $model;
    }

    public function addPaymentMethod(array $params)
    {
        if (!empty($params['billing_details'])) {
            $params['billing_details'] = (string)json_encode($params['billing_details']);
        }

        if (!empty($params['card'])) {
            $params['card'] = (string)json_encode($params['card']);
        }
        $params['metadata'] = null;
        if (!empty($params['metadata'])) {
            $params['metadata'] = (string)json_encode($params['metadata']);
        }
        $params['default'] = 1;

        $create = $this->newQuery()->create($params);

        if ($create) {
            $cardValidityCheck = setting_value('visiosoft.module.payment::check_credit_card_validity');
            if ($cardValidityCheck) {
                $validate = event(new CardCreatedEvent($params['owner_id']));
                if (!$validate) {
                    $create->update(['metadata' => 'invalid_card']);
                    $create->delete();
                    throw new \Exception("invalid_card");
                }
            }

            $this->newQuery()
                ->where('id', '!=', $create->id)
                ->where('owner_id', Auth::id())->update(['default' => 0]);
        }

        return $create;
    }

    public function getDefaultCardByCompany($ownerID, $companyID, $select = ['*'])
    {
        if ($select[0] != '*') {
            foreach ($select as $key => $value) {
                $select[$key] = "payment_payment_method." . $value;
            }
        }

        return $this->model->newQuery()
            ->join('company_employee', 'company_employee.user_id', '=', 'payment_payment_method.owner_id')
            ->whereNotNull('company_employee.payment_method_id')
            ->whereNull('payment_payment_method.deleted_at')
            ->where('owner_id', $ownerID)
            ->where('company_employee.company_id', $companyID)
            ->select($select)
            ->first();
    }


    /**
     * @param $ownerID
     * @param $companyID
     * @param $select
     * @return mixed
     */
    public function getCardToPay($ownerID, $companyID = null, $select = ['*'])
    {
        $card = [];
        $priority = setting_value('visiosoft.module.payment::payment_priority');
        if ($priority == "company" && !is_null($companyID)) {
            $tmp = $this->getDefaultCardByCompany($ownerID, $companyID, $select);
            if (!is_null($tmp)) {
                $card = $tmp;
            } else {
                $card = $this->getDefaultCardByOwner($ownerID, $select);
            }
        } else {
            $card = $this->getDefaultCardByOwner($ownerID, $select);
        }
        return $card;
    }

    /**
     * @param $ownerId
     * @param $select (Example : ['id','name','created_at'])
     */
    public function getDefaultCardByOwner($ownerId, $select = ['*'])
    {
        return $this->model->newQuery()
            ->where('owner_id', $ownerId)
            ->where('default', 1)
            ->whereNull('deleted_at')
            ->orderBy('id', 'DESC')
            ->select($select)
            ->first();
    }

    /**
     * @param $ownerId
     * @param $select (Example : ['id','name','created_at'])
     * @param $orderBy (Example : ['id','DESC'])
     */
    public function getCardsByOwner($ownerId, $select = ['*'], $orderBy = ['id', 'DESC'])
    {
        return $this->model->newQuery()
            ->where('owner_id', $ownerId)
            ->whereNull('deleted_at')
            ->select($select)
            ->orderBy($orderBy[0], $orderBy[1])
            ->get();
    }

    /**
     * @param $ownerId
     * @param $service
     */
    public function getCustomerIdByService($ownerId, $service)
    {
        return $this->model->newQuery()
            ->where('owner_id', $ownerId)
            ->where('service', $service)
            ->whereNull('deleted_at')
            ->whereNotNull('customer')
            ->pluck('customer')
            ->first();
    }

}
