<?php namespace Visiosoft\PaymentModule\Job;

use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Visiosoft\CurrencyModule\CurrencyModule;
use Visiosoft\InvoiceModule\InvoiceUser\Contract\InvoiceUserRepositoryInterface;
use Visiosoft\InvoiceModule\Job\CreateUserInvoiceJob;
use Visiosoft\VehicleModule\Vehicle\VehicleWithoutAttributeModel;

class AutoInvoiceJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $userId;
    protected $payments;

    public function __construct($userId, $payments)
    {
        $this->userId = $userId;
        $this->payments = $payments;
    }

    public function handle()
    {
        $details = [];
        $user = $this->getUser($this->userId);
        $getVehicleSessions = $this->getVehicleSessions($this->payments);
        $vehicles = $getVehicleSessions['vehicles'];
        $vehicleSessions = $getVehicleSessions['sessions'];
        if (!empty($vehicleSessions)) {
            foreach ($vehicleSessions as $vehicleId => $sessions) {
                $plate = $vehicles[$vehicleId];
                $vehicle = $this->getVehicle($vehicleId, $plate);
                $details[$vehicleId] = $this->getInvoiceDetails($user, $vehicle, $sessions, '');
            }
            $mainCreditorUserId = setting_value('visiosoft.module.invoice::main_creditor');
            $creditor = $this->getCreditor($mainCreditorUserId, 'business');
            $creditorUser = $this->getUser($mainCreditorUserId);
            if (!empty($details)) {
                foreach ($details as $detail) {
                    $debtor = $this->getDebtor($detail['user']['id'], 'personal', $creditor->external_id);
                    $records = $this->generateInvoiceRecords($detail);
                    $debtorUser = $this->getUser($detail['user']['id']);
                    $detail['invoice_records'] = $records;
                    $detail['creditor_external_id'] = $creditor->external_id;
                    $detail['debtor_external_id'] = $debtor->external_id;
                    $detail['creditor_id'] = $creditor->id;
                    $detail['debtor_id'] = $debtor->id;
                    $detail['debtor_phone'] = $debtorUser['phone'];
                    $detail['debtor_email'] = $debtorUser['email'];
                    $detail['creditor_phone'] = $creditorUser['phone'];
                    $detail['creditor_email'] = $creditorUser['email'];
                    CreateUserInvoiceJob::dispatch($detail);
                }
            }
        }
    }

    protected function getUser($userId)
    {
        $userRepository = app(UserRepositoryInterface::class);
        $user = $userRepository->newQuery()->select([
            'id',
            'phone',
            'ssn',
            'email',
            'first_name',
            'last_name'
        ])->find($userId);
        if (!$user) {
            return [];
        }
        return $user->toArray();
    }

    protected function getInvoiceDetails($user, $vehicle, $vehicleSessions, $plateImageUrl)
    {
        $vatPercent = $this->getVat();
        $taxable = $vehicleSessions['park_cost']
            + $vehicleSessions['park_offense_cost']
            + $vehicleSessions['park_overstay_cost']
            + $vehicleSessions['charge_cost']
            + $vehicleSessions['charge_overstay_cost'];
        $administrationFee = $this->calculateAdministrationFee($taxable);
        $vatInclude = $taxable + $taxable * $vatPercent / 100;
        $paymentAmount = $vatInclude + $administrationFee;
        $currency = CurrencyModule::getDefault()->symbol;
        return [
            'user' => $user,
            'vehicle' => $vehicle,
            'invoice_records' => [],
            'park_sessions' => $vehicleSessions['park'],
            'charge_sessions' => $vehicleSessions['charge'],
            'charge_overstay_cost' => $vehicleSessions['charge_overstay_cost'],
            'charge_penalty_cost' => $vehicleSessions['charge_overstay_cost'],
            'charge_cost' => $vehicleSessions['charge_cost'],
            'parking_overstay_cost' => $vehicleSessions['park_overstay_cost'],
            'parking_offense_cost' => $vehicleSessions['park_offense_cost'],
            'parking_penalty_cost' => $vehicleSessions['park_offense_cost'] + $vehicleSessions['park_overstay_cost'],
            'park_cost' => $vehicleSessions['park_cost'],
            'park_duration' => $vehicleSessions['park_duration'],
            'charge_consumption' => $vehicleSessions['charge_consumption'],
            'plate' => $vehicle['plate'],
            'plate_image_url' => $plateImageUrl,
            'payment_amount' => $paymentAmount,
            'vat_percent' => $vatPercent,
            'taxable' => $taxable,
            'administration_fee' => $administrationFee,
            'currency' => 'SEK',
            'creditor_address' => null,
            'creditor_phone' => null,
            'creditor_email' => null,
            'debtor_address' => null,
            'debtor_phone' => null,
            'debtor_email' => null,
            'creditor_id' => null,
            'debtor_id' => null,
            'creditor_external_id' => null,
            'debtor_external_id' => null
        ];
    }

    protected function getVehicle($vehicleId, $plate)
    {
        $vehicle = VehicleWithoutAttributeModel::select([
            'id',
            'plate',
            'brand_text',
            'model_text'
        ])->find($vehicleId);
        if ($vehicle) {
            return $vehicle->toArray();
        }
        return ['id' => $vehicleId, 'plate' => $plate];
    }

    protected function getVehicleSessions($payments)
    {
        $vehicleSessions = [];
        $vehicles = [];
        if (!empty($payments)) {
            foreach ($payments as $payment) {
                if (empty($vehicleSessions[$payment['vehicle_id']])) {
                    $vehicleSessions[$payment['vehicle_id']] = [
                        'park' => [],
                        'charge' => [],
                        'park_duration' => 0,
                        'charge_consumption' => 0,
                        'park_cost' => 0,
                        'charge_cost' => 0,
                        'charge_penalty_cost' => 0,
                        'charge_overstay_cost' => 0,
                        'park_penalty_cost' => 0,
                        'park_offense_cost' => 0,
                        'park_overstay_cost' => 0
                    ];
                }
                $vehicles[$payment['vehicle_id']] = $payment['plate'];
                if ($payment['category_id'] == 1) {
                    if (!empty($payment['charging_transaction'])) {

                        $vehicleSessions[$payment['vehicle_id']]['charge_consumption']
                            = $vehicleSessions[$payment['vehicle_id']]['charge_consumption']
                            + $payment['charging_transaction']['total_consumption'];

                        $vehicleSessions[$payment['vehicle_id']]['charge_cost']
                            = $vehicleSessions[$payment['vehicle_id']]['charge_cost']
                            + $payment['amount'];

                        $vehicleSessions[$payment['vehicle_id']]['charge_overstay_cost']
                            = $vehicleSessions[$payment['vehicle_id']]['charge_overstay_cost']
                            + $payment['charging_transaction']['overstay_cost'];

                        $vehicleSessions[$payment['vehicle_id']]['charge_penalty_cost']
                            = $vehicleSessions[$payment['vehicle_id']]['charge_penalty_cost']
                            + $payment['charging_transaction']['overstay_cost'];

                        $vehicleSessions[$payment['vehicle_id']]['charge'][] = $payment['charging_transaction'];

                    }
                }
                if ($payment['category_id'] == 2) {
                    if (!empty($payment['park_session'])) {

                        $vehicleSessions[$payment['vehicle_id']]['park_duration']
                            = $vehicleSessions[$payment['vehicle_id']]['park_duration']
                            + strtotime($payment['park_session']['ended_at'])
                            - strtotime($payment['park_session']['started_at']);

                        $vehicleSessions[$payment['vehicle_id']]['park_cost']
                            = $vehicleSessions[$payment['vehicle_id']]['park_cost']
                            + $payment['amount'];

                        $vehicleSessions[$payment['vehicle_id']]['park_offense_cost']
                            = $vehicleSessions[$payment['vehicle_id']]['park_offense_cost']
                            + $payment['park_session']['offense_amount'];

                        $vehicleSessions[$payment['vehicle_id']]['park_overstay_cost']
                            = $vehicleSessions[$payment['vehicle_id']]['park_overstay_cost']
                            + $payment['park_session']['overstay_amount'];

                        $vehicleSessions[$payment['vehicle_id']]['park_penalty_cost']
                            = $vehicleSessions[$payment['vehicle_id']]['park_penalty_cost']
                            + $payment['park_session']['offense_amount']
                            + $payment['park_session']['overstay_amount'];

                        $vehicleSessions[$payment['vehicle_id']]['park'][] = $payment['park_session'];

                    }
                }
            }
        }
        return ['sessions' => $vehicleSessions, 'vehicles' => $vehicles];
    }

    protected function calculateAdministrationFee($amount)
    {
        $administrationFee = $this->getAdministrationFee();
        return $amount + ($amount * $administrationFee['fee_percent'] / 100) + $administrationFee['fee_amount'];
    }

    protected function calculateVAT($amount)
    {
        $vatPercent = $this->getVat();
        return $amount + ($vatPercent / 100) * $amount;
    }

    protected function getVat()
    {
        return setting_value('visiosoft.module.invoice::vat');
    }

    protected function getAdministrationFee(): array
    {
        return [
            'fee_amount' => setting_value('visiosoft.module.payment::administration_fee_amount'),
            'fee_percent' => setting_value('visiosoft.module.payment::administration_fee_percent')
        ];
    }

    protected function generateInvoiceRecords($details): array
    {
        $records = [];
        $sequencer = -1;
        if (!empty($details['park_sessions'])) {
            foreach ($details['park_sessions'] as $session) {
                $sequencer = $sequencer + 1;
                $records[] = [
                    'sequence' => $sequencer,
                    'type' => 'message',
                    'message' => trans('visiosoft.module.invoice::invoice.parking_session_transaction_number', ['transaction_number' => $session['payment_transaction_id']]),
                    'vat' => 0,
                    'amount' => 0,
                    'payment_id' => null
                ];
                $records[] = [
                    'sequence' => $sequencer,
                    'type' => 'standard',
                    'amount' => $session['payment_amount'],
                    'message' => trans('visiosoft.module.invoice::invoice.parking_session'),
                    'vat' => $details['vat_percent'],
                    'payment_id' => $session['payment_transaction_id']
                ];
                $records[] = [
                    'sequence' => $sequencer,
                    'type' => 'message',
                    'message' => trans('visiosoft.module.invoice::invoice.started_at', ['started_at' => $session['started_at']]),
                    'vat' => 0,
                    'amount' => 0,
                    'payment_id' => null
                ];
                $records[] = [
                    'sequence' => $sequencer,
                    'type' => 'message',
                    'message' => trans('visiosoft.module.invoice::invoice.ended_at', ['ended_at' => $session['ended_at']]),
                    'vat' => 0,
                    'amount' => 0,
                    'payment_id' => null
                ];
                $records[] = [
                    'sequence' => $sequencer,
                    'type' => 'message',
                    'message' => trans('visiosoft.module.invoice::invoice.location', ['location' => $session['park']['name'] . ' ' . $session['park']['address']]),
                    'vat' => 0,
                    'amount' => 0,
                    'payment_id' => null
                ];
                if (!empty($session['penalty_amount'])) {
                    $sequencer = $sequencer + 1;
                    $records[] = [
                        'sequence' => $sequencer,
                        'type' => 'message',
                        'message' => trans('visiosoft.module.invoice::invoice.transaction_number', ['transaction_number' => $session['payment_transaction_id']]),
                        'vat' => 0,
                        'amount' => 0,
                        'payment_id' => null
                    ];
                    $records[] = [
                        'sequence' => $sequencer,
                        'type' => 'standard',
                        'message' => trans('visiosoft.module.invoice::invoice.parking_session_penalty_cost'),
                        'vat' => $details['vat_percent'],
                        'amount' => $session['penalty_amount'],
                        'payment_id' => null
                    ];
                }
            }
        }

        if (!empty($details['charge_sessions'])) {
            foreach ($details['charge_sessions'] as $session) {
                $sequencer = $sequencer + 1;
                $records[] = [
                    'sequence' => $sequencer,
                    'type' => 'message',
                    'message' => trans('visiosoft.module.invoice::invoice.ev_charging_transaction_no', ['transaction_number' => $session['payment_id']]),
                    'vat' => 0,
                    'amount' => 0,
                    'payment_id' => null
                ];
                $records[] = [
                    'sequence' => $sequencer,
                    'type' => 'standard',
                    'amount' => $session['payment_amount'],
                    'message' => trans('visiosoft.module.invoice::invoice.charging_session_cost'),
                    'vat' => $details['vat_percent'],
                    'payment_id' => $session['payment_id']
                ];
                $records[] = [
                    'sequence' => $sequencer,
                    'type' => 'message',
                    'message' => trans('visiosoft.module.invoice::invoice.electricity_consumption', ['consumption' => $session['total_consumption']]),
                    'vat' => 0,
                    'amount' => 0,
                    'payment_id' => null
                ];
                $records[] = [
                    'sequence' => $sequencer,
                    'type' => 'message',
                    'message' => trans('visiosoft.module.invoice::invoice.location', ['location' => $session['station']['name'] . ' ' . $session['station']['address']]),
                    'vat' => 0,
                    'amount' => 0,
                    'payment_id' => null
                ];
                if (!empty($session['penalty_amount'])) {
                    $sequencer = $sequencer + 1;
                    $records[] = [
                        'sequence' => $sequencer,
                        'type' => 'message',
                        'message' => trans('visiosoft.module.invoice::invoice.ev_charging_penalty'),
                        'vat' => 0,
                        'amount' => 0,
                        'payment_id' => null
                    ];
                    $records[] = [
                        'sequence' => $sequencer,
                        'type' => 'message',
                        'message' => trans('visiosoft.module.invoice::invoice.transaction_number', ['transaction_number' => $session['payment_id']]),
                        'vat' => 0,
                        'amount' => 0,
                        'payment_id' => null
                    ];
                    $records[] = [
                        'sequence' => $sequencer,
                        'type' => 'standard',
                        'message' => trans('visiosoft.module.invoice::invoice.penalty_cost'),
                        'vat' => $details['vat_percent'],
                        'amount' => $session['penalty_amount'],
                        'payment_id' => null
                    ];
                }
            }
        }
        $sequencer = $sequencer + 1;
        $records[] = [
            'sequence' => $sequencer,
            'type' => 'standard',
            'message' => trans('visiosoft.module.invoice::invoice.administration_fee'),
            'vat' => $details['vat_percent'],
            'amount' => $details['administration_fee']
        ];
        return $records;
    }

    protected function getUserRepository()
    {
        return app(UserRepositoryInterface::class);
    }

    protected function getInvoiceUserRepository()
    {
        return app(InvoiceUserRepositoryInterface::class);
    }

    protected function getUserDetail($userId)
    {
        $userRepository = $this->getUserRepository();
        $user = $userRepository->newQuery()->find($userId);
        if (!$user) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'User']), 404);
        }
        return $user;
    }

    protected function getDebtor($userId, $personType, $creditorExternalId)
    {
        $invoiceUserRepository = $this->getInvoiceUserRepository();
        $invoiceReceiver = $invoiceUserRepository->getUser($userId, 'receiver', $personType);
        $user = $this->getUserDetail($userId);
        if (!$invoiceReceiver) {
            $params = [
                'creditor_id' => $creditorExternalId,
                'ssn' => $user->ssn,
                'name' => $user->name(),
                'address' => $user->extended_address,
                'city' => $user->city,
                'country_code' => $user->country_code,
                'vat_number' => $user->vat_number,
                'phone' => $user->phone,
                'email' => $user->email,
                'invoice_address' => [
                    'address' => $user->extended_address,
                    'zipcode' => $user->zip_code,
                    'city' => $user->city,
                    'country_code' => $user->country_code,
                    'email' => $user->email,
                    'delivery_method' => "Email"
                ]
            ];
            $invoiceReceiver = $invoiceUserRepository->createUser($userId, 'receiver', $personType, $params);
        }
        return $invoiceReceiver;
    }

    protected function getCreditor($userId, $personType)
    {
        $invoiceUserRepository = $this->getInvoiceUserRepository();
        $invoiceSender = $invoiceUserRepository->getUser($userId, 'sender', $personType);
        $user = $this->getUserDetail($userId);
        if (!$invoiceSender) {
            $params = [
                'organization_number' => $user->organization_number,
                'name' => $user->name(),
                'address' => $user->extended_address,
                'city' => $user->city,
                'country_code' => $user->country_code,
                'vat_number' => $user->vat_number,
                'invoice_address' => [
                    'address' => $user->extended_address,
                    'zipcode' => $user->zip_code,
                    'city' => $user->city,
                    'country_code' => $user->country_code,
                    'email' => $user->email,
                    'delivery_method' => "Email"
                ]
            ];
            $invoiceSender = $invoiceUserRepository->createUser($userId, 'sender', $personType, $params);
        }
        return $invoiceSender;
    }
}
